#!python
from __future__ import print_function

from os import listdir, environ
from os.path import isfile, join, splitext
from subprocess import Popen, PIPE, STDOUT

import argparse
import os
import re
import sys

################################################################################
#               help functions
################################################################################
def listTests():
	list = [splitext(f)[0] for f in listdir('./.expect')
		if not f.startswith('.') and f.endswith('.txt')
		]

	return list

def sh(cmd, dry_run = False, print2stdout = True):
	if dry_run :
		print("cmd: %s" % cmd)
		return 0, None
	else :
		proc = Popen(cmd, stdout=None if print2stdout else PIPE, stderr=STDOUT, shell=True)
		out, err = proc.communicate()
		return proc.returncode, out

def file_replace(fname, pat, s_after):
    # first, see if the pattern is even in the file.
    with open(fname) as f:
        if not any(re.search(pat, line) for line in f):
            return # pattern does not occur in file so we are done.

    # pattern is in the file, so perform replace operation.
    with open(fname) as f:
        out_fname = fname + ".tmp"
        out = open(out_fname, "w")
        for line in f:
            out.write(re.sub(pat, s_after, line))
        out.close()
        os.rename(out_fname, fname)

def fix_MakeLevel(file) :
	if environ.get('MAKELEVEL') :
		file_replace(file, "make\[%i\]" % int(environ.get('MAKELEVEL')), 'make' )

def fileContainsOnly(file, text) :
	with open(file) as f:
		ff = f.read().strip()
		result = ff == text.strip()
		#
		# print("Comparing :\n\t'%s'\nWith:\n\t'%s'" % (ff, text))
		# print("Result is : \n\t", end="")
		# print(result)

		return result;


################################################################################
#               running test functions
################################################################################
def run_test_instance(test, generate, dry_run):

	out_file = (".out/%s.log" % test) if not generate else (".expect/%s.txt" % test)

	sh("rm -f %s" % out_file, dry_run)
	sh("rm -f %s > /dev/null 2>&1" % test, dry_run)

	# build, skipping to next test on error
	make_ret, _ = sh("%s %s 2> %s 1> /dev/null" % (make_cmd, test, out_file), dry_run)

	if make_ret == 0 :
		# fetch optional input
		stdinput = "< .in/%s.txt" % test if isfile(".in/%s.txt" % test) else ""

		# run test
		sh("./%s %s > %s 2>&1" % (test, stdinput, out_file), dry_run)

	retcode = 0
	error = None

	fix_MakeLevel(out_file)

	if generate :
		if not dry_run and fileContainsOnly(out_file, "make: *** No rule to make target `%s'.  Stop." % test) :
			retcode = 1;
			error = "\t\tNo make target for test %s!" % test
			sh("rm %s" % out_file, False)

	else :
		# diff the output of the files
		diff_cmd = ("diff --old-group-format='\t\tmissing lines :\n"
					"%%<' \\\n"
					"--new-group-format='\t\tnew lines :\n"
					"%%>' \\\n"
					"--unchanged-group-format='%%=' \\"
					"--changed-group-format='\t\texpected :\n"
					"%%<\n"
					"\t\tgot :\n"
					"%%>' \\\n"
					"--new-line-format='\t\t%%dn\t%%L' \\\n"
					"--old-line-format='\t\t%%dn\t%%L' \\\n"
					"--unchanged-line-format='' \\\n"
					".expect/%s.txt .out/%s.log")

		retcode, error = sh(diff_cmd % (test, test), dry_run, False)

	# clean the executable
	sh("rm -f %s > /dev/null 2>&1" % test, dry_run)

	return retcode, error

def run_tests(tests, generate, dry_run) :
	sh("%s clean > /dev/null 2>&1" % make_cmd, dry_run)
	sh('mkdir -p .out .expect', dry_run)

	if generate :
		print( "Regenerate tests for: " )

	failed = False;
	for t in tests:
		print("%20s  " % t, end="")
		sys.stdout.flush()
		test_failed, error = run_test_instance(t, generate, dry_run)
		failed = test_failed or failed

		if generate :
			failed_txt = "ERROR"
			success_txt = "Done"
		else :
			failed_txt = "FAILED"
			success_txt = "PASSED"

		print(failed_txt if test_failed else success_txt)
		if error :
			print(error, file=sys.stderr)

	sh("%s clean > /dev/null 2>&1" % make_cmd, dry_run)

	return 1 if failed else 0

################################################################################
#               main loop
################################################################################
parser = argparse.ArgumentParser(description='Script which runs cforall tests')
parser.add_argument('--dry-run', help='Don\'t run the tests, only output the commands', action='store_true')
parser.add_argument('--list', help='List all test available', action='store_true')
parser.add_argument('--all', help='Run all test available', action='store_true')
parser.add_argument('--regenerate-expected', help='Regenerate the .expect by running the specified tets, can be used with --all option', action='store_true')
parser.add_argument('tests', metavar='test', type=str, nargs='*', help='a list of tests to run')

options = parser.parse_args()

if (len(options.tests) > 0  and     options.all and not options.list) \
or (len(options.tests) == 0 and not options.all and not options.list) :
	print('ERROR: must have option \'--all\' or non-empty test list', file=sys.stderr)
	parser.print_help()
	sys.exit(1)

allTests = listTests()

if options.all or options.list :
	tests = allTests

else :
	tests = []
	for test in options.tests:
		if test in allTests or options.regenerate_expected :
			tests.append(test)
		else :
			print('ERROR: No expected file for test %s, ignoring it' % test, file=sys.stderr)

	if len(tests) == 0 :
		print('ERROR: No valid test to run', file=sys.stderr)
		sys.exit(1)

tests.sort()
make_flags = environ.get('MAKEFLAGS')
make_cmd = "make" if make_flags and "-j" in make_flags else "make -j8"

if options.list :
	print("\n".join(tests))

else :
	sys.exit( run_tests(tests, options.regenerate_expected, options.dry_run) )
