//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Visitor.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 11:14:51 2015
// Update Count     : 2
//

#include <cassert>
#include "Visitor.h"
#include "Initializer.h"
#include "Statement.h"
#include "Type.h"
#include "Declaration.h"
#include "Expression.h"
#include "Constant.h"

Visitor::Visitor() {}

Visitor::~Visitor() {}

void Visitor::visit( ObjectDecl *objectDecl ) {
	maybeAccept( objectDecl->get_type(), *this );
	maybeAccept( objectDecl->get_init(), *this );
	maybeAccept( objectDecl->get_bitfieldWidth(), *this );
}

void Visitor::visit( FunctionDecl *functionDecl ) {
	maybeAccept( functionDecl->get_functionType(), *this );
	acceptAll( functionDecl->get_oldDecls(), *this );
	maybeAccept( functionDecl->get_statements(), *this );
}

void Visitor::visit( AggregateDecl *aggregateDecl ) {
	acceptAll( aggregateDecl->get_parameters(), *this );
	acceptAll( aggregateDecl->get_members(), *this );
}

void Visitor::visit( StructDecl *aggregateDecl ) {
	visit( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( UnionDecl *aggregateDecl ) {
	visit( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( EnumDecl *aggregateDecl ) {
	visit( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( ContextDecl *aggregateDecl ) {
	visit( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( NamedTypeDecl *typeDecl ) {
	acceptAll( typeDecl->get_parameters(), *this );
	acceptAll( typeDecl->get_assertions(), *this );
	maybeAccept( typeDecl->get_base(), *this );
}

void Visitor::visit( TypeDecl *typeDecl ) {
	visit( static_cast< NamedTypeDecl* >( typeDecl ) );
}

void Visitor::visit( TypedefDecl *typeDecl ) {
	visit( static_cast< NamedTypeDecl* >( typeDecl ) );
}

void Visitor::visit( CompoundStmt *compoundStmt ) {
	acceptAll( compoundStmt->get_kids(), *this );
}

void Visitor::visit( ExprStmt *exprStmt ) {
	maybeAccept( exprStmt->get_expr(), *this );
}

void Visitor::visit( IfStmt *ifStmt ) {
	maybeAccept( ifStmt->get_condition(), *this );
	maybeAccept( ifStmt->get_thenPart(), *this );
	maybeAccept( ifStmt->get_elsePart(), *this );
}

void Visitor::visit( WhileStmt *whileStmt ) {
	maybeAccept( whileStmt->get_condition(), *this );
	maybeAccept( whileStmt->get_body(), *this );
}

void Visitor::visit( ForStmt *forStmt ) {
	// ForStmt still needs to be fixed
	maybeAccept( forStmt->get_initialization(), *this );
	maybeAccept( forStmt->get_condition(), *this );
	maybeAccept( forStmt->get_increment(), *this );
	maybeAccept( forStmt->get_body(), *this );
}

void Visitor::visit( SwitchStmt *switchStmt ) {
	maybeAccept( switchStmt->get_condition(), *this );
	acceptAll( switchStmt->get_branches(), *this );
}

void Visitor::visit( ChooseStmt *switchStmt ) {
	maybeAccept( switchStmt->get_condition(), *this );
	acceptAll( switchStmt->get_branches(), *this );
}

void Visitor::visit( FallthruStmt *fallthruStmt ) {}

void Visitor::visit( CaseStmt *caseStmt ) {
	maybeAccept( caseStmt->get_condition(), *this );
	acceptAll( caseStmt->get_statements(), *this );
}

void Visitor::visit( BranchStmt *branchStmt ) {
}

void Visitor::visit( ReturnStmt *returnStmt ) {
	maybeAccept( returnStmt->get_expr(), *this );
}

void Visitor::visit( TryStmt *tryStmt ) {
	maybeAccept( tryStmt->get_block(), *this );
	acceptAll( tryStmt->get_catchers(), *this );
}

void Visitor::visit( CatchStmt *catchStmt ) {
	maybeAccept( catchStmt->get_decl(), *this );
	maybeAccept( catchStmt->get_body(), *this );
}

void Visitor::visit( FinallyStmt *finalStmt ) {
	maybeAccept( finalStmt->get_block(), *this );
}

void Visitor::visit( NullStmt *nullStmt ) {
}

void Visitor::visit( DeclStmt *declStmt ) {
	maybeAccept( declStmt->get_decl(), *this );
}

void Visitor::visit( ApplicationExpr *applicationExpr ) {
	acceptAll( applicationExpr->get_results(), *this );
	maybeAccept( applicationExpr->get_function(), *this );
	acceptAll( applicationExpr->get_args(), *this );
}

void Visitor::visit( UntypedExpr *untypedExpr ) {
	acceptAll( untypedExpr->get_results(), *this );
	acceptAll( untypedExpr->get_args(), *this );
}

void Visitor::visit( NameExpr *nameExpr ) {
	acceptAll( nameExpr->get_results(), *this );
}

void Visitor::visit( AddressExpr *addressExpr ) {
	acceptAll( addressExpr->get_results(), *this );
	maybeAccept( addressExpr->get_arg(), *this );
}

void Visitor::visit( LabelAddressExpr *labAddressExpr ) {
	acceptAll( labAddressExpr->get_results(), *this );
	maybeAccept( labAddressExpr->get_arg(), *this );
}

void Visitor::visit( CastExpr *castExpr ) {
	acceptAll( castExpr->get_results(), *this );
	maybeAccept( castExpr->get_arg(), *this );
}

void Visitor::visit( UntypedMemberExpr *memberExpr ) {
	acceptAll( memberExpr->get_results(), *this );
	maybeAccept( memberExpr->get_aggregate(), *this );
}

void Visitor::visit( MemberExpr *memberExpr ) {
	acceptAll( memberExpr->get_results(), *this );
	maybeAccept( memberExpr->get_aggregate(), *this );
}

void Visitor::visit( VariableExpr *variableExpr ) {
	acceptAll( variableExpr->get_results(), *this );
}

void Visitor::visit( ConstantExpr *constantExpr ) {
	acceptAll( constantExpr->get_results(), *this );
	maybeAccept( constantExpr->get_constant(), *this );
}

void Visitor::visit( SizeofExpr *sizeofExpr ) {
	acceptAll( sizeofExpr->get_results(), *this );
	if ( sizeofExpr->get_isType() ) {
		maybeAccept( sizeofExpr->get_type(), *this );
	} else {
		maybeAccept( sizeofExpr->get_expr(), *this );
	}
}

void Visitor::visit( AttrExpr *attrExpr ) {
	acceptAll( attrExpr->get_results(), *this );
	if ( attrExpr->get_isType() ) {
		maybeAccept( attrExpr->get_type(), *this );
	} else {
		maybeAccept( attrExpr->get_expr(), *this );
	}
}

void Visitor::visit( LogicalExpr *logicalExpr ) {
	acceptAll( logicalExpr->get_results(), *this );
	maybeAccept( logicalExpr->get_arg1(), *this );
	maybeAccept( logicalExpr->get_arg2(), *this );
}

void Visitor::visit( ConditionalExpr *conditionalExpr ) {
	acceptAll( conditionalExpr->get_results(), *this );
	maybeAccept( conditionalExpr->get_arg1(), *this );
	maybeAccept( conditionalExpr->get_arg2(), *this );
	maybeAccept( conditionalExpr->get_arg3(), *this );
}

void Visitor::visit( CommaExpr *commaExpr ) {
	acceptAll( commaExpr->get_results(), *this );
	maybeAccept( commaExpr->get_arg1(), *this );
	maybeAccept( commaExpr->get_arg2(), *this );
}

void Visitor::visit( TupleExpr *tupleExpr ) {
	acceptAll( tupleExpr->get_results(), *this );
	acceptAll( tupleExpr->get_exprs(), *this );
}

void Visitor::visit( SolvedTupleExpr *tupleExpr ) {
	acceptAll( tupleExpr->get_results(), *this );
	acceptAll( tupleExpr->get_exprs(), *this );
}

void Visitor::visit( TypeExpr *typeExpr ) {
	acceptAll( typeExpr->get_results(), *this );
	maybeAccept( typeExpr->get_type(), *this );
}

void Visitor::visit( UntypedValofExpr *valofExpr ) {
	acceptAll( valofExpr->get_results(), *this );
	maybeAccept( valofExpr->get_body(), *this );
}

void Visitor::visit( VoidType *voidType ) {
	acceptAll( voidType->get_forall(), *this );
}

void Visitor::visit( BasicType *basicType ) {
	acceptAll( basicType->get_forall(), *this );
}

void Visitor::visit( PointerType *pointerType ) {
	acceptAll( pointerType->get_forall(), *this );
	maybeAccept( pointerType->get_base(), *this );
}

void Visitor::visit( ArrayType *arrayType ) {
	acceptAll( arrayType->get_forall(), *this );
	maybeAccept( arrayType->get_dimension(), *this );
	maybeAccept( arrayType->get_base(), *this );
}

void Visitor::visit( FunctionType *functionType ) {
	acceptAll( functionType->get_forall(), *this );
	acceptAll( functionType->get_returnVals(), *this );
	acceptAll( functionType->get_parameters(), *this );
}

void Visitor::visit( ReferenceToType *aggregateUseType ) {
	acceptAll( aggregateUseType->get_forall(), *this );
	acceptAll( aggregateUseType->get_parameters(), *this );
}

void Visitor::visit( StructInstType *aggregateUseType ) {
	visit( static_cast< ReferenceToType* >( aggregateUseType ) );
}

void Visitor::visit( UnionInstType *aggregateUseType ) {
	visit( static_cast< ReferenceToType* >( aggregateUseType ) );
}

void Visitor::visit( EnumInstType *aggregateUseType ) {
	visit( static_cast< ReferenceToType* >( aggregateUseType ) );
}

void Visitor::visit( ContextInstType *aggregateUseType ) {
	visit( static_cast< ReferenceToType* >( aggregateUseType ) );
	acceptAll( aggregateUseType->get_members(), *this );
}

void Visitor::visit( TypeInstType *aggregateUseType ) {
	visit( static_cast< ReferenceToType* >( aggregateUseType ) );
}

void Visitor::visit( TupleType *tupleType ) {
	acceptAll( tupleType->get_forall(), *this );
	acceptAll( tupleType->get_types(), *this );
}

void Visitor::visit( TypeofType *typeofType ) {
	assert( typeofType->get_expr() );
	typeofType->get_expr()->accept( *this );
}

void Visitor::visit( AttrType *attrType ) {
	if ( attrType->get_isType() ) {
		assert( attrType->get_type() );
		attrType->get_type()->accept( *this );
	} else {
		assert( attrType->get_expr() );
		attrType->get_expr()->accept( *this );
	} // if
}

void Visitor::visit( SingleInit *singleInit ) {
	singleInit->get_value()->accept( *this );
}

void Visitor::visit( ListInit *listInit ) {
	acceptAll( listInit->get_designators(), *this );
	acceptAll( listInit->get_initializers(), *this );
}

void Visitor::visit( Subrange *subrange ) {}

void Visitor::visit( Constant *constant ) {}
// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
