#include "InitTweak.h"
#include "SynTree/Visitor.h"
#include "SynTree/Statement.h"
#include "SynTree/Initializer.h"
#include "SynTree/Expression.h"
#include "GenPoly/GenPoly.h"

namespace InitTweak {
	namespace {
		class HasDesignations : public Visitor {
		public:
			bool hasDesignations = false;
			template<typename Init>
			void handleInit( Init * init ) {
				if ( ! init->get_designators().empty() ) hasDesignations = true;
				else Visitor::visit( init );
			}
			virtual void visit( SingleInit * singleInit ) { handleInit( singleInit); }
			virtual void visit( ListInit * listInit ) { handleInit( listInit); }
		};

		class InitExpander : public Visitor {
			public:
			InitExpander() {}
			virtual void visit( SingleInit * singleInit );
			virtual void visit( ListInit * listInit );
			std::list< Expression * > argList;
		};

		void InitExpander::visit( SingleInit * singleInit ) {
			argList.push_back( singleInit->get_value()->clone() );
		}

		void InitExpander::visit( ListInit * listInit ) {
			// xxx - for now, assume no nested list inits
			std::list<Initializer*>::iterator it = listInit->begin_initializers();
			for ( ; it != listInit->end_initializers(); ++it ) {
				(*it)->accept( *this );
			}
		}
	}

	std::list< Expression * > makeInitList( Initializer * init ) {
		InitExpander expander;
		maybeAccept( init, expander );
		return expander.argList;
	}

	bool isDesignated( Initializer * init ) {
		HasDesignations finder;
		maybeAccept( init, finder );
		return finder.hasDesignations;
	}

	bool tryConstruct( ObjectDecl * objDecl ) {
		return ! LinkageSpec::isBuiltin( objDecl->get_linkage() ) &&
			(objDecl->get_init() == NULL ||
				( objDecl->get_init() != NULL && objDecl->get_init()->get_maybeConstructed() )) &&
			! isDesignated( objDecl->get_init() );
	}

	Expression * getCtorDtorCall( Statement * stmt ) {
		if ( stmt == NULL ) return NULL;
		if ( ExprStmt * exprStmt = dynamic_cast< ExprStmt * >( stmt ) ) {
			return exprStmt->get_expr();
		} else if ( CompoundStmt * compoundStmt = dynamic_cast< CompoundStmt * >( stmt ) ) {
			// could also be a compound statement with a loop, in the case of an array
			if( compoundStmt->get_kids().size() == 2 ) {
				// loop variable and loop
				ForStmt * forStmt = dynamic_cast< ForStmt * >( compoundStmt->get_kids().back() );
				assert( forStmt && forStmt->get_body() );
				return getCtorDtorCall( forStmt->get_body() );
			} else if ( compoundStmt->get_kids().size() == 1 ) {
				// should be the call statement, but in any case there's only one option
				return getCtorDtorCall( compoundStmt->get_kids().front() );
			} else {
				assert( false && "too many statements in compoundStmt for getCtorDtorCall" );
			}
		} if ( ImplicitCtorDtorStmt * impCtorDtorStmt = dynamic_cast< ImplicitCtorDtorStmt * > ( stmt ) ) {
			return getCtorDtorCall( impCtorDtorStmt->get_callStmt() );
		} else {
			// should never get here
			assert( false && "encountered unknown call statement" );
		}
	}

	bool isInstrinsicSingleArgCallStmt( Statement * stmt ) {
		Expression * callExpr = getCtorDtorCall( stmt );
		if ( ! callExpr ) return false;
		ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( callExpr );
		assert( appExpr );
		VariableExpr * function = dynamic_cast< VariableExpr * >( appExpr->get_function() );
		assert( function );
		// check for Intrinsic only - don't want to remove all overridable ctor/dtors because autogenerated ctor/dtor
		// will call all member dtors, and some members may have a user defined dtor.
		FunctionType * funcType = GenPoly::getFunctionType( function->get_var()->get_type() );
		assert( funcType );
		return function->get_var()->get_linkage() == LinkageSpec::Intrinsic && funcType->get_parameters().size() == 1;
	}

	namespace {
		template<typename CallExpr>
		Expression *& callArg( CallExpr * callExpr, unsigned int pos ) {
			if ( pos >= callExpr->get_args().size() ) assert( false && "asking for argument that doesn't exist. Return NULL/throw exception?" );
			for ( Expression *& arg : callExpr->get_args() ) {
				if ( pos == 0 ) return arg;
				pos--;
			}
			assert( false );
		}
	}

	Expression *& getCallArg( Expression * callExpr, unsigned int pos ) {
		if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( callExpr ) ) {
			return callArg( appExpr, pos );
		} else if ( UntypedExpr * untypedExpr = dynamic_cast< UntypedExpr * >( callExpr ) ) {
			return callArg( untypedExpr, pos );
		} else {
			assert( false && "Unexpected expression type passed to getCallArg" );
		}
	}

	namespace {
		std::string funcName( Expression * func ) {
			if ( NameExpr * nameExpr = dynamic_cast< NameExpr * >( func ) ) {
				return nameExpr->get_name();
			} else if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( func ) ) {
				return varExpr->get_var()->get_name();
			}	else if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( func ) ) {
				return funcName( castExpr->get_arg() );
			} else {
				assert( false && "Unexpected expression type being called as a function in call expression" );
			}
		}
	}

	std::string getFunctionName( Expression * expr ) {
		if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( expr ) ) {
			return funcName( appExpr->get_function() );
		} else if ( UntypedExpr * untypedExpr = dynamic_cast< UntypedExpr * > ( expr ) ) {
			return funcName( untypedExpr->get_function() );
		} else {
			std::cerr << expr << std::endl;
			assert( false && "Unexpected expression type passed to getFunctionName" );
		}
	}

	Type * getPointerBase( Type * type ) {
		if ( PointerType * ptrType = dynamic_cast< PointerType * >( type ) ) {
			return ptrType->get_base();
		} else if ( ArrayType * arrayType = dynamic_cast< ArrayType * >( type ) ) {
			return arrayType->get_base();
		} else {
			return NULL;
		}
	}

	Type * isPointerType( Type * type ) {
		if ( getPointerBase( type ) ) return type;
		else return NULL;
	}
}
