//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// main.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Fri May 15 23:12:02 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Jun 09 15:10:05 2015
// Update Count     : 68
//

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstdio>
#include <getopt.h>
#include "Parser/Parser.h"
#include "Parser/ParseNode.h"
#include "Parser/LinkageSpec.h"
#include "SynTree/Declaration.h"
#include "SynTree/Visitor.h"
#include "GenPoly/Lvalue.h"
#include "GenPoly/Specialize.h"
#include "GenPoly/Box.h"
#include "GenPoly/CopyParams.h"
#include "CodeGen/Generate.h"
#include "CodeGen/FixNames.h"
#include "ControlStruct/Mutate.h"
#include "Tuples/Mutate.h"
#include "Tuples/FunctionChecker.h"
#include "SymTab/Mangler.h"
#include "SymTab/Indexer.h"
#include "SymTab/Validate.h"
#include "ResolvExpr/AlternativePrinter.h"
#include "ResolvExpr/Resolver.h"
#include "MakeLibCfa.h"
#include "InitTweak/Mutate.h"
#include "InitTweak/RemoveInit.h"
//#include "Explain/GenProlog.h"
//#include "Try/Visit.h"

#include "SemanticError.h"
#include "UnimplementedError.h"

#include "../config.h"

using namespace std;

#define OPTPRINT(x) \
	if ( errorp ) std::cerr << x << std::endl;

void parse(FILE * input, LinkageSpec::Type t, bool shouldExit = false );

bool
	astp = false,
	exprp = false,
	expraltp = false,
	grammarp = false,
	libcfap = false,
	resolvep = false,									// used in AlternativeFinder
	symtabp = false,
	parsep = false,
	validp = false,
	preludep = true,
	protop = false,
	codegenp = false,
	errorp = false;

enum { Ast, Expr, ExprAlt, Grammar, LibCFA, Nopreamble, Prototypes, Resolver, Symbol, Parse, };

static struct option long_opts[] = {
	{ "ast", no_argument, 0, Ast },
	{ "expr", no_argument, 0, Expr },
	{ "expralt", no_argument, 0, ExprAlt },
	{ "grammar", no_argument, 0, Grammar },
	{ "libcfa", no_argument, 0, LibCFA },
	{ "nopreamble", no_argument, 0, Nopreamble },
	{ "prototypes", no_argument, 0, Prototypes },
	{ "resolver", no_argument, 0, Resolver },
	{ "symbol", no_argument, 0, Symbol },
	{ "parse", no_argument, 0, Parse },
	{ 0, 0, 0, 0 }
};

int main( int argc, char *argv[] ) {
	FILE *input;
	std::ostream *output = &std::cout;
	int long_index;
	std::list< Declaration* > translationUnit;

	opterr = 0;											// prevent getopt from printing error messages
	
	int c;
	while ( (c = getopt_long( argc, argv, "aefglnpqrsxyzD:", long_opts, &long_index )) != -1 ) {
		switch ( c ) {
		  case Ast:
		  case 'a':										// dump AST
			astp = true;
			break;
		  case Expr:
		  case 'e':										// dump AST after expression analysis
			exprp = true;
			break;
		  case ExprAlt:
		  case 'f':										// print alternatives for expressions
			expraltp = true;
			break;
		  case Grammar:
		  case 'g':										// bison debugging info (grammar rules)
			grammarp = true;
			break;
		  case LibCFA:
		  case 'l':										// generate libcfa.c
			libcfap = true;
			break;
		  case Nopreamble:
		  case 'n':										// do not read preamble
			preludep = false;
			break;
		  case Prototypes:
		  case 'p':										// generate prototypes for preamble functions
			protop = true;
			break;
		  case Parse:
		  case 'q':										// dump parse tree
			parsep = true;
			break;
		  case Resolver:
		  case 'r':										// print resolver steps
			resolvep = true;
			break;
		  case Symbol:
		  case 's':										// print symbol table events
			symtabp = true;
			break;
		  case 'x':										// dump AST after decl validation pass
			validp = true;
			break;
		  case 'y':
			errorp = true;
			break;
		  case 'z':
			codegenp = true;
			break;
		  case 'D':										// ignore -Dxxx
			break;
		  case '?':
			cout << "Unknown option: '" << (char)optopt << "'" << endl;
			exit(1);
		  default:
			abort();
		} // switch
	} // while

	try {
		// choose to read the program from a file or stdin
		if ( optind < argc ) {
			input = fopen( argv[ optind ], "r" );
			if ( ! input ) {
				std::cout << "Error: can't open " << argv[optind] << std::endl;
				exit( 1 );
			} // if
			optind += 1;
		} else {
			input = stdin;
		} // if

		if ( optind < argc ) {
			output = new ofstream( argv[ optind ] );
		} // if
	
		Parser::get_parser().set_debug( grammarp );

		// read in the builtins and the prelude
		if ( preludep ) {								// include gcc builtins
			FILE * builtins = fopen( CFA_LIBDIR "/builtins.cf", "r" );
			if ( builtins == NULL ) {
				std::cerr << "Error: can't open builtins" << std::endl;
				exit( 1 );
			} // if

			parse( builtins, LinkageSpec::Compiler );

			if ( ! libcfap ) {
				// read the prelude in, if we're not generating the cfa library
				FILE * prelude = fopen( CFA_LIBDIR "/prelude.cf", "r" );
				if ( prelude == NULL ) {
					std::cerr << "Error: can't open prelude" << std::endl;
					exit( 1 );
				} // if
		    
		    parse( prelude, LinkageSpec::Intrinsic );
			} // if
		} // if

		if ( libcfap ) {
			parse( input, LinkageSpec::Intrinsic );	
		} else {
			parse( input, LinkageSpec::Cforall, grammarp );	
		}
  
		if ( parsep ) {
			Parser::get_parser().get_parseTree()->printList( std::cout );
			Parser::get_parser().freeTree();
			return 0;
		} // if

		buildList( Parser::get_parser().get_parseTree(), translationUnit );

		Parser::get_parser().freeTree();
		if ( astp ) {
			printAll( translationUnit, std::cout );
			return 0;
		} // if

		// add the assignment statement after the 
		// initialization of a type parameter
		OPTPRINT( "tweak" )
		InitTweak::tweak( translationUnit );
		OPTPRINT( "validate" )
		SymTab::validate( translationUnit, symtabp );
		if ( symtabp ) {
			return 0;
		} // if

		if ( expraltp ) {
			ResolvExpr::AlternativePrinter printer( std::cout );
			acceptAll( translationUnit, printer );
			return 0;
		} // if

		if ( validp ) {
			printAll( translationUnit, std::cout );
			return 0;
		} // if

		OPTPRINT( "mutate" )
		ControlStruct::mutate( translationUnit );
		OPTPRINT( "fixNames" ) 
		CodeGen::fixNames( translationUnit );

		if ( libcfap ) {
			protop = true;
			// generate the bodies of cfa library functions
			LibCfa::makeLibCfa( translationUnit );
		} // if

		OPTPRINT( "resolve" )
		ResolvExpr::resolve( translationUnit );
		if ( exprp ) {
			printAll( translationUnit, std::cout );
		}

		OPTPRINT( "copyParams" );
		GenPoly::copyParams( translationUnit );
		OPTPRINT( "convertSpecializations" )
		GenPoly::convertSpecializations( translationUnit );		
		OPTPRINT( "convertLvalue" )
		GenPoly::convertLvalue( translationUnit );
		OPTPRINT( "box" )
		GenPoly::box( translationUnit );

    // print the tree right before code generation
		if ( codegenp ) {
			printAll( translationUnit, std::cout );
			return 0;
		} // if

		CodeGen::generate( translationUnit, *output, protop );

		if ( output != &std::cout ) {
			delete output;
		} // if

	} catch ( SemanticError &e ) {
		if ( errorp ) {
			printAll( translationUnit, std::cerr );
		}
		e.print( std::cerr );
		if ( output != &std::cout ) {
			delete output;
		} // if
		return 1;
	} catch ( UnimplementedError &e ) {
		std::cout << "Sorry, " << e.get_what() << " is not currently implemented" << std::endl;
		if ( output != &std::cout ) {
			delete output;
		} // if
		return 1;
	} catch ( CompilerError &e ) {
		std::cerr << "Compiler Error: " << e.get_what() << std::endl;
		std::cerr << "(please report bugs to " << std::endl;
		if ( output != &std::cout ) {
			delete output;
		} // if
		return 1;
	} // try

	return 0;
} // main

void parse(FILE * input, LinkageSpec::Type linkage, bool shouldExit) {
	Parser::get_parser().set_linkage( linkage );
	Parser::get_parser().parse( input );

	fclose( input );
	if ( shouldExit || Parser::get_parser().get_parseStatus() != 0 ) {
		exit( Parser::get_parser().get_parseStatus() );
	} // if
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End:  //
