//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// threads.c --
//
// Author           : Thierry Delisle
// Created On       : Mon Nov 28 12:27:26 2016
// Last Modified By : Thierry Delisle
// Last Modified On : Mon Nov 28 12:27:26 2016
// Update Count     : 0
//

extern "C" {
#include <stddef.h>
#include <malloc.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
}

#include "threads"
#include "assert"
#include "libhdr.h"

extern "C" { extern void coInvokeStub( void ); }

// minimum feasible stack size in bytes
#define MinStackSize 1000

static coroutine main_coroutine;
static coroutine* current_coroutine = &main_coroutine;

extern "C" {
	struct machine_context_t {
		void *SP;
		void *FP;
		void *PC;
	};
}

extern "C" { void CtxSwitch( void *from, void *to ) asm ("CtxSwitch"); }// assembler routine that performs the context switch

static size_t pageSize = 0;				// architecture pagesize

void ctxSwitchDirect(coroutine* src, coroutine* dst) {
	// THREAD_GETMEM( This )->disableInterrupts();

	// set state of current coroutine to inactive
	src->state = Inactive;

	// set new coroutine that task is executing
	current_coroutine = dst;			

	// context switch to specified coroutine
	CtxSwitch( src->stack.context, dst->stack.context );
	// when CtxSwitch returns we are back in the src coroutine		

	// set state of new coroutine to active
	src->state = Active;

	// THREAD_GETMEM( This )->enableInterrupts();
} //ctxSwitchDirect

void invokeCoroutine(coVtable* vtable, void* this);

forall(dtype T | coroutine_t(T))
void startCoroutine(T* this, void (*invoke)(coVtable*, void*));

// used by all constructors
void create_stack( coStack_t* this, unsigned int storageSize ) {
	//TEMP HACK do this on proper kernel startup
	if(pageSize == 0ul) pageSize = sysconf( _SC_PAGESIZE );

	size_t cxtSize = libCeiling( sizeof(machine_context_t), 8 ); // minimum alignment

	if ( (intptr_t)this->storage == 0 ) {
		this->userStack = false;
		this->size = libCeiling( storageSize, 16 );
		// use malloc/memalign because "new" raises an exception for out-of-memory
		
		// assume malloc has 8 byte alignment so add 8 to allow rounding up to 16 byte alignment
		LIB_DEBUG_DO( this->storage = memalign( pageSize, cxtSize + this->size + pageSize ) );
		LIB_NO_DEBUG_DO( this->storage = malloc( cxtSize + this->size + 8 ) );

		LIB_DEBUG_DO(
			if ( mprotect( this->storage, pageSize, PROT_NONE ) == -1 ) {
				abortf( "(uMachContext &)%p.createContext() : internal error, mprotect failure, error(%d) %s.", this, (int)errno, strerror( (int)errno ) );
			} // if
		);

		if ( (intptr_t)this->storage == 0 ) {
			abortf( "Attempt to allocate %d bytes of storage for coroutine or task execution-state but insufficient memory available.", this->size );
		} // if

		LIB_DEBUG_DO( this->limit = (char *)this->storage + pageSize );
		LIB_NO_DEBUG_DO( this->limit = (char *)libCeiling( (unsigned long)this->storage, 16 ) ); // minimum alignment

	} else {
		assertf( ((size_t)this->storage & (libAlign() - 1)) != 0ul, "Stack storage %p for task/coroutine must be aligned on %d byte boundary.", this->storage, (int)libAlign() );
		this->userStack = true;
		this->size = storageSize - cxtSize;

		if ( this->size % 16 != 0u ) this->size -= 8;

		this->limit = (char *)libCeiling( (unsigned long)this->storage, 16 ); // minimum alignment
	} // if
	assertf( this->size >= MinStackSize, "Stack size %d provides less than minimum of %d bytes for a stack.", this->size, MinStackSize );

	this->base = (char *)this->limit + this->size;
	this->context = this->base;
	this->top = (char *)this->context + cxtSize;
}

coroutine* this_coroutine() {
	return current_coroutine;
}

void ?{}(coStack_t* this) {
	this->size		= 10240;	// size of stack
	this->storage	= NULL;	// pointer to stack
	this->limit		= NULL;	// stack grows towards stack limit
	this->base		= NULL;	// base of stack
	this->context	= NULL;	// address of cfa_context_t
	this->top		= NULL;	// address of top of storage
	this->userStack	= false;	
	create_stack(this, this->size);
}

void ?{}(coroutine* this)
{
	this->name = "Anonymous Coroutine";
	this->errno_ = 0;
	this->state = Start;
      this->notHalted = true;
	this->starter = NULL;
	this->last = NULL;
}

forall(dtype T | coroutine_t(T))
void start(T* this) {
	startCoroutine(this, invokeCoroutine);
}

void suspend() {
      coroutine* src = this_coroutine();		// optimization

	assertf( src->last != 0, 
		"Attempt to suspend coroutine %.256s (%p) that has never been resumed.\n"
		"Possible cause is a suspend executed in a member called by a coroutine user rather than by the coroutine main.",
		src->name, src );
	assertf( src->last->notHalted, 
		"Attempt by coroutine %.256s (%p) to suspend back to terminated coroutine %.256s (%p).\n"
		"Possible cause is terminated coroutine's main routine has already returned.",
		src->name, src, src->last->name, src->last );

	ctxSwitchDirect( src, src->last );
}

forall(dtype T | coroutine_t(T))
void resume(T* cor) {
	coroutine* src = this_coroutine();		// optimization
	coroutine* dst = this_coroutine(cor);

	fprintf(stderr, "Resuming %p from %p\n", dst, src);
	if ( src != dst ) {				// not resuming self ?
		assertf( dst->notHalted , 
			"Attempt by coroutine %.256s (%p) to resume terminated coroutine %.256s (%p).\n"
			"Possible cause is terminated coroutine's main routine has already returned.",
			src->name, src, dst->name, dst );
		fprintf(stderr, "Assigning last pointer\n");
		dst->last = src;					// set last resumer
	} // if
	ctxSwitchDirect( src, dst );				// always done for performance testing
}

// Local Variables: //
// mode: c //
// tab-width: 4 //
// End: //