//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Validate.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:50:04 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Aug 28 13:47:23 2017
// Update Count     : 359
//

// The "validate" phase of translation is used to take a syntax tree and convert it into a standard form that aims to be
// as regular in structure as possible.  Some assumptions can be made regarding the state of the tree after this pass is
// complete, including:
//
// - No nested structure or union definitions; any in the input are "hoisted" to the level of the containing struct or
//   union.
//
// - All enumeration constants have type EnumInstType.
//
// - The type "void" never occurs in lists of function parameter or return types.  A function
//   taking no arguments has no argument types.
//
// - No context instances exist; they are all replaced by the set of declarations signified by the context, instantiated
//   by the particular set of type arguments.
//
// - Every declaration is assigned a unique id.
//
// - No typedef declarations or instances exist; the actual type is substituted for each instance.
//
// - Each type, struct, and union definition is followed by an appropriate assignment operator.
//
// - Each use of a struct or union is connected to a complete definition of that struct or union, even if that
//   definition occurs later in the input.

#include "Validate.h"

#include <cassert>                     // for assertf, assert
#include <cstddef>                     // for size_t
#include <list>                        // for list
#include <string>                      // for string
#include <utility>                     // for pair

#include "CodeGen/CodeGenerator.h"     // for genName
#include "CodeGen/OperatorTable.h"     // for isCtorDtor, isCtorDtorAssign
#include "Common/PassVisitor.h"        // for PassVisitor, WithDeclsToAdd
#include "Common/ScopedMap.h"          // for ScopedMap
#include "Common/SemanticError.h"      // for SemanticError
#include "Common/UniqueName.h"         // for UniqueName
#include "Common/utility.h"            // for operator+, cloneAll, deleteAll
#include "Concurrency/Keywords.h"      // for applyKeywords
#include "FixFunction.h"               // for FixFunction
#include "Indexer.h"                   // for Indexer
#include "InitTweak/GenInit.h"         // for fixReturnStatements
#include "InitTweak/InitTweak.h"       // for isCtorDtorAssign
#include "Parser/LinkageSpec.h"        // for C
#include "ResolvExpr/typeops.h"        // for typesCompatible
#include "SymTab/AddVisit.h"           // for addVisit
#include "SymTab/Autogen.h"            // for SizeType
#include "SynTree/Attribute.h"         // for noAttributes, Attribute
#include "SynTree/Constant.h"          // for Constant
#include "SynTree/Declaration.h"       // for ObjectDecl, DeclarationWithType
#include "SynTree/Expression.h"        // for CompoundLiteralExpr, Expressio...
#include "SynTree/Initializer.h"       // for ListInit, Initializer
#include "SynTree/Label.h"             // for operator==, Label
#include "SynTree/Mutator.h"           // for Mutator
#include "SynTree/Type.h"              // for Type, TypeInstType, EnumInstType
#include "SynTree/TypeSubstitution.h"  // for TypeSubstitution
#include "SynTree/Visitor.h"           // for Visitor

class CompoundStmt;
class ReturnStmt;
class SwitchStmt;


#define debugPrint( x ) if ( doDebug ) { std::cout << x; }

namespace SymTab {
	struct HoistStruct final : public WithDeclsToAdd, public WithGuards {
		/// Flattens nested struct types
		static void hoistStruct( std::list< Declaration * > &translationUnit );

		void previsit( EnumInstType * enumInstType );
		void previsit( StructInstType * structInstType );
		void previsit( UnionInstType * unionInstType );
		void previsit( StructDecl * aggregateDecl );
		void previsit( UnionDecl * aggregateDecl );

	  private:
		template< typename AggDecl > void handleAggregate( AggDecl *aggregateDecl );

		AggregateDecl * parentAggr = nullptr;
	};

	/// Fix return types so that every function returns exactly one value
	struct ReturnTypeFixer {
		static void fix( std::list< Declaration * > &translationUnit );

		void postvisit( FunctionDecl * functionDecl );
		void postvisit( FunctionType * ftype );
	};

	/// Replaces enum types by int, and function or array types in function parameter and return lists by appropriate pointers.
	struct EnumAndPointerDecay {
		void previsit( EnumDecl *aggregateDecl );
		void previsit( FunctionType *func );
	};

	/// Associates forward declarations of aggregates with their definitions
	struct LinkReferenceToTypes final : public WithIndexer, public WithGuards {
		LinkReferenceToTypes( const Indexer *indexer );
		void postvisit( TypeInstType *typeInst );

		void postvisit( EnumInstType *enumInst );
		void postvisit( StructInstType *structInst );
		void postvisit( UnionInstType *unionInst );
		void postvisit( TraitInstType *traitInst );

		void postvisit( EnumDecl *enumDecl );
		void postvisit( StructDecl *structDecl );
		void postvisit( UnionDecl *unionDecl );
		void postvisit( TraitDecl * traitDecl );

		void previsit( StructDecl *structDecl );
		void previsit( UnionDecl *unionDecl );

		void renameGenericParams( std::list< TypeDecl * > & params );

	  private:
		const Indexer *local_indexer;

		typedef std::map< std::string, std::list< EnumInstType * > > ForwardEnumsType;
		typedef std::map< std::string, std::list< StructInstType * > > ForwardStructsType;
		typedef std::map< std::string, std::list< UnionInstType * > > ForwardUnionsType;
		ForwardEnumsType forwardEnums;
		ForwardStructsType forwardStructs;
		ForwardUnionsType forwardUnions;
		/// true if currently in a generic type body, so that type parameter instances can be renamed appropriately
		bool inGeneric = false;
	};

	/// Replaces array and function types in forall lists by appropriate pointer type and assigns each Object and Function declaration a unique ID.
	struct ForallPointerDecay final {
		void previsit( ObjectDecl * object );
		void previsit( FunctionDecl * func );
		void previsit( StructDecl * aggrDecl );
		void previsit( UnionDecl * aggrDecl );
	};

	struct ReturnChecker : public WithGuards {
		/// Checks that return statements return nothing if their return type is void
		/// and return something if the return type is non-void.
		static void checkFunctionReturns( std::list< Declaration * > & translationUnit );

		void previsit( FunctionDecl * functionDecl );
		void previsit( ReturnStmt * returnStmt );

		typedef std::list< DeclarationWithType * > ReturnVals;
		ReturnVals returnVals;
	};

	struct EliminateTypedef final : public WithVisitorRef<EliminateTypedef>, public WithGuards {
		EliminateTypedef() : scopeLevel( 0 ) {}
		/// Replaces typedefs by forward declarations
		static void eliminateTypedef( std::list< Declaration * > &translationUnit );

		Type * postmutate( TypeInstType * aggregateUseType );
		Declaration * postmutate( TypedefDecl * typeDecl );
		void premutate( TypeDecl * typeDecl );
		void premutate( FunctionDecl * funcDecl );
		void premutate( ObjectDecl * objDecl );
		DeclarationWithType * postmutate( ObjectDecl * objDecl );

		void premutate( CastExpr * castExpr );

		void premutate( CompoundStmt * compoundStmt );
		CompoundStmt * postmutate( CompoundStmt * compoundStmt );

		void premutate( StructDecl * structDecl );
		Declaration * postmutate( StructDecl * structDecl );
		void premutate( UnionDecl * unionDecl );
		Declaration * postmutate( UnionDecl * unionDecl );
		void premutate( EnumDecl * enumDecl );
		Declaration * postmutate( EnumDecl * enumDecl );
		Declaration * postmutate( TraitDecl * contextDecl );

		void premutate( FunctionType * ftype );

	  private:
		template<typename AggDecl>
		AggDecl *handleAggregate( AggDecl * aggDecl );

		template<typename AggDecl>
		void addImplicitTypedef( AggDecl * aggDecl );

		typedef std::unique_ptr<TypedefDecl> TypedefDeclPtr;
		typedef ScopedMap< std::string, std::pair< TypedefDeclPtr, int > > TypedefMap;
		typedef std::map< std::string, TypeDecl * > TypeDeclMap;
		TypedefMap typedefNames;
		TypeDeclMap typedeclNames;
		int scopeLevel;
		bool inFunctionType = false;
	};

	struct VerifyCtorDtorAssign {
		/// ensure that constructors, destructors, and assignment have at least one
		/// parameter, the first of which must be a pointer, and that ctor/dtors have no
		/// return values.
		static void verify( std::list< Declaration * > &translationUnit );

		void previsit( FunctionDecl *funcDecl );
	};

	/// ensure that generic types have the correct number of type arguments
	struct ValidateGenericParameters {
		void previsit( StructInstType * inst );
		void previsit( UnionInstType * inst );
	};

	struct ArrayLength {
		/// for array types without an explicit length, compute the length and store it so that it
		/// is known to the rest of the phases. For example,
		///   int x[] = { 1, 2, 3 };
		///   int y[][2] = { { 1, 2, 3 }, { 1, 2, 3 } };
		/// here x and y are known at compile-time to have length 3, so change this into
		///   int x[3] = { 1, 2, 3 };
		///   int y[3][2] = { { 1, 2, 3 }, { 1, 2, 3 } };
		static void computeLength( std::list< Declaration * > & translationUnit );

		void previsit( ObjectDecl * objDecl );
	};

	struct CompoundLiteral final : public WithDeclsToAdd, public WithVisitorRef<CompoundLiteral> {
		Type::StorageClasses storageClasses;

		void premutate( ObjectDecl *objectDecl );
		Expression * postmutate( CompoundLiteralExpr *compLitExpr );
	};

	struct LabelAddressFixer final : public WithGuards {
		std::set< Label > labels;

		void premutate( FunctionDecl * funcDecl );
		Expression * postmutate( AddressExpr * addrExpr );
	};

	FunctionDecl * dereferenceOperator = nullptr;
	struct FindSpecialDeclarations final {
		void previsit( FunctionDecl * funcDecl );
	};

	void validate( std::list< Declaration * > &translationUnit, __attribute__((unused)) bool doDebug ) {
		PassVisitor<EnumAndPointerDecay> epc;
		PassVisitor<LinkReferenceToTypes> lrt( nullptr );
		PassVisitor<ForallPointerDecay> fpd;
		PassVisitor<CompoundLiteral> compoundliteral;
		PassVisitor<ValidateGenericParameters> genericParams;
		PassVisitor<FindSpecialDeclarations> finder;
		PassVisitor<LabelAddressFixer> labelAddrFixer;

		EliminateTypedef::eliminateTypedef( translationUnit );
		HoistStruct::hoistStruct( translationUnit ); // must happen after EliminateTypedef, so that aggregate typedefs occur in the correct order
		ReturnTypeFixer::fix( translationUnit ); // must happen before autogen
		acceptAll( translationUnit, epc ); // must happen before VerifyCtorDtorAssign, because void return objects should not exist; before LinkReferenceToTypes because it is an indexer and needs correct types for mangling
		acceptAll( translationUnit, lrt ); // must happen before autogen, because sized flag needs to propagate to generated functions
		acceptAll( translationUnit, genericParams );  // check as early as possible - can't happen before LinkReferenceToTypes
		VerifyCtorDtorAssign::verify( translationUnit );  // must happen before autogen, because autogen examines existing ctor/dtors
		ReturnChecker::checkFunctionReturns( translationUnit );
		InitTweak::fixReturnStatements( translationUnit ); // must happen before autogen
		Concurrency::applyKeywords( translationUnit );
		acceptAll( translationUnit, fpd ); // must happen before autogenerateRoutines, after Concurrency::applyKeywords because uniqueIds must be set on declaration before resolution
		autogenerateRoutines( translationUnit ); // moved up, used to be below compoundLiteral - currently needs EnumAndPointerDecay
		Concurrency::implementMutexFuncs( translationUnit );
		Concurrency::implementThreadStarter( translationUnit );
		mutateAll( translationUnit, compoundliteral );
		ArrayLength::computeLength( translationUnit );
		acceptAll( translationUnit, finder ); // xxx - remove this pass soon
		mutateAll( translationUnit, labelAddrFixer );
	}

	void validateType( Type *type, const Indexer *indexer ) {
		PassVisitor<EnumAndPointerDecay> epc;
		PassVisitor<LinkReferenceToTypes> lrt( indexer );
		PassVisitor<ForallPointerDecay> fpd;
		type->accept( epc );
		type->accept( lrt );
		type->accept( fpd );
	}

	void HoistStruct::hoistStruct( std::list< Declaration * > &translationUnit ) {
		PassVisitor<HoistStruct> hoister;
		acceptAll( translationUnit, hoister );
	}

	bool isStructOrUnion( Declaration *decl ) {
		return dynamic_cast< StructDecl * >( decl ) || dynamic_cast< UnionDecl * >( decl );
	}

	template< typename AggDecl >
	void HoistStruct::handleAggregate( AggDecl *aggregateDecl ) {
		if ( parentAggr ) {
			// Add elements in stack order corresponding to nesting structure.
			declsToAddBefore.push_front( aggregateDecl );
		} else {
			GuardValue( parentAggr );
			parentAggr = aggregateDecl;
		} // if
		// Always remove the hoisted aggregate from the inner structure.
		GuardAction( [aggregateDecl]() { filter( aggregateDecl->members, isStructOrUnion, false ); } );
	}

	void HoistStruct::previsit( EnumInstType * inst ) {
		if ( inst->baseEnum ) {
			declsToAddBefore.push_front( inst->baseEnum );
		}
	}

	void HoistStruct::previsit( StructInstType * inst ) {
		if ( inst->baseStruct ) {
			declsToAddBefore.push_front( inst->baseStruct );
		}
	}

	void HoistStruct::previsit( UnionInstType * inst ) {
		if ( inst->baseUnion ) {
			declsToAddBefore.push_front( inst->baseUnion );
		}
	}

	void HoistStruct::previsit( StructDecl * aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void HoistStruct::previsit( UnionDecl * aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void EnumAndPointerDecay::previsit( EnumDecl *enumDecl ) {
		// Set the type of each member of the enumeration to be EnumConstant
		for ( std::list< Declaration * >::iterator i = enumDecl->get_members().begin(); i != enumDecl->get_members().end(); ++i ) {
			ObjectDecl * obj = dynamic_cast< ObjectDecl * >( *i );
			assert( obj );
			obj->set_type( new EnumInstType( Type::Qualifiers( Type::Const ), enumDecl->get_name() ) );
		} // for
	}

	namespace {
		template< typename DWTList >
		void fixFunctionList( DWTList & dwts, FunctionType * func ) {
			// the only case in which "void" is valid is where it is the only one in the list; then it should be removed
			// entirely. other fix ups are handled by the FixFunction class
			typedef typename DWTList::iterator DWTIterator;
			DWTIterator begin( dwts.begin() ), end( dwts.end() );
			if ( begin == end ) return;
			PassVisitor<FixFunction> fixer;
			DWTIterator i = begin;
			*i = (*i)->acceptMutator( fixer );
			if ( fixer.pass.isVoid ) {
				DWTIterator j = i;
				++i;
				delete *j;
				dwts.erase( j );
				if ( i != end ) {
					throw SemanticError( func, "invalid type void in function type " );
				} // if
			} else {
				++i;
				for ( ; i != end; ++i ) {
					PassVisitor<FixFunction> fixer;
					*i = (*i)->acceptMutator( fixer );
					if ( fixer.pass.isVoid ) {
						throw SemanticError( func, "invalid type void in function type " );
					} // if
				} // for
			} // if
		}
	}

	void EnumAndPointerDecay::previsit( FunctionType *func ) {
		// Fix up parameters and return types
		fixFunctionList( func->get_parameters(), func );
		fixFunctionList( func->get_returnVals(), func );
	}

	LinkReferenceToTypes::LinkReferenceToTypes( const Indexer *other_indexer ) {
		if ( other_indexer ) {
			local_indexer = other_indexer;
		} else {
			local_indexer = &indexer;
		} // if
	}

	void LinkReferenceToTypes::postvisit( EnumInstType *enumInst ) {
		EnumDecl *st = local_indexer->lookupEnum( enumInst->get_name() );
		// it's not a semantic error if the enum is not found, just an implicit forward declaration
		if ( st ) {
			//assert( ! enumInst->get_baseEnum() || enumInst->get_baseEnum()->get_members().empty() || ! st->get_members().empty() );
			enumInst->set_baseEnum( st );
		} // if
		if ( ! st || st->get_members().empty() ) {
			// use of forward declaration
			forwardEnums[ enumInst->get_name() ].push_back( enumInst );
		} // if
	}

	void checkGenericParameters( ReferenceToType * inst ) {
		for ( Expression * param : inst->parameters ) {
			if ( ! dynamic_cast< TypeExpr * >( param ) ) {
				throw SemanticError( inst, "Expression parameters for generic types are currently unsupported: " );
			}
		}
	}

	void LinkReferenceToTypes::postvisit( StructInstType *structInst ) {
		StructDecl *st = local_indexer->lookupStruct( structInst->get_name() );
		// it's not a semantic error if the struct is not found, just an implicit forward declaration
		if ( st ) {
			//assert( ! structInst->get_baseStruct() || structInst->get_baseStruct()->get_members().empty() || ! st->get_members().empty() );
			structInst->set_baseStruct( st );
		} // if
		if ( ! st || st->get_members().empty() ) {
			// use of forward declaration
			forwardStructs[ structInst->get_name() ].push_back( structInst );
		} // if
		checkGenericParameters( structInst );
	}

	void LinkReferenceToTypes::postvisit( UnionInstType *unionInst ) {
		UnionDecl *un = local_indexer->lookupUnion( unionInst->get_name() );
		// it's not a semantic error if the union is not found, just an implicit forward declaration
		if ( un ) {
			unionInst->set_baseUnion( un );
		} // if
		if ( ! un || un->get_members().empty() ) {
			// use of forward declaration
			forwardUnions[ unionInst->get_name() ].push_back( unionInst );
		} // if
		checkGenericParameters( unionInst );
	}

	template< typename Decl >
	void normalizeAssertions( std::list< Decl * > & assertions ) {
		// ensure no duplicate trait members after the clone
		auto pred = [](Decl * d1, Decl * d2) {
			// only care if they're equal
			DeclarationWithType * dwt1 = dynamic_cast<DeclarationWithType *>( d1 );
			DeclarationWithType * dwt2 = dynamic_cast<DeclarationWithType *>( d2 );
			if ( dwt1 && dwt2 ) {
				if ( dwt1->get_name() == dwt2->get_name() && ResolvExpr::typesCompatible( dwt1->get_type(), dwt2->get_type(), SymTab::Indexer() ) ) {
					// std::cerr << "=========== equal:" << std::endl;
					// std::cerr << "d1: " << d1 << std::endl;
					// std::cerr << "d2: " << d2 << std::endl;
					return false;
				}
			}
			return d1 < d2;
		};
		std::set<Decl *, decltype(pred)> unique_members( assertions.begin(), assertions.end(), pred );
		// if ( unique_members.size() != assertions.size() ) {
		// 	std::cerr << "============different" << std::endl;
		// 	std::cerr << unique_members.size() << " " << assertions.size() << std::endl;
		// }

		std::list< Decl * > order;
		order.splice( order.end(), assertions );
		std::copy_if( order.begin(), order.end(), back_inserter( assertions ), [&]( Decl * decl ) {
			return unique_members.count( decl );
		});
	}

	// expand assertions from trait instance, performing the appropriate type variable substitutions
	template< typename Iterator >
	void expandAssertions( TraitInstType * inst, Iterator out ) {
		assertf( inst->baseTrait, "Trait instance not linked to base trait: %s", toString( inst ).c_str() );
		std::list< DeclarationWithType * > asserts;
		for ( Declaration * decl : inst->baseTrait->members ) {
			asserts.push_back( strict_dynamic_cast<DeclarationWithType *>( decl->clone() ) );
		}
		// substitute trait decl parameters for instance parameters
		applySubstitution( inst->baseTrait->parameters.begin(), inst->baseTrait->parameters.end(), inst->parameters.begin(), asserts.begin(), asserts.end(), out );
	}

	void LinkReferenceToTypes::postvisit( TraitDecl * traitDecl ) {
		if ( traitDecl->name == "sized" ) {
			// "sized" is a special trait - flick the sized status on for the type variable
			assertf( traitDecl->parameters.size() == 1, "Built-in trait 'sized' has incorrect number of parameters: %zd", traitDecl->parameters.size() );
			TypeDecl * td = traitDecl->parameters.front();
			td->set_sized( true );
		}

		// move assertions from type parameters into the body of the trait
		for ( TypeDecl * td : traitDecl->parameters ) {
			for ( DeclarationWithType * assert : td->assertions ) {
				if ( TraitInstType * inst = dynamic_cast< TraitInstType * >( assert->get_type() ) ) {
					expandAssertions( inst, back_inserter( traitDecl->members ) );
				} else {
					traitDecl->members.push_back( assert->clone() );
				}
			}
			deleteAll( td->assertions );
			td->assertions.clear();
		} // for
	}

	void LinkReferenceToTypes::postvisit( TraitInstType * traitInst ) {
		// handle other traits
		TraitDecl *traitDecl = local_indexer->lookupTrait( traitInst->name );
		if ( ! traitDecl ) {
			throw SemanticError( traitInst->location, "use of undeclared trait " + traitInst->name );
		} // if
		if ( traitDecl->get_parameters().size() != traitInst->get_parameters().size() ) {
			throw SemanticError( traitInst, "incorrect number of trait parameters: " );
		} // if
		traitInst->baseTrait = traitDecl;

		// need to carry over the 'sized' status of each decl in the instance
		for ( auto p : group_iterate( traitDecl->get_parameters(), traitInst->get_parameters() ) ) {
			TypeExpr * expr = dynamic_cast< TypeExpr * >( std::get<1>(p) );
			if ( ! expr ) {
				throw SemanticError( std::get<1>(p), "Expression parameters for trait instances are currently unsupported: " );
			}
			if ( TypeInstType * inst = dynamic_cast< TypeInstType * >( expr->get_type() ) ) {
				TypeDecl * formalDecl = std::get<0>(p);
				TypeDecl * instDecl = inst->get_baseType();
				if ( formalDecl->get_sized() ) instDecl->set_sized( true );
			}
		}
		// normalizeAssertions( traitInst->members );
	}

	void LinkReferenceToTypes::postvisit( EnumDecl *enumDecl ) {
		// visit enum members first so that the types of self-referencing members are updated properly
		if ( ! enumDecl->get_members().empty() ) {
			ForwardEnumsType::iterator fwds = forwardEnums.find( enumDecl->get_name() );
			if ( fwds != forwardEnums.end() ) {
				for ( std::list< EnumInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst )->set_baseEnum( enumDecl );
				} // for
				forwardEnums.erase( fwds );
			} // if
		} // if
	}

	void LinkReferenceToTypes::renameGenericParams( std::list< TypeDecl * > & params ) {
		// rename generic type parameters uniquely so that they do not conflict with user-defined function forall parameters, e.g.
		//   forall(otype T)
		//   struct Box {
		//     T x;
		//   };
		//   forall(otype T)
		//   void f(Box(T) b) {
		//     ...
		//   }
		// The T in Box and the T in f are different, so internally the naming must reflect that.
		GuardValue( inGeneric );
		inGeneric = ! params.empty();
		for ( TypeDecl * td : params ) {
			td->name = "__" + td->name + "_generic_";
		}
	}

	void LinkReferenceToTypes::previsit( StructDecl * structDecl ) {
		renameGenericParams( structDecl->parameters );
	}

	void LinkReferenceToTypes::previsit( UnionDecl * unionDecl ) {
		renameGenericParams( unionDecl->parameters );
	}

	void LinkReferenceToTypes::postvisit( StructDecl *structDecl ) {
		// visit struct members first so that the types of self-referencing members are updated properly
		// xxx - need to ensure that type parameters match up between forward declarations and definition (most importantly, number of type parameters and their defaults)
		if ( ! structDecl->get_members().empty() ) {
			ForwardStructsType::iterator fwds = forwardStructs.find( structDecl->get_name() );
			if ( fwds != forwardStructs.end() ) {
				for ( std::list< StructInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst )->set_baseStruct( structDecl );
				} // for
				forwardStructs.erase( fwds );
			} // if
		} // if
	}

	void LinkReferenceToTypes::postvisit( UnionDecl *unionDecl ) {
		if ( ! unionDecl->get_members().empty() ) {
			ForwardUnionsType::iterator fwds = forwardUnions.find( unionDecl->get_name() );
			if ( fwds != forwardUnions.end() ) {
				for ( std::list< UnionInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst )->set_baseUnion( unionDecl );
				} // for
				forwardUnions.erase( fwds );
			} // if
		} // if
	}

	void LinkReferenceToTypes::postvisit( TypeInstType *typeInst ) {
		// ensure generic parameter instances are renamed like the base type
		if ( inGeneric && typeInst->baseType ) typeInst->name = typeInst->baseType->name;
		if ( NamedTypeDecl *namedTypeDecl = local_indexer->lookupType( typeInst->get_name() ) ) {
			if ( TypeDecl *typeDecl = dynamic_cast< TypeDecl * >( namedTypeDecl ) ) {
				typeInst->set_isFtype( typeDecl->get_kind() == TypeDecl::Ftype );
			} // if
		} // if
	}

	/// Fix up assertions - flattens assertion lists, removing all trait instances
	void forallFixer( std::list< TypeDecl * > & forall, BaseSyntaxNode * node ) {
		for ( TypeDecl * type : forall ) {
			std::list< DeclarationWithType * > asserts;
			asserts.splice( asserts.end(), type->assertions );
			// expand trait instances into their members
			for ( DeclarationWithType * assertion : asserts ) {
				if ( TraitInstType *traitInst = dynamic_cast< TraitInstType * >( assertion->get_type() ) ) {
					// expand trait instance into all of its members
					expandAssertions( traitInst, back_inserter( type->assertions ) );
					delete traitInst;
				} else {
					// pass other assertions through
					type->assertions.push_back( assertion );
				} // if
			} // for
			// apply FixFunction to every assertion to check for invalid void type
			for ( DeclarationWithType *& assertion : type->assertions ) {
				PassVisitor<FixFunction> fixer;
				assertion = assertion->acceptMutator( fixer );
				if ( fixer.pass.isVoid ) {
					throw SemanticError( node, "invalid type void in assertion of function " );
				} // if
			} // for
			// normalizeAssertions( type->assertions );
		} // for
	}

	void ForallPointerDecay::previsit( ObjectDecl *object ) {
		forallFixer( object->type->forall, object );
		if ( PointerType *pointer = dynamic_cast< PointerType * >( object->type ) ) {
			forallFixer( pointer->base->forall, object );
		} // if
		object->fixUniqueId();
	}

	void ForallPointerDecay::previsit( FunctionDecl *func ) {
		forallFixer( func->type->forall, func );
		func->fixUniqueId();
	}

	void ForallPointerDecay::previsit( StructDecl * aggrDecl ) {
		forallFixer( aggrDecl->parameters, aggrDecl );
	}

	void ForallPointerDecay::previsit( UnionDecl * aggrDecl ) {
		forallFixer( aggrDecl->parameters, aggrDecl );
	}

	void ReturnChecker::checkFunctionReturns( std::list< Declaration * > & translationUnit ) {
		PassVisitor<ReturnChecker> checker;
		acceptAll( translationUnit, checker );
	}

	void ReturnChecker::previsit( FunctionDecl * functionDecl ) {
		GuardValue( returnVals );
		returnVals = functionDecl->get_functionType()->get_returnVals();
	}

	void ReturnChecker::previsit( ReturnStmt * returnStmt ) {
		// Previously this also checked for the existence of an expr paired with no return values on
		// the  function return type. This is incorrect, since you can have an expression attached to
		// a return statement in a void-returning function in C. The expression is treated as if it
		// were cast to void.
		if ( ! returnStmt->get_expr() && returnVals.size() != 0 ) {
			throw SemanticError( returnStmt, "Non-void function returns no values: " );
		}
	}


	bool isTypedef( Declaration *decl ) {
		return dynamic_cast< TypedefDecl * >( decl );
	}

	void EliminateTypedef::eliminateTypedef( std::list< Declaration * > &translationUnit ) {
		PassVisitor<EliminateTypedef> eliminator;
		mutateAll( translationUnit, eliminator );
		if ( eliminator.pass.typedefNames.count( "size_t" ) ) {
			// grab and remember declaration of size_t
			SizeType = eliminator.pass.typedefNames["size_t"].first->get_base()->clone();
		} else {
			// xxx - missing global typedef for size_t - default to long unsigned int, even though that may be wrong
			// eventually should have a warning for this case.
			SizeType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );
		}
		filter( translationUnit, isTypedef, true );
	}

	Type * EliminateTypedef::postmutate( TypeInstType * typeInst ) {
		// instances of typedef types will come here. If it is an instance
		// of a typdef type, link the instance to its actual type.
		TypedefMap::const_iterator def = typedefNames.find( typeInst->get_name() );
		if ( def != typedefNames.end() ) {
			Type *ret = def->second.first->base->clone();
			ret->get_qualifiers() |= typeInst->get_qualifiers();
			// attributes are not carried over from typedef to function parameters/return values
			if ( ! inFunctionType ) {
				ret->attributes.splice( ret->attributes.end(), typeInst->attributes );
			} else {
				deleteAll( ret->attributes );
				ret->attributes.clear();
			}
			// place instance parameters on the typedef'd type
			if ( ! typeInst->parameters.empty() ) {
				ReferenceToType *rtt = dynamic_cast<ReferenceToType*>(ret);
				if ( ! rtt ) {
					throw SemanticError( typeInst->location, "Cannot apply type parameters to base type of " + typeInst->name );
				}
				rtt->get_parameters().clear();
				cloneAll( typeInst->parameters, rtt->parameters );
				mutateAll( rtt->parameters, *visitor );  // recursively fix typedefs on parameters
			} // if
			delete typeInst;
			return ret;
		} else {
			TypeDeclMap::const_iterator base = typedeclNames.find( typeInst->get_name() );
			assertf( base != typedeclNames.end(), "Cannot find typedecl name %s", typeInst->name.c_str() );
			typeInst->set_baseType( base->second );
		} // if
		return typeInst;
	}

	struct VarLenChecker : WithShortCircuiting {
		void previsit( FunctionType * ) { visit_children = false; }
		void previsit( ArrayType * at ) {
			isVarLen |= at->isVarLen;
		}
		bool isVarLen = false;
	};

	bool isVariableLength( Type * t ) {
		PassVisitor<VarLenChecker> varLenChecker;
		maybeAccept( t, varLenChecker );
		return varLenChecker.pass.isVarLen;
	}

	Declaration *EliminateTypedef::postmutate( TypedefDecl * tyDecl ) {
		if ( typedefNames.count( tyDecl->get_name() ) == 1 && typedefNames[ tyDecl->get_name() ].second == scopeLevel ) {
			// typedef to the same name from the same scope
			// must be from the same type

			Type * t1 = tyDecl->get_base();
			Type * t2 = typedefNames[ tyDecl->get_name() ].first->get_base();
			if ( ! ResolvExpr::typesCompatible( t1, t2, Indexer() ) ) {
				throw SemanticError( tyDecl->location, "Cannot redefine typedef: " + tyDecl->name );
			}
			// Cannot redefine VLA typedefs. Note: this is slightly incorrect, because our notion of VLAs
			// at this point in the translator is imprecise. In particular, this will disallow redefining typedefs
			// with arrays whose dimension is an enumerator or a cast of a constant/enumerator. The effort required
			// to fix this corner case likely outweighs the utility of allowing it.
			if ( isVariableLength( t1 ) || isVariableLength( t2 ) ) {
				throw SemanticError( tyDecl->location, "Cannot redefine typedef: " + tyDecl->name );
			}
		} else {
			typedefNames[ tyDecl->get_name() ] = std::make_pair( TypedefDeclPtr( tyDecl ), scopeLevel );
		} // if

		// When a typedef is a forward declaration:
		//    typedef struct screen SCREEN;
		// the declaration portion must be retained:
		//    struct screen;
		// because the expansion of the typedef is:
		//    void rtn( SCREEN *p ) => void rtn( struct screen *p )
		// hence the type-name "screen" must be defined.
		// Note, qualifiers on the typedef are superfluous for the forward declaration.

		Type *designatorType = tyDecl->get_base()->stripDeclarator();
		if ( StructInstType *aggDecl = dynamic_cast< StructInstType * >( designatorType ) ) {
			return new StructDecl( aggDecl->get_name(), DeclarationNode::Struct, noAttributes, tyDecl->get_linkage() );
		} else if ( UnionInstType *aggDecl = dynamic_cast< UnionInstType * >( designatorType ) ) {
			return new UnionDecl( aggDecl->get_name(), noAttributes, tyDecl->get_linkage() );
		} else if ( EnumInstType *enumDecl = dynamic_cast< EnumInstType * >( designatorType ) ) {
			return new EnumDecl( enumDecl->get_name(), noAttributes, tyDecl->get_linkage() );
		} else {
			return tyDecl->clone();
		} // if
	}

	void EliminateTypedef::premutate( TypeDecl * typeDecl ) {
		TypedefMap::iterator i = typedefNames.find( typeDecl->get_name() );
		if ( i != typedefNames.end() ) {
			typedefNames.erase( i ) ;
		} // if

		typedeclNames[ typeDecl->get_name() ] = typeDecl;
	}

	void EliminateTypedef::premutate( FunctionDecl * ) {
		GuardScope( typedefNames );
	}

	void EliminateTypedef::premutate( ObjectDecl * ) {
		GuardScope( typedefNames );
	}

	DeclarationWithType *EliminateTypedef::postmutate( ObjectDecl * objDecl ) {
		if ( FunctionType *funtype = dynamic_cast<FunctionType *>( objDecl->get_type() ) ) { // function type?
			// replace the current object declaration with a function declaration
			FunctionDecl * newDecl = new FunctionDecl( objDecl->get_name(), objDecl->get_storageClasses(), objDecl->get_linkage(), funtype, 0, objDecl->get_attributes(), objDecl->get_funcSpec() );
			objDecl->get_attributes().clear();
			objDecl->set_type( nullptr );
			delete objDecl;
			return newDecl;
		} // if
		return objDecl;
	}

	void EliminateTypedef::premutate( CastExpr * ) {
		GuardScope( typedefNames );
	}

	void EliminateTypedef::premutate( CompoundStmt * ) {
		GuardScope( typedefNames );
		scopeLevel += 1;
		GuardAction( [this](){ scopeLevel -= 1; } );
	}

	CompoundStmt *EliminateTypedef::postmutate( CompoundStmt * compoundStmt ) {
		// remove and delete decl stmts
		filter( compoundStmt->kids, [](Statement * stmt) {
			if ( DeclStmt *declStmt = dynamic_cast< DeclStmt * >( stmt ) ) {
				if ( dynamic_cast< TypedefDecl * >( declStmt->get_decl() ) ) {
					return true;
				} // if
			} // if
			return false;
		}, true);
		return compoundStmt;
	}

	// there may be typedefs nested within aggregates. in order for everything to work properly, these should be removed
	// as well
	template<typename AggDecl>
	AggDecl *EliminateTypedef::handleAggregate( AggDecl * aggDecl ) {
		filter( aggDecl->members, isTypedef, true );
		return aggDecl;
	}

	template<typename AggDecl>
	void EliminateTypedef::addImplicitTypedef( AggDecl * aggDecl ) {
		if ( typedefNames.count( aggDecl->get_name() ) == 0 ) {
			Type *type = nullptr;
			if ( StructDecl * newDeclStructDecl = dynamic_cast< StructDecl * >( aggDecl ) ) {
				type = new StructInstType( Type::Qualifiers(), newDeclStructDecl->get_name() );
			} else if ( UnionDecl * newDeclUnionDecl = dynamic_cast< UnionDecl * >( aggDecl ) ) {
				type = new UnionInstType( Type::Qualifiers(), newDeclUnionDecl->get_name() );
			} else if ( EnumDecl * newDeclEnumDecl = dynamic_cast< EnumDecl * >( aggDecl )  ) {
				type = new EnumInstType( Type::Qualifiers(), newDeclEnumDecl->get_name() );
			} // if
			TypedefDeclPtr tyDecl( new TypedefDecl( aggDecl->get_name(), aggDecl->location, Type::StorageClasses(), type, aggDecl->get_linkage() ) );
			typedefNames[ aggDecl->get_name() ] = std::make_pair( std::move( tyDecl ), scopeLevel );
		} // if
	}

	void EliminateTypedef::premutate( StructDecl * structDecl ) {
		addImplicitTypedef( structDecl );
	}


	Declaration *EliminateTypedef::postmutate( StructDecl * structDecl ) {
		return handleAggregate( structDecl );
	}

	void EliminateTypedef::premutate( UnionDecl * unionDecl ) {
		addImplicitTypedef( unionDecl );
	}

	Declaration *EliminateTypedef::postmutate( UnionDecl * unionDecl ) {
		return handleAggregate( unionDecl );
	}

	void EliminateTypedef::premutate( EnumDecl * enumDecl ) {
		addImplicitTypedef( enumDecl );
	}

	Declaration *EliminateTypedef::postmutate( EnumDecl * enumDecl ) {
		return handleAggregate( enumDecl );
	}

	Declaration *EliminateTypedef::postmutate( TraitDecl * traitDecl ) {
		return handleAggregate( traitDecl );
	}

	void EliminateTypedef::premutate( FunctionType * ) {
		GuardValue( inFunctionType );
		inFunctionType = true;
	}

	void VerifyCtorDtorAssign::verify( std::list< Declaration * > & translationUnit ) {
		PassVisitor<VerifyCtorDtorAssign> verifier;
		acceptAll( translationUnit, verifier );
	}

	void VerifyCtorDtorAssign::previsit( FunctionDecl * funcDecl ) {
		FunctionType * funcType = funcDecl->get_functionType();
		std::list< DeclarationWithType * > &returnVals = funcType->get_returnVals();
		std::list< DeclarationWithType * > &params = funcType->get_parameters();

		if ( CodeGen::isCtorDtorAssign( funcDecl->get_name() ) ) { // TODO: also check /=, etc.
			if ( params.size() == 0 ) {
				throw SemanticError( funcDecl, "Constructors, destructors, and assignment functions require at least one parameter " );
			}
			ReferenceType * refType = dynamic_cast< ReferenceType * >( params.front()->get_type() );
			if ( ! refType ) {
				throw SemanticError( funcDecl, "First parameter of a constructor, destructor, or assignment function must be a reference " );
			}
			if ( CodeGen::isCtorDtor( funcDecl->get_name() ) && returnVals.size() != 0 ) {
				throw SemanticError( funcDecl, "Constructors and destructors cannot have explicit return values " );
			}
		}
	}

	template< typename Aggr >
	void validateGeneric( Aggr * inst ) {
		std::list< TypeDecl * > * params = inst->get_baseParameters();
		if ( params ) {
			std::list< Expression * > & args = inst->get_parameters();

			// insert defaults arguments when a type argument is missing (currently only supports missing arguments at the end of the list).
			// A substitution is used to ensure that defaults are replaced correctly, e.g.,
			//   forall(otype T, otype alloc = heap_allocator(T)) struct vector;
			//   vector(int) v;
			// after insertion of default values becomes
			//   vector(int, heap_allocator(T))
			// and the substitution is built with T=int so that after substitution, the result is
			//   vector(int, heap_allocator(int))
			TypeSubstitution sub;
			auto paramIter = params->begin();
			for ( size_t i = 0; paramIter != params->end(); ++paramIter, ++i ) {
				if ( i < args.size() ) {
					TypeExpr * expr = strict_dynamic_cast< TypeExpr * >( *std::next( args.begin(), i ) );
					sub.add( (*paramIter)->get_name(), expr->get_type()->clone() );
				} else if ( i == args.size() ) {
					Type * defaultType = (*paramIter)->get_init();
					if ( defaultType ) {
						args.push_back( new TypeExpr( defaultType->clone() ) );
						sub.add( (*paramIter)->get_name(), defaultType->clone() );
					}
				}
			}

			sub.apply( inst );
			if ( args.size() < params->size() ) throw SemanticError( inst, "Too few type arguments in generic type " );
			if ( args.size() > params->size() ) throw SemanticError( inst, "Too many type arguments in generic type " );
		}
	}

	void ValidateGenericParameters::previsit( StructInstType * inst ) {
		validateGeneric( inst );
	}

	void ValidateGenericParameters::previsit( UnionInstType * inst ) {
		validateGeneric( inst );
	}

	void CompoundLiteral::premutate( ObjectDecl *objectDecl ) {
		storageClasses = objectDecl->get_storageClasses();
	}

	Expression *CompoundLiteral::postmutate( CompoundLiteralExpr *compLitExpr ) {
		// transform [storage_class] ... (struct S){ 3, ... };
		// into [storage_class] struct S temp =  { 3, ... };
		static UniqueName indexName( "_compLit" );

		ObjectDecl *tempvar = new ObjectDecl( indexName.newName(), storageClasses, LinkageSpec::C, nullptr, compLitExpr->get_result(), compLitExpr->get_initializer() );
		compLitExpr->set_result( nullptr );
		compLitExpr->set_initializer( nullptr );
		delete compLitExpr;
		declsToAddBefore.push_back( tempvar );					// add modified temporary to current block
		return new VariableExpr( tempvar );
	}

	void ReturnTypeFixer::fix( std::list< Declaration * > &translationUnit ) {
		PassVisitor<ReturnTypeFixer> fixer;
		acceptAll( translationUnit, fixer );
	}

	void ReturnTypeFixer::postvisit( FunctionDecl * functionDecl ) {
		FunctionType * ftype = functionDecl->get_functionType();
		std::list< DeclarationWithType * > & retVals = ftype->get_returnVals();
		assertf( retVals.size() == 0 || retVals.size() == 1, "Function %s has too many return values: %zu", functionDecl->get_name().c_str(), retVals.size() );
		if ( retVals.size() == 1 ) {
			// ensure all function return values have a name - use the name of the function to disambiguate (this also provides a nice bit of help for debugging).
			// ensure other return values have a name.
			DeclarationWithType * ret = retVals.front();
			if ( ret->get_name() == "" ) {
				ret->set_name( toString( "_retval_", CodeGen::genName( functionDecl ) ) );
			}
			ret->get_attributes().push_back( new Attribute( "unused" ) );
		}
	}

	void ReturnTypeFixer::postvisit( FunctionType * ftype ) {
		// xxx - need to handle named return values - this information needs to be saved somehow
		// so that resolution has access to the names.
		// Note that this pass needs to happen early so that other passes which look for tuple types
		// find them in all of the right places, including function return types.
		std::list< DeclarationWithType * > & retVals = ftype->get_returnVals();
		if ( retVals.size() > 1 ) {
			// generate a single return parameter which is the tuple of all of the return values
			TupleType * tupleType = strict_dynamic_cast< TupleType * >( ResolvExpr::extractResultType( ftype ) );
			// ensure return value is not destructed by explicitly creating an empty ListInit node wherein maybeConstruct is false.
			ObjectDecl * newRet = new ObjectDecl( "", Type::StorageClasses(), LinkageSpec::Cforall, 0, tupleType, new ListInit( std::list<Initializer*>(), noDesignators, false ) );
			deleteAll( retVals );
			retVals.clear();
			retVals.push_back( newRet );
		}
	}

	void ArrayLength::computeLength( std::list< Declaration * > & translationUnit ) {
		PassVisitor<ArrayLength> len;
		acceptAll( translationUnit, len );
	}

	void ArrayLength::previsit( ObjectDecl * objDecl ) {
		if ( ArrayType * at = dynamic_cast< ArrayType * >( objDecl->get_type() ) ) {
			if ( at->get_dimension() ) return;
			if ( ListInit * init = dynamic_cast< ListInit * >( objDecl->get_init() ) ) {
				at->set_dimension( new ConstantExpr( Constant::from_ulong( init->get_initializers().size() ) ) );
			}
		}
	}

	struct LabelFinder {
		std::set< Label > & labels;
		LabelFinder( std::set< Label > & labels ) : labels( labels ) {}
		void previsit( Statement * stmt ) {
			for ( Label & l : stmt->labels ) {
				labels.insert( l );
			}
		}
	};

	void LabelAddressFixer::premutate( FunctionDecl * funcDecl ) {
		GuardValue( labels );
		PassVisitor<LabelFinder> finder( labels );
		funcDecl->accept( finder );
	}

	Expression * LabelAddressFixer::postmutate( AddressExpr * addrExpr ) {
		// convert &&label into label address
		if ( AddressExpr * inner = dynamic_cast< AddressExpr * >( addrExpr->arg ) ) {
			if ( NameExpr * nameExpr = dynamic_cast< NameExpr * >( inner->arg ) ) {
				if ( labels.count( nameExpr->name ) ) {
					Label name = nameExpr->name;
					delete addrExpr;
					return new LabelAddressExpr( name );
				}
			}
		}
		return addrExpr;
	}

	void FindSpecialDeclarations::previsit( FunctionDecl * funcDecl ) {
		if ( ! dereferenceOperator ) {
			if ( funcDecl->get_name() == "*?" && funcDecl->get_linkage() == LinkageSpec::Intrinsic ) {
				FunctionType * ftype = funcDecl->get_functionType();
				if ( ftype->get_parameters().size() == 1 && ftype->get_parameters().front()->get_type()->get_qualifiers() == Type::Qualifiers() ) {
					dereferenceOperator = funcDecl;
				}
			}
		}
	}
} // namespace SymTab

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
