//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// signal.c --
//
// Author           : Thierry Delisle
// Created On       : Mon Jun 5 14:20:42 2017
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Jan 23 17:59:30 2018
// Update Count     : 7
//

#include "preemption.h"

#define ftype `ftype`
extern "C" {
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
}
#undef ftype

#include "bits/signal.h"

//TODO move to defaults
#define __CFA_DEFAULT_PREEMPTION__ 10000

//TODO move to defaults
__attribute__((weak)) unsigned int default_preemption() {
	return __CFA_DEFAULT_PREEMPTION__;
}

// FwdDeclarations : timeout handlers
static void preempt( processor   * this );
static void timeout( thread_desc * this );

// FwdDeclarations : Signal handlers
void sigHandler_ctxSwitch( __CFA_SIGPARMS__ );
void sigHandler_segv     ( __CFA_SIGPARMS__ );
void sigHandler_abort    ( __CFA_SIGPARMS__ );

// FwdDeclarations : alarm thread main
void * alarm_loop( __attribute__((unused)) void * args );

// Machine specific register name
#if   defined(__x86_64__)
#define CFA_REG_IP REG_RIP
#elif defined(__i386__)
#define CFA_REG_IP REG_EIP
#elif defined(__ARM_ARCH__)
#define CFA_REG_IP REG_R15
#endif

KERNEL_STORAGE(event_kernel_t, event_kernel);         // private storage for event kernel
event_kernel_t * event_kernel;                        // kernel public handle to even kernel
static pthread_t alarm_thread;                        // pthread handle to alarm thread

void ?{}(event_kernel_t & this) with( this ) {
	alarms{};
	lock{};
}

//=============================================================================================
// Kernel Preemption logic
//=============================================================================================

// Get next expired node
static inline alarm_node_t * get_expired( alarm_list_t * alarms, __cfa_time_t currtime ) {
	if( !alarms->head ) return NULL;                          // If no alarms return null
	if( alarms->head->alarm >= currtime ) return NULL;        // If alarms head not expired return null
	return pop(alarms);                                       // Otherwise just pop head
}

// Tick one frame of the Discrete Event Simulation for alarms
void tick_preemption() {
	alarm_node_t * node = NULL;                     // Used in the while loop but cannot be declared in the while condition
	alarm_list_t * alarms = &event_kernel->alarms;  // Local copy for ease of reading
	__cfa_time_t currtime = __kernel_get_time();    // Check current time once so we everything "happens at once"

	//Loop throught every thing expired
	while( node = get_expired( alarms, currtime ) ) {

		// Check if this is a kernel
		if( node->kernel_alarm ) {
			preempt( node->proc );
		}
		else {
			timeout( node->thrd );
		}

		// Check if this is a periodic alarm
		__cfa_time_t period = node->period;
		if( period > 0 ) {
			node->alarm = currtime + period;    // Alarm is periodic, add currtime to it (used cached current time)
			insert( alarms, node );             // Reinsert the node for the next time it triggers
		}
		else {
			node->set = false;                  // Node is one-shot, just mark it as not pending
		}
	}

	// If there are still alarms pending, reset the timer
	if( alarms->head ) { __kernel_set_timer( alarms->head->alarm - currtime ); }
}

// Update the preemption of a processor and notify interested parties
void update_preemption( processor * this, __cfa_time_t duration ) {
	alarm_node_t * alarm = this->preemption_alarm;

	// Alarms need to be enabled
	if ( duration > 0 && !alarm->set ) {
		alarm->alarm = __kernel_get_time() + duration;
		alarm->period = duration;
		register_self( alarm );
	}
	// Zero duraction but alarm is set
	else if ( duration == 0 && alarm->set ) {
		unregister_self( alarm );
		alarm->alarm = 0;
		alarm->period = 0;
	}
	// If alarm is different from previous, change it
	else if ( duration > 0 && alarm->period != duration ) {
		unregister_self( alarm );
		alarm->alarm = __kernel_get_time() + duration;
		alarm->period = duration;
		register_self( alarm );
	}
}

//=============================================================================================
// Kernel Signal Tools
//=============================================================================================

__cfaabi_dbg_debug_do( static thread_local void * last_interrupt = 0; )

extern "C" {
	// Disable interrupts by incrementing the counter
	void disable_interrupts() {
		__attribute__((unused)) unsigned short new_val = __atomic_add_fetch_2( &disable_preempt_count, 1, __ATOMIC_SEQ_CST );
		verify( new_val < 65_000u );              // If this triggers someone is disabling interrupts without enabling them
	}

	// Enable interrupts by decrementing the counter
	// If counter reaches 0, execute any pending CtxSwitch
	void enable_interrupts( __cfaabi_dbg_ctx_param ) {
		processor   * proc = this_processor;      // Cache the processor now since interrupts can start happening after the atomic add
		thread_desc * thrd = this_thread;         // Cache the thread now since interrupts can start happening after the atomic add

		unsigned short prev = __atomic_fetch_add_2( &disable_preempt_count, -1, __ATOMIC_SEQ_CST );
		verify( prev != 0u );                     // If this triggers someone is enabled already enabled interruptsverify( prev != 0u );

		// Check if we need to prempt the thread because an interrupt was missed
		if( prev == 1 && proc->pending_preemption ) {
			proc->pending_preemption = false;
			BlockInternal( thrd );
		}

		// For debugging purposes : keep track of the last person to enable the interrupts
		__cfaabi_dbg_debug_do( proc->last_enable = caller; )
	}

	// Disable interrupts by incrementint the counter
	// Don't execute any pending CtxSwitch even if counter reaches 0
	void enable_interrupts_noPoll() {
		__attribute__((unused)) unsigned short prev = __atomic_fetch_add_2( &disable_preempt_count, -1, __ATOMIC_SEQ_CST );
		verifyf( prev != 0u, "Incremented from %u\n", prev );                     // If this triggers someone is enabled already enabled interrupts
	}
}

// sigprocmask wrapper : unblock a single signal
static inline void signal_unblock( int sig ) {
	sigset_t mask;
	sigemptyset( &mask );
	sigaddset( &mask, sig );

	if ( pthread_sigmask( SIG_UNBLOCK, &mask, NULL ) == -1 ) {
	    abortf( "internal error, pthread_sigmask" );
	}
}

// sigprocmask wrapper : block a single signal
static inline void signal_block( int sig ) {
	sigset_t mask;
	sigemptyset( &mask );
	sigaddset( &mask, sig );

	if ( pthread_sigmask( SIG_BLOCK, &mask, NULL ) == -1 ) {
	    abortf( "internal error, pthread_sigmask" );
	}
}

// kill wrapper : signal a processor
static void preempt( processor * this ) {
	pthread_kill( this->kernel_thread, SIGUSR1 );
}

// reserved for future use
static void timeout( thread_desc * this ) {
	//TODO : implement waking threads
}


// Check if a CtxSwitch signal handler shoud defer
// If true  : preemption is safe
// If false : preemption is unsafe and marked as pending
static inline bool preemption_ready() {
	bool ready = disable_preempt_count == 0 && !preemption_in_progress; // Check if preemption is safe
	this_processor->pending_preemption = !ready;                        // Adjust the pending flag accordingly
	return ready;
}

//=============================================================================================
// Kernel Signal Startup/Shutdown logic
//=============================================================================================

// Startup routine to activate preemption
// Called from kernel_startup
void kernel_start_preemption() {
	__cfaabi_dbg_print_safe("Kernel : Starting preemption\n");

	// Start with preemption disabled until ready
	disable_preempt_count = 1;

	// Initialize the event kernel
	event_kernel = (event_kernel_t *)&storage_event_kernel;
	(*event_kernel){};

	// Setup proper signal handlers
	__kernel_sigaction( SIGUSR1, sigHandler_ctxSwitch, SA_SIGINFO | SA_RESTART );         // CtxSwitch handler

	signal_block( SIGALRM );

	pthread_create( &alarm_thread, NULL, alarm_loop, NULL );
}

// Shutdown routine to deactivate preemption
// Called from kernel_shutdown
void kernel_stop_preemption() {
	__cfaabi_dbg_print_safe("Kernel : Preemption stopping\n");

	// Block all signals since we are already shutting down
	sigset_t mask;
	sigfillset( &mask );
	sigprocmask( SIG_BLOCK, &mask, NULL );

	// Notify the alarm thread of the shutdown
	sigval val = { 1 };
	pthread_sigqueue( alarm_thread, SIGALRM, val );

	// Wait for the preemption thread to finish
	pthread_join( alarm_thread, NULL );

	// Preemption is now fully stopped

	__cfaabi_dbg_print_safe("Kernel : Preemption stopped\n");
}

// Raii ctor/dtor for the preemption_scope
// Used by thread to control when they want to receive preemption signals
void ?{}( preemption_scope & this, processor * proc ) {
	(this.alarm){ proc, zero_time, zero_time };
	this.proc = proc;
	this.proc->preemption_alarm = &this.alarm;

	update_preemption( this.proc, from_us(this.proc->cltr->preemption) );
}

void ^?{}( preemption_scope & this ) {
	disable_interrupts();

	update_preemption( this.proc, zero_time );
}

//=============================================================================================
// Kernel Signal Handlers
//=============================================================================================

// Context switch signal handler
// Receives SIGUSR1 signal and causes the current thread to yield
void sigHandler_ctxSwitch( __CFA_SIGPARMS__ ) {
#if defined( __ARM_ARCH )
	__cfaabi_dbg_debug_do( last_interrupt = (void *)(cxt->uc_mcontext.arm_pc); )
#else
	__cfaabi_dbg_debug_do( last_interrupt = (void *)(cxt->uc_mcontext.gregs[CFA_REG_IP]); )
#endif

		// Check if it is safe to preempt here
	if( !preemption_ready() ) { return; }

	__cfaabi_dbg_print_buffer_decl(" KERNEL: preempting core %p (%p).\n", this_processor, this_thread);

	preemption_in_progress = true;                      // Sync flag : prevent recursive calls to the signal handler
	signal_unblock( SIGUSR1 );                          // We are about to CtxSwitch out of the signal handler, let other handlers in
	preemption_in_progress = false;                     // Clear the in progress flag

	// Preemption can occur here

	BlockInternal( (thread_desc*)this_thread );         // Do the actual CtxSwitch
}

// Main of the alarm thread
// Waits on SIGALRM and send SIGUSR1 to whom ever needs it
void * alarm_loop( __attribute__((unused)) void * args ) {
	// Block sigalrms to control when they arrive
	sigset_t mask;
	sigemptyset( &mask );
	sigaddset( &mask, SIGALRM );

	if ( pthread_sigmask( SIG_BLOCK, &mask, NULL ) == -1 ) {
	    abortf( "internal error, pthread_sigmask" );
	}

	// Main loop
	while( true ) {
		// Wait for a sigalrm
		siginfo_t info;
		int sig = sigwaitinfo( &mask, &info );

		if( sig < 0 ) {
			//Error!
			int err = errno;
			switch( err ) {
				case EAGAIN :
				case EINTR :
					continue;
       			case EINVAL :
				 	abortf("Timeout was invalid.");
				default:
				 	abortf("Unhandled error %d", err);
			}
		}

		// If another signal arrived something went wrong
		assertf(sig == SIGALRM, "Kernel Internal Error, sigwait: Unexpected signal %d (%d : %d)\n", sig, info.si_code, info.si_value.sival_int);

		// __cfaabi_dbg_print_safe("Kernel : Caught alarm from %d with %d\n", info.si_code, info.si_value.sival_int );
		// Switch on the code (a.k.a. the sender) to
		switch( info.si_code )
		{
		// Timers can apparently be marked as sent for the kernel
		// In either case, tick preemption
		case SI_TIMER:
		case SI_KERNEL:
			// __cfaabi_dbg_print_safe("Kernel : Preemption thread tick\n");
			lock( event_kernel->lock __cfaabi_dbg_ctx2 );
			tick_preemption();
			unlock( event_kernel->lock );
			break;
		// Signal was not sent by the kernel but by an other thread
		case SI_QUEUE:
			// For now, other thread only signal the alarm thread to shut it down
			// If this needs to change use info.si_value and handle the case here
			goto EXIT;
		}
	}

EXIT:
	__cfaabi_dbg_print_safe("Kernel : Preemption thread stopping\n");
	return NULL;
}

// Local Variables: //
// mode: c //
// tab-width: 4 //
// End: //
