//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Initializer.h --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Wed Jan 13 15:29:53 2016
// Update Count     : 17
//

#ifndef INITIALIZER_H
#define INITIALIZER_H

#include "SynTree.h"
#include "Visitor.h"
#include "Mutator.h"

#include <cassert>

// Initializer: base class for object initializers (provide default values)
class Initializer {
  public:
	//	Initializer( std::string _name = std::string(""), int _pos = 0 );
	Initializer( bool maybeConstructed );
	virtual ~Initializer();

	static std::string designator_name( Expression *designator );

	//	void set_name( std::string newValue ) { name = newValue; }
	//	std::string get_name() const { return name; }

	//	void set_pos( int newValue ) { pos = newValue; }
	//	int get_pos() const { return pos; }
	virtual void set_designators( std::list<Expression *> & ) { assert(false); }
	virtual std::list<Expression *> &get_designators() {
		assert(false);
		std::list<Expression *> *ret = 0; return *ret;	// never reached
	}

	bool get_maybeConstructed() { return maybeConstructed; }

	virtual Initializer *clone() const = 0;
	virtual void accept( Visitor &v ) = 0;
	virtual Initializer *acceptMutator( Mutator &m ) = 0;
	virtual void print( std::ostream &os, int indent = 0 );
  private:
	//	std::string name;
	//	int pos;
	bool maybeConstructed;
};

// SingleInit represents an initializer for a common object (e.g., int x = 4)
class SingleInit : public Initializer {
  public:
	SingleInit( Expression *value, std::list< Expression *> &designators, bool maybeConstructed );
	SingleInit( const SingleInit &other );
	virtual ~SingleInit();

	Expression *get_value() { return value; }
	void set_value( Expression *newValue ) { value = newValue; }

	void set_designators( std::list<Expression *> &newValue ) { designators = newValue; }
	std::list<Expression *> &get_designators() { return designators; }

	virtual SingleInit *clone() const;
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Initializer *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 );
  private:
	//Constant *value;
	Expression *value;	// has to be a compile-time constant
	std::list< Expression * > designators;
};

// ListInit represents an initializer that is composed recursively of a list of initializers; this is used to initialize
// an array or aggregate
class ListInit : public Initializer {
  public:
	ListInit( std::list<Initializer*> &,
			  std::list<Expression *> &designators, bool maybeConstructed );
	virtual ~ListInit();

	void set_designators( std::list<Expression *> &newValue ) { designators = newValue; }
	std::list<Expression *> &get_designators() { return designators; }
	void set_initializers( std::list<Initializer*> &newValue ) { initializers = newValue; }
	std::list<Initializer*> &get_initializers() { return initializers; }

	std::list<Initializer*>::iterator begin_initializers() { return initializers.begin(); }
	std::list<Initializer*>::iterator end_initializers() { return initializers.end(); }

	virtual ListInit *clone() const;
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Initializer *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 );
  private:
	std::list<Initializer*> initializers;  // order *is* important
	std::list<Expression *> designators;
};

// ConstructorInit represents an initializer that is either a constructor expression or
// a C-style initializer.
class ConstructorInit : public Initializer {
  public:
	ConstructorInit( Expression * ctor, Initializer * init );
	virtual ~ConstructorInit();

	void set_ctor( Expression * newValue ) { ctor = newValue; }
	Expression * get_ctor() const { return ctor; }
	void set_init( Initializer * newValue ) { init = newValue; }
	Initializer * get_init() const { return init; }

	virtual ConstructorInit *clone() const;
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Initializer *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 );

  private:
	Expression * ctor;
	// C-style initializer made up of SingleInit and ListInit nodes to use as a fallback
	// if an appropriate constructor definition is not found by the resolver
	Initializer * init;
};

#endif // INITIALIZER_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
