//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// RemoveInit.h --
//
// Author           : Rob Schluntz
// Created On       : Fri May 13 11:26:36 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Jul 22 09:30:33 2017
// Update Count     : 4
//

#pragma once

#include <list>               // for list
#include <memory>             // for shared_ptr
#include <string>             // for string, allocator

#include "SynTree/SynTree.h"  // for Visitor Nodes

// helper functions for initialization
namespace InitTweak {
	bool isConstructor( const std::string & );
	bool isDestructor( const std::string & );
	bool isAssignment( const std::string & );
	bool isCtorDtor( const std::string & );
	bool isCtorDtorAssign( const std::string & );

	FunctionDecl * isAssignment( Declaration * decl );
	FunctionDecl * isDestructor( Declaration * decl );
	FunctionDecl * isDefaultConstructor( Declaration * decl );
	FunctionDecl * isCopyConstructor( Declaration * decl );
	FunctionDecl * isCopyFunction( Declaration * decl, const std::string & fname );

	/// transform Initializer into an argument list that can be passed to a call expression
	std::list< Expression * > makeInitList( Initializer * init );

	/// True if the resolver should try to construct objDecl
	bool tryConstruct( ObjectDecl * objDecl );

	/// True if the Initializer contains designations
	bool isDesignated( Initializer * init );

	/// True if the ObjectDecl's Initializer nesting level is not deeper than the depth of its
	/// type, where the depth of its type is the number of nested ArrayTypes + 1
	bool checkInitDepth( ObjectDecl * objDecl );

	/// returns the declaration of the function called by the expr (must be ApplicationExpr or UntypedExpr)
	DeclarationWithType * getFunction( Expression * expr );

	/// Non-Null if expr is a call expression whose target function is intrinsic
	ApplicationExpr * isIntrinsicCallExpr( Expression * expr );

	/// True if stmt is a call statement where the function called is intrinsic and takes one parameter.
	/// Intended to be used for default ctor/dtor calls, but might have use elsewhere.
	/// Currently has assertions that make it less than fully general.
	bool isIntrinsicSingleArgCallStmt( Statement * stmt );

	/// True if stmt is a call statement where the function called is intrinsic.
	bool isIntrinsicCallStmt( Statement * stmt );

	/// get all Ctor/Dtor call expressions from a Statement
	void collectCtorDtorCalls( Statement * stmt, std::list< Expression * > & matches );

	/// get the Ctor/Dtor call expression from a Statement that looks like a generated ctor/dtor call
	Expression * getCtorDtorCall( Statement * stmt );

	/// returns the name of the function being called
	std::string getFunctionName( Expression * expr );

	/// returns the argument to a call expression in position N indexed from 0
	Expression *& getCallArg( Expression * callExpr, unsigned int pos );

	/// returns the base type of a PointerType or ArrayType, else returns NULL
	Type * getPointerBase( Type * );

	/// returns the argument if it is a PointerType or ArrayType, else returns NULL
	Type * isPointerType( Type * );

	/// returns true if expr is trivially a compile-time constant
	bool isConstExpr( Expression * expr );
	bool isConstExpr( Initializer * init );

	class InitExpander {
	public:
		// expand by stepping through init to get each list of arguments
		InitExpander( Initializer * init );

		// always expand to expr
		InitExpander( Expression * expr );

		// iterator-like interface
		std::list< Expression * > operator*();
		InitExpander & operator++();

		// builds statement which has the same semantics as a C-style list initializer
		// (for array initializers) using callExpr as the base expression to perform initialization
		Statement * buildListInit( UntypedExpr * callExpr );
		void addArrayIndex( Expression * index, Expression * dimension );
		void clearArrayIndices();

		class ExpanderImpl;

		typedef std::list< Expression * > IndexList;
	private:
		std::shared_ptr< ExpanderImpl > expander;
		std::list< Expression * > cur;

		// invariant: list of size 2N (elements come in pairs [index, dimension])
		IndexList indices;
	};
} // namespace

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
