//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CopyParams.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue May 19 07:33:31 2015
// Update Count     : 1
//

#include <cassert>                 // for assert
#include <list>                    // for list, _List_iterator, _List_const_...
#include <map>                     // for map, _Rb_tree_const_iterator, map<...
#include <set>                     // for set, set<>::const_iterator
#include <string>                  // for string, operator==
#include <utility>                 // for pair

#include "Common/SemanticError.h"  // for SemanticError
#include "Common/UniqueName.h"     // for UniqueName
#include "SynTree/Declaration.h"   // for DeclarationWithType, TypeDecl, Fun...
#include "SynTree/Expression.h"    // for VariableExpr, ApplicationExpr, Add...
#include "SynTree/Label.h"         // for Label, noLabels
#include "SynTree/Statement.h"     // for CompoundStmt, DeclStmt, ExprStmt
#include "SynTree/SynTree.h"       // for UniqueId
#include "SynTree/Type.h"          // for FunctionType, TypeInstType, Type
#include "SynTree/Visitor.h"       // for acceptAll, Visitor

namespace GenPoly {
	class CopyParams : public Visitor {
	  public:
		CopyParams();

		virtual void visit( FunctionDecl *funcDecl );
		virtual void visit( AddressExpr *addrExpr );

	  private:
		std::set< UniqueId > modVars;
		UniqueName namer;
	};

	/// creates local copies of polymorphic function parameters
	void copyParams( std::list< Declaration* > &translationUnit ) {
		CopyParams copier;
		acceptAll( translationUnit, copier );
	}

	CopyParams::CopyParams() : namer( "_cp" ) {}

	void CopyParams::visit( FunctionDecl *funcDecl ) {
		if ( funcDecl->get_statements() ) {
			funcDecl->get_statements()->accept( *this );

			if ( ! modVars.empty() ) {
				std::map< std::string, DeclarationWithType* > assignOps;
				// xxx - this needs to use constructors, not assignment operators
				// assume the assignment operator is the first assert param after any "type" parameter
				for ( Type::ForallList::const_iterator tyVar = funcDecl->get_functionType()->get_forall().begin(); tyVar != funcDecl->get_functionType()->get_forall().end(); ++tyVar ) {
					if ( (*tyVar)->get_kind() == TypeDecl::Any ) {
						assert( !(*tyVar)->get_assertions().empty() );
						assert( (*tyVar)->get_assertions().front()->get_name() == "?=?" );
						assignOps[ (*tyVar)->get_name() ] = (*tyVar)->get_assertions().front();
					} // if
				} // for
				for ( std::list< DeclarationWithType* >::iterator param = funcDecl->get_functionType()->get_parameters().begin(); param != funcDecl->get_functionType()->get_parameters().end(); ++param ) {
					std::set< UniqueId >::const_iterator var = modVars.find( (*param)->get_uniqueId() );
					if ( var != modVars.end() ) {
						TypeInstType *typeInst = dynamic_cast< TypeInstType* >( (*param)->get_type() );
						assert( typeInst );
						std::map< std::string, DeclarationWithType* >::const_iterator assignOp = assignOps.find( typeInst->get_name() );
						if ( assignOp != assignOps.end() ) {
							DeclarationWithType *oldParam = *param;
							*param = (*param)->clone();
							(*param)->set_mangleName( namer.newName( (*param)->get_mangleName() ) );
							ApplicationExpr *assign = new ApplicationExpr( new VariableExpr( assignOp->second ) );
							assign->get_args().push_back( new VariableExpr( oldParam ) );
							assign->get_args().push_back( new VariableExpr( *param ) );
							funcDecl->get_statements()->get_kids().push_front( new ExprStmt( noLabels, assign ) );
							funcDecl->get_statements()->get_kids().push_front( new DeclStmt( noLabels, oldParam ) );
						} // if
						modVars.erase( var );
					} // if
				} // for
			} // if
		} // if
	}

	// this test is insufficient because it is possible for values to be modified by being passed to other polymorphic
	// routines (e.g., assignment operators) without having their addresses explicitly taken. Some thought is needed to
	// make sure that all of the correct cases are identified where copies are necessary.
	//
	// As a temporary measure, for correctness at the expense of performance, ignore the modVars list entirely and copy
	// every parameter of TypeInstType* when visiting the FunctionDecl.
	void CopyParams::visit( AddressExpr *addrExpr ) {
		if ( VariableExpr *varExpr = dynamic_cast< VariableExpr* >( addrExpr->get_arg() ) ) {
			if ( dynamic_cast< TypeInstType* >( varExpr->get_var()->get_type() ) ) {
				modVars.insert( varExpr->get_var()->get_uniqueId() );
			} // if
		} // if
	}
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
