//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// DeclStats.cc --
//
// Author           : Aaron Moss
// Created On       : Wed Jan 31 16:40:00 2016
// Last Modified By : Aaron Moss
// Last Modified On : Wed Jan 31 16:40:00 2016
// Update Count     : 1
//

#include "DeclStats.h"

#include <iostream>
#include <map>
#include <string>
#include <unordered_map>
#include <unordered_set>

#include "Common/VectorMap.h"
#include "GenPoly/GenPoly.h"
#include "Parser/LinkageSpec.h"
#include "SynTree/Declaration.h"
#include "SynTree/Visitor.h"

namespace CodeTools {
	
	class DeclStats : public Visitor {
		struct Stats {
			unsigned n_decls;     ///< Total number of declarations
			/// Count of declarations with each number of assertion parameters
			VectorMap<unsigned> n_type_params;
			/// Count of declarations with each name
			std::unordered_map<std::string, unsigned> by_name;

			/// Count of declarations with each number of parameters
			VectorMap<unsigned> n_params;
			/// Count of declarations with each number of return types
			VectorMap<unsigned> n_returns;
			/// Count of declarations with each number of polymorphic parameters
			VectorMap<unsigned> n_poly_params;
			/// Count of declarations with each number of polymorphic return types
			VectorMap<unsigned> n_poly_returns;
			/// Count of declarations with each percentage of polymorphic parameters
			std::map<unsigned, unsigned> p_poly_params;
			/// Count of declarations with each percentage of polymorphic returns
			std::map<unsigned, unsigned> p_poly_returns;

			/// Count of declarations with each number of assertions
			std::map<unsigned, unsigned> n_assertions;
			/// Count of assertions with each number of parameters
			VectorMap<unsigned> n_assn_params;
			/// Count of assertions with each number of return types
			VectorMap<unsigned> n_assn_returns;
			/// Count of assertions with each number of polymorphic parameters
			VectorMap<unsigned> n_assn_poly_params;
			/// Count of assertions with each number of polymorphic return types
			VectorMap<unsigned> n_assn_poly_returns;
			/// Count of assertions with each percentage of polymorphic parameters
			std::map<unsigned, unsigned> p_assn_poly_params;
			/// Count of assertions with each percentage of polymorphic returns
			std::map<unsigned, unsigned> p_assn_poly_returns;

			Stats()
				: n_decls(0), n_type_params(), by_name(),
				  n_params(), n_returns(), n_poly_params(), n_poly_returns(), p_poly_params(), p_poly_returns(),
				  n_assertions(), n_assn_params(), n_assn_returns(), n_assn_poly_params(), n_assn_poly_returns(), p_assn_poly_params(), p_assn_poly_returns() {}

		private:
			template<typename T>
			static void sum(T& a, const T& b) { a += b; }

			static void sum(VectorMap<unsigned>& a, const VectorMap<unsigned>& b) {
				a.reserve( b.size() );
				for ( unsigned i = 0; i < b.size(); ++i ) {
					a[i] += b[i];
				}
			}

			template<typename K>
			static void sum(std::map<K, unsigned>& a, const std::map<K, unsigned>& b) {
				for ( const auto& entry : b ) {
					a[ entry.first ] += entry.second;
				}
			}

			template<typename K>
			static void sum(std::unordered_map<K, unsigned>& a, const std::unordered_map<K, unsigned>& b) {
				for ( const auto& entry : b ) {
					a[ entry.first ] += entry.second;
				}
			}

		public:
			Stats& operator+= (const Stats& o) {
				sum( n_decls, o.n_decls );
				sum( n_type_params, o.n_type_params );
				sum( by_name, o.by_name );
				
				sum( n_params, o.n_params );
				sum( n_returns, o.n_returns );
				sum( n_poly_params, o.n_poly_params );
				sum( n_poly_returns, o.n_poly_returns );
				sum( p_poly_params, o.p_poly_params );
				sum( p_poly_returns, o.p_poly_returns );

				sum( n_assertions, o.n_assertions );
				sum( n_assn_params, o.n_assn_params );
				sum( n_assn_returns, o.n_assn_returns );
				sum( n_assn_poly_params, o.n_assn_poly_params );
				sum( n_assn_poly_returns, o.n_assn_poly_returns );
				sum( p_assn_poly_params, o.p_assn_poly_params );
				sum( p_assn_poly_returns, o.p_assn_poly_returns );

				return *this;
			}
		};

		Stats for_linkage[LinkageSpec::NoOfSpecs];   ///< Stores separate stats per linkage
		std::unordered_set<std::string> seen_names;  ///< Stores manglenames already seen to avoid double-counting
		Stats total;

		void analyzeFunc( FunctionType* fnTy, VectorMap<unsigned>& by_params, VectorMap<unsigned>& by_poly_params,
		                  VectorMap<unsigned>& by_returns, VectorMap<unsigned>& by_poly_returns,
		                  std::map<unsigned, unsigned>& p_poly_params, std::map<unsigned, unsigned>& p_poly_returns ) {
			unsigned n_params = 0;
			unsigned n_poly_params = 0;
			for ( auto pdecl : fnTy->get_parameters() ) {
				n_params += pdecl->get_type()->size();
				if ( GenPoly::hasPolyBase( pdecl->get_type() ) ) ++n_poly_params;
			}
			++by_params.at( n_params );
			++by_poly_params.at( n_poly_params );
			if ( n_params > 0 ) ++p_poly_params[ n_poly_params*100/n_params ];

			unsigned n_returns = 0;
			unsigned n_poly_returns = 0;
			for ( auto rdecl : fnTy->get_returnVals() ) {
				n_returns += rdecl->get_type()->size();
				if ( GenPoly::hasPolyBase( rdecl->get_type() ) ) ++n_poly_returns;
			}
			++by_returns.at( n_returns );
			++by_poly_returns.at( n_poly_returns );
			if ( n_returns > 0 ) ++p_poly_returns[ n_poly_returns*100/n_returns ];
		}

	public:
		using Visitor::visit;

		virtual void visit( FunctionDecl *decl ) {
			// skip if already seen declaration for this function
			const std::string& mangleName = decl->get_mangleName().empty() ? decl->get_name() : decl->get_mangleName();
			if ( ! seen_names.insert( mangleName ).second ) return;
			
			Stats& stats = for_linkage[ decl->get_linkage() ];

			++stats.n_decls;
			FunctionType* fnTy = decl->get_functionType();
			const Type::ForallList& forall = fnTy->get_forall();
			++stats.n_type_params.at( forall.size() );
			unsigned n_assertions = 0;
			for ( TypeDecl* fdecl : forall ) {
				n_assertions += fdecl->get_assertions().size();
				for ( DeclarationWithType* assn : fdecl->get_assertions() ) {
					FunctionType *assnTy = 0;
					if ( ObjectDecl *assnObj = dynamic_cast<ObjectDecl*>(assn) ) {
						if ( PointerType *ptrTy = dynamic_cast<PointerType*>(assnObj->get_type()) ) {
							assnTy = dynamic_cast<FunctionType*>(ptrTy->get_base());
						} else assnTy = dynamic_cast<FunctionType*>(assnObj->get_type());
					} else if ( FunctionDecl *assnDecl = dynamic_cast<FunctionDecl*>(assn) ) {
						assnTy = assnDecl->get_functionType();
					}
					if ( assnTy ) analyzeFunc( assnTy, stats.n_assn_params, stats.n_assn_poly_params, stats.n_assn_returns,
					                           stats.n_assn_poly_returns, stats.p_assn_poly_params, stats.p_assn_poly_returns );
				}
			}
			++stats.n_assertions[ n_assertions ];

			++stats.by_name[ decl->get_name() ];

			analyzeFunc( fnTy, stats.n_params, stats.n_poly_params, stats.n_returns,
			             stats.n_poly_returns, stats.p_poly_params, stats.p_poly_returns );
		}

	private:
		template<typename F>
		void printAll( const char* name, F extract ) {
			std::cout << "\"" << name << "\",";
			for ( const auto& stats : for_linkage ) {
				std::cout << "," << extract(stats);
			}
			std::cout << "," << extract(total) << std::endl;
		}

		template<typename F>
		void printAllMap( const char* name, F extract ) {
			for ( const auto& entry : extract(total) ) {
				const auto& key = entry.first;
				std::cout << "\"" << name << "\"," << key;
				for ( const auto& stats : for_linkage ) {
					const auto& map = extract(stats);
					auto it = map.find( key );
					if ( it == map.end() ) std::cout << ",0";
					else std::cout << "," << it->second;
				}
				std::cout  << "," << entry.second << std::endl;
			}
		}

		template<typename F>
		void printAllHisto( const char* name, F extract ) {
			VectorMap<unsigned> histos[LinkageSpec::NoOfSpecs];
			VectorMap<unsigned> thisto;

			for ( const auto& entry : extract(total) ) { ++thisto.at( entry.second ); }

			for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
				// can't be a higher count in one of the sub-histograms than the total
				histos[i].reserve( thisto.size() );

				for ( const auto& entry : extract(for_linkage[i]) ) { ++histos[i][entry.second]; }
			}

			for ( unsigned i = 0; i < thisto.size(); ++i ) {
				std::cout << "\"" << name << "\"," << i;
				for ( const auto& histo : histos ) {
					std::cout << "," << histo[i];
				}
				std::cout << "," << thisto[i] << std::endl;
			}
		}

		template<typename F>
		void printAllSparseHisto( const char* name, F extract ) {
			std::map<unsigned, unsigned> histos[LinkageSpec::NoOfSpecs];
			std::map<unsigned, unsigned> thisto;

			for ( const auto& entry : extract(total) ) { ++thisto[ entry.second ]; }

			for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
				for ( const auto& entry : extract(for_linkage[i]) ) { ++histos[i][entry.second]; }
			}

			for ( const auto& entry : thisto ) {
				const auto& key = entry.first;
				std::cout << "\"" << name << "\"," << key;
				for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
					auto it = histos[i].find( key );
					if ( it == histos[i].end() ) std::cout << ",0";
					else std::cout << "," << it->second;
				}
				std::cout << "," << entry.second << std::endl;
			}
		}
		
	public:
		void print() {
			for ( auto& stats : for_linkage ) {
				total += stats;
			}

			std::cout << ",,\"intrinsic\",\"Cforall\",\"C\",\"autogen\",\"builtin\",\"TOTAL\"" << std::endl;

			printAllMap("n_type_params", [](const Stats& stats) { return stats.n_type_params; });
			printAll("n_decls", [](const Stats& stats) { return stats.n_decls; });
			printAll("unique_names", [](const Stats& stats) { return stats.by_name.size(); });
			printAllSparseHisto("overloads", [](const Stats& stats) { return stats.by_name; });
			
			printAllMap("n_poly_params", [](const Stats& stats) { return stats.n_poly_params; });
			printAllMap("n_params", [](const Stats& stats) { return stats.n_params; });
			printAllMap("%_poly_params", [](const Stats& stats) { return stats.p_poly_params; });
			printAllMap("n_poly_returns", [](const Stats& stats) { return stats.n_poly_returns; });
			printAllMap("n_returns", [](const Stats& stats) { return stats.n_returns; });
			printAllMap("%_poly_returns", [](const Stats& stats) { return stats.p_poly_returns; });

			printAllMap("n_assertions", [](const Stats& stats) { return stats.n_assertions; });
			printAllMap("n_assn_poly_params", [](const Stats& stats) { return stats.n_assn_poly_params; });
			printAllMap("n_assn_params", [](const Stats& stats) { return stats.n_assn_params; });
			printAllMap("%_assn_poly_params", [](const Stats& stats) { return stats.p_assn_poly_params; });
			printAllMap("n_assn_poly_returns", [](const Stats& stats) { return stats.n_assn_poly_returns; });
			printAllMap("n_assn_returns", [](const Stats& stats) { return stats.n_assn_returns; });
			printAllMap("%_assn_poly_returns", [](const Stats& stats) { return stats.p_assn_poly_returns; });
		}
	};

	void printDeclStats( std::list< Declaration * > &translationUnit ) {
		DeclStats stats;
		acceptAll( translationUnit, stats );
		stats.print();
	}
	
} // namespace CodeTools

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
