//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Constant.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Feb 13 18:11:22 2019
// Update Count     : 32
//

#include <cassert>   // for strict_dynamic_cast, assertf
#include <iostream>  // for operator<<, ostream, basic_ostream
#include <string>    // for to_string, string, char_traits, operator<<

#include "Constant.h"
#include "Expression.h" // for ConstantExpr
#include "Type.h"    // for BasicType, Type, Type::Qualifiers, PointerType

Constant::Constant( Type * type, std::string rep, unsigned long long val ) : type( type ), rep( rep ), val( val ) {}
Constant::Constant( Type * type, std::string rep, double val ) : type( type ), rep( rep ), val( val ) {}

Constant::Constant( const Constant &other ) : BaseSyntaxNode( other ), rep( other.rep ), val( other.val ) {
	type = other.type->clone();
}

Constant::~Constant() { delete type; }

Constant Constant::from_bool( bool b ) {
	return Constant( new BasicType( Type::Qualifiers(), BasicType::Bool ), b ? "1" : "0" , (unsigned long long int)b );
}

Constant Constant::from_char( char c ) {
	return Constant( new BasicType( Type::Qualifiers(), BasicType::Char ), std::to_string( c ), (unsigned long long int)c );
}

Constant Constant::from_int( int i ) {
	return Constant( new BasicType( Type::Qualifiers(), BasicType::SignedInt ), std::to_string( i ), (unsigned long long int)i );
}

Constant Constant::from_ulong( unsigned long i ) {
	return Constant( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), std::to_string( i ), (unsigned long long int)i );
}

Constant Constant::from_double( double d ) {
	return Constant( new BasicType( Type::Qualifiers(), BasicType::Double ), std::to_string( d ), d );
}

Constant Constant::from_string( std::string const & cEscapedVal, Type *charType ) {
	assert(cEscapedVal.length() >= 2);
	assert(cEscapedVal.front() == '"');
	assert(cEscapedVal.back() == '"');
	ArrayType * at = new ArrayType( noQualifiers, charType,
									new ConstantExpr( Constant::from_ulong( cEscapedVal.size() + 1 - 2 ) ), // +1 for '\0' and -2 for '"'
									false, false );
	return Constant( at, cEscapedVal, (unsigned long long int)0 );
}

Constant Constant::null( Type * ptrtype ) {
	if ( nullptr == ptrtype ) {
		ptrtype = new PointerType(
			Type::Qualifiers(),
			new VoidType( Type::Qualifiers() )
			);
	}

	return Constant( ptrtype, "0", (unsigned long long int)0 );
}

unsigned long long Constant::get_ival() const {
	assertf( strict_dynamic_cast<BasicType*>(type)->isInteger(), "Attempt to retrieve ival from non-integer constant." );
	return val.ival;
}

double Constant::get_dval() const {
	assertf( ! strict_dynamic_cast<BasicType*>(type)->isInteger(), "Attempt to retrieve dval from integer constant." );
	return val.dval;
}

void Constant::print( std::ostream &os, Indenter ) const {
	os << "(" << rep << " " << val.ival;
	if ( type ) {
		os << ": ";
		type->print( os );
	} // if
  os << ")";
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
