//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Specialize.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Wed Jan 20 13:00:00 2016
// Update Count     : 24
//

#include <cassert>

#include "Specialize.h"
#include "GenPoly.h"
#include "PolyMutator.h"

#include "Parser/ParseNode.h"

#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Type.h"
#include "SynTree/TypeSubstitution.h"
#include "SynTree/Mutator.h"
#include "ResolvExpr/FindOpenVars.h"
#include "Common/UniqueName.h"
#include "Common/utility.h"

namespace GenPoly {
	const std::list<Label> noLabels;

	class Specialize : public PolyMutator {
	  public:
		Specialize( std::string paramPrefix = "_p" );

		virtual Expression * mutate( ApplicationExpr *applicationExpr );
		virtual Expression * mutate( AddressExpr *castExpr );
		virtual Expression * mutate( CastExpr *castExpr );
		virtual Expression * mutate( LogicalExpr *logicalExpr );
		virtual Expression * mutate( ConditionalExpr *conditionalExpr );
		virtual Expression * mutate( CommaExpr *commaExpr );

	  private:
		Expression *createThunkFunction( FunctionType *funType, Expression *actual, InferredParams *inferParams = 0 );
		Expression *doSpecialization( Type *formalType, Expression *actual, InferredParams *inferParams = 0 );
		void handleExplicitParams( ApplicationExpr *appExpr );

		UniqueName thunkNamer;
		std::string paramPrefix;
	};

	void convertSpecializations( std::list< Declaration* >& translationUnit ) {
		Specialize specializer;
		mutateAll( translationUnit, specializer );
	}

	Specialize::Specialize( std::string paramPrefix )
		: thunkNamer( "_thunk" ), paramPrefix( paramPrefix ) {
	}

	/// Looks up open variables in actual type, returning true if any of them are bound in the environment or formal type.
	bool needsSpecialization( Type *formalType, Type *actualType, TypeSubstitution *env ) {
		if ( env ) {
			using namespace ResolvExpr;
			OpenVarSet openVars, closedVars;
			AssertionSet need, have;
			findOpenVars( formalType, openVars, closedVars, need, have, false );
			findOpenVars( actualType, openVars, closedVars, need, have, true );
			for ( OpenVarSet::const_iterator openVar = openVars.begin(); openVar != openVars.end(); ++openVar ) {
				Type *boundType = env->lookup( openVar->first );
				if ( ! boundType ) continue;
				if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( boundType ) ) {
					if ( closedVars.find( typeInst->get_name() ) == closedVars.end() ) {
						return true;
					} // if
				} else {
					return true;
				} // if
			} // for
			return false;
		} else {
			return false;
		} // if
	}

	/// Generates a thunk that calls `actual` with type `funType` and returns its address
	Expression * Specialize::createThunkFunction( FunctionType *funType, Expression *actual, InferredParams *inferParams ) {
		FunctionType *newType = funType->clone();
		if ( env ) {
			TypeSubstitution newEnv( *env );
			// it is important to replace only occurrences of type variables that occur free in the
			// thunk's type
			newEnv.applyFree( newType );
		} // if
		// create new thunk with same signature as formal type (C linkage, empty body)
		FunctionDecl *thunkFunc = new FunctionDecl( thunkNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, newType, new CompoundStmt( std::list< std::string >() ), false, false );
		thunkFunc->fixUniqueId();

		// thread thunk parameters into call to actual function, naming thunk parameters as we go
		UniqueName paramNamer( paramPrefix );
		ApplicationExpr *appExpr = new ApplicationExpr( actual );
		for ( std::list< DeclarationWithType* >::iterator param = thunkFunc->get_functionType()->get_parameters().begin(); param != thunkFunc->get_functionType()->get_parameters().end(); ++param ) {
			(*param )->set_name( paramNamer.newName() );
			appExpr->get_args().push_back( new VariableExpr( *param ) );
		} // for
		appExpr->set_env( maybeClone( env ) );
		if ( inferParams ) {
			appExpr->get_inferParams() = *inferParams;
		} // if

		// handle any specializations that may still be present
		std::string oldParamPrefix = paramPrefix;
		paramPrefix += "p";
		// save stmtsToAdd in oldStmts
		std::list< Statement* > oldStmts;
		oldStmts.splice( oldStmts.end(), stmtsToAdd );
		handleExplicitParams( appExpr );
		paramPrefix = oldParamPrefix;
		// write any statements added for recursive specializations into the thunk body
		thunkFunc->get_statements()->get_kids().splice( thunkFunc->get_statements()->get_kids().end(), stmtsToAdd );
		// restore oldStmts into stmtsToAdd
		stmtsToAdd.splice( stmtsToAdd.end(), oldStmts );

		// add return (or valueless expression) to the thunk
		Statement *appStmt;
		if ( funType->get_returnVals().empty() ) {
			appStmt = new ExprStmt( noLabels, appExpr );
		} else {
			appStmt = new ReturnStmt( noLabels, appExpr );
		} // if
		thunkFunc->get_statements()->get_kids().push_back( appStmt );

		// add thunk definition to queue of statements to add
		stmtsToAdd.push_back( new DeclStmt( noLabels, thunkFunc ) );
		// return address of thunk function as replacement expression
		return new AddressExpr( new VariableExpr( thunkFunc ) );
	}

	Expression * Specialize::doSpecialization( Type *formalType, Expression *actual, InferredParams *inferParams ) {
		assert( ! actual->get_results().empty() ); // using front, should have this assert
		if ( needsSpecialization( formalType, actual->get_results().front(), env ) ) {
			FunctionType *funType;
			if ( ( funType = getFunctionType( formalType ) ) ) {
				ApplicationExpr *appExpr;
				VariableExpr *varExpr;
				if ( ( appExpr = dynamic_cast<ApplicationExpr*>( actual ) ) ) {
					return createThunkFunction( funType, appExpr->get_function(), inferParams );
				} else if ( ( varExpr = dynamic_cast<VariableExpr*>( actual ) ) ) {
					return createThunkFunction( funType, varExpr, inferParams );
				} else {
					// This likely won't work, as anything that could build an ApplicationExpr probably hit one of the previous two branches
					return createThunkFunction( funType, actual, inferParams );
				}
			} else {
				return actual;
			} // if
		} else {
			return actual;
		} // if
	}

	void Specialize::handleExplicitParams( ApplicationExpr *appExpr ) {
		// create thunks for the explicit parameters
		assert( ! appExpr->get_function()->get_results().empty() );
		FunctionType *function = getFunctionType( appExpr->get_function()->get_results().front() );
		assert( function );
		std::list< DeclarationWithType* >::iterator formal;
		std::list< Expression* >::iterator actual;
		for ( formal = function->get_parameters().begin(), actual = appExpr->get_args().begin(); formal != function->get_parameters().end() && actual != appExpr->get_args().end(); ++formal, ++actual ) {
			*actual = doSpecialization( (*formal )->get_type(), *actual, &appExpr->get_inferParams() );
		}
	}

	Expression * Specialize::mutate( ApplicationExpr *appExpr ) {
		appExpr->get_function()->acceptMutator( *this );
		mutateAll( appExpr->get_args(), *this );

		// create thunks for the inferred parameters
		for ( InferredParams::iterator inferParam = appExpr->get_inferParams().begin(); inferParam != appExpr->get_inferParams().end(); ++inferParam ) {
			inferParam->second.expr = doSpecialization( inferParam->second.formalType, inferParam->second.expr, &appExpr->get_inferParams() );
		}

		handleExplicitParams( appExpr );

		return appExpr;
	}

	Expression * Specialize::mutate( AddressExpr *addrExpr ) {
		addrExpr->get_arg()->acceptMutator( *this );
		assert( ! addrExpr->get_results().empty() );
		addrExpr->set_arg( doSpecialization( addrExpr->get_results().front(), addrExpr->get_arg() ) );
		return addrExpr;
	}

	Expression * Specialize::mutate( CastExpr *castExpr ) {
		castExpr->get_arg()->acceptMutator( *this );
		if ( castExpr->get_results().empty() ) {
			// can't specialize if we don't have a return value
			return castExpr;
		}
		Expression *specialized = doSpecialization( castExpr->get_results().front(), castExpr->get_arg() );
		if ( specialized != castExpr->get_arg() ) {
			// assume here that the specialization incorporates the cast
			return specialized;
		} else {
			return castExpr;
		}
	}

	Expression * Specialize::mutate( LogicalExpr *logicalExpr ) {
		return logicalExpr;
	}

	Expression * Specialize::mutate( ConditionalExpr *condExpr ) {
		return condExpr;
	}

	Expression * Specialize::mutate( CommaExpr *commaExpr ) {
		return commaExpr;
	}
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
