//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Expression.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Apr 26 12:58:05 2016
// Update Count     : 40
//

#include <iostream>
#include <cassert>
#include <list>
#include <algorithm>

#include <iterator>

#include "Type.h"
#include "Initializer.h"
#include "Expression.h"
#include "Declaration.h"
#include "Statement.h"
#include "TypeSubstitution.h"
#include "Common/utility.h"


Expression::Expression( Expression *_aname ) : env( 0 ), argName( _aname ) {}

Expression::Expression( const Expression &other ) : env( maybeClone( other.env ) ), argName( maybeClone( other.get_argName() ) ) {
	cloneAll( other.results, results );
}

Expression::~Expression() {
	delete env;
	delete argName;	// xxx -- there's a problem in cloning ConstantExpr I still don't know how to fix
	deleteAll( results );
}

void Expression::add_result( Type *t ) {
	if ( TupleType *tuple = dynamic_cast< TupleType* >( t ) ) {
		std::copy( tuple->get_types().begin(), tuple->get_types().end(), back_inserter( results ) );
	} else {
		results.push_back(t);
	} // if
}

void Expression::print( std::ostream &os, int indent ) const {
	if ( env ) {
		os << std::string( indent, ' ' ) << "with environment:" << std::endl;
		env->print( os, indent+2 );
	} // if

	if ( argName ) {
		os << std::string( indent, ' ' ) << "with designator:";
		argName->print( os, indent+2 );
	} // if
}

ConstantExpr::ConstantExpr( Constant _c, Expression *_aname ) : Expression( _aname ), constant( _c ) {
	add_result( constant.get_type()->clone() );
}

ConstantExpr::ConstantExpr( const ConstantExpr &other) : Expression( other ), constant( other.constant ) {
}

ConstantExpr::~ConstantExpr() {}

void ConstantExpr::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) << "constant expression " ;
	constant.print( os );
	Expression::print( os, indent );
	os << std::endl;
}

VariableExpr::VariableExpr( DeclarationWithType *_var, Expression *_aname ) : Expression( _aname ), var( _var ) {
	assert( var );
	assert( var->get_type() );
	add_result( var->get_type()->clone() );
	for ( std::list< Type* >::iterator i = get_results().begin(); i != get_results().end(); ++i ) {
		(*i)->set_isLvalue( true );
	} // for
}

VariableExpr::VariableExpr( const VariableExpr &other ) : Expression( other ), var( other.var ) {
}

VariableExpr::~VariableExpr() {
	// don't delete the declaration, since it points somewhere else in the tree
}

void VariableExpr::print( std::ostream &os, int indent ) const {
	os << "Variable Expression: ";

	Declaration *decl = get_var();
	// if ( decl != 0) decl->print(os, indent + 2);
	if ( decl != 0) decl->printShort(os, indent + 2);
	os << std::endl;
	Expression::print( os, indent );
}

SizeofExpr::SizeofExpr( Expression *expr_, Expression *_aname ) :
		Expression( _aname ), expr(expr_), type(0), isType(false) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) );
}

SizeofExpr::SizeofExpr( Type *type_, Expression *_aname ) :
		Expression( _aname ), expr(0), type(type_), isType(true) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) );
}

SizeofExpr::SizeofExpr( const SizeofExpr &other ) :
	Expression( other ), expr( maybeClone( other.expr ) ), type( maybeClone( other.type ) ), isType( other.isType ) {
}

SizeofExpr::~SizeofExpr() {
	delete expr;
	delete type;
}

void SizeofExpr::print( std::ostream &os, int indent) const {
	os << "Sizeof Expression on: ";

	if (isType)
		type->print(os, indent + 2);
	else
		expr->print(os, indent + 2);

	os << std::endl;
	Expression::print( os, indent );
}

AlignofExpr::AlignofExpr( Expression *expr_, Expression *_aname ) :
		Expression( _aname ), expr(expr_), type(0), isType(false) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) );
}

AlignofExpr::AlignofExpr( Type *type_, Expression *_aname ) :
		Expression( _aname ), expr(0), type(type_), isType(true) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) );
}

AlignofExpr::AlignofExpr( const AlignofExpr &other ) :
	Expression( other ), expr( maybeClone( other.expr ) ), type( maybeClone( other.type ) ), isType( other.isType ) {
}

AlignofExpr::~AlignofExpr() {
	delete expr;
	delete type;
}

void AlignofExpr::print( std::ostream &os, int indent) const {
	os << std::string( indent, ' ' ) << "Alignof Expression on: ";

	if (isType)
		type->print(os, indent + 2);
	else
		expr->print(os, indent + 2);

	os << std::endl;
	Expression::print( os, indent );
}

UntypedOffsetofExpr::UntypedOffsetofExpr( Type *type_, const std::string &member_, Expression *_aname ) :
		Expression( _aname ), type(type_), member(member_) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) );
}

UntypedOffsetofExpr::UntypedOffsetofExpr( const UntypedOffsetofExpr &other ) :
	Expression( other ), type( maybeClone( other.type ) ), member( other.member ) {}

UntypedOffsetofExpr::~UntypedOffsetofExpr() {
	delete type;
}

void UntypedOffsetofExpr::print( std::ostream &os, int indent) const {
	os << std::string( indent, ' ' ) << "Untyped Offsetof Expression on member " << member << " of ";

	if ( type ) {
		type->print(os, indent + 2);
	} else {
		os << "<NULL>";
	}

	os << std::endl;
	Expression::print( os, indent );
}

OffsetofExpr::OffsetofExpr( Type *type_, DeclarationWithType *member_, Expression *_aname ) :
		Expression( _aname ), type(type_), member(member_) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) );
}

OffsetofExpr::OffsetofExpr( const OffsetofExpr &other ) :
	Expression( other ), type( maybeClone( other.type ) ), member( maybeClone( other.member ) ) {}

OffsetofExpr::~OffsetofExpr() {
	delete type;
	delete member;
}

void OffsetofExpr::print( std::ostream &os, int indent) const {
	os << std::string( indent, ' ' ) << "Offsetof Expression on member ";

	if ( member ) {
		os << member->get_name();
	} else {
		os << "<NULL>";
	}

	os << " of ";

	if ( type ) {
		type->print(os, indent + 2);
	} else {
		os << "<NULL>";
	}

	os << std::endl;
	Expression::print( os, indent );
}

OffsetPackExpr::OffsetPackExpr( StructInstType *type_, Expression *aname_ ) : Expression( aname_ ), type( type_ ) {
	add_result( new ArrayType( Type::Qualifiers(), new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), 0, false, false ) );
}

OffsetPackExpr::OffsetPackExpr( const OffsetPackExpr &other ) : Expression( other ), type( maybeClone( other.type ) ) {}

OffsetPackExpr::~OffsetPackExpr() { delete type; }

void OffsetPackExpr::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) << "Offset pack expression on ";

	if ( type ) {
		type->print(os, indent + 2);
	} else {
		os << "<NULL>";
	}

	os << std::endl;
	Expression::print( os, indent );
}

AttrExpr::AttrExpr( Expression *attr, Expression *expr_, Expression *_aname ) :
		Expression( _aname ), attr( attr ), expr(expr_), type(0), isType(false) {
}

AttrExpr::AttrExpr( Expression *attr, Type *type_, Expression *_aname ) :
		Expression( _aname ), attr( attr ), expr(0), type(type_), isType(true) {
}

AttrExpr::AttrExpr( const AttrExpr &other ) :
		Expression( other ), attr( maybeClone( other.attr ) ), expr( maybeClone( other.expr ) ), type( maybeClone( other.type ) ), isType( other.isType ) {
}

AttrExpr::~AttrExpr() {
	delete attr;
	delete expr;
	delete type;
}

void AttrExpr::print( std::ostream &os, int indent) const {
	os << std::string( indent, ' ' ) << "Attr ";
	attr->print( os, indent + 2 );
	if ( isType || expr ) {
		os << "applied to: ";

		if (isType)
			type->print(os, indent + 2);
		else
			expr->print(os, indent + 2);
	} // if

	os << std::endl;
	Expression::print( os, indent );
}

CastExpr::CastExpr( Expression *arg_, Type *toType, Expression *_aname ) : Expression( _aname ), arg(arg_) {
	add_result(toType);
}

CastExpr::CastExpr( Expression *arg_, Expression *_aname ) : Expression( _aname ), arg(arg_) {
}

CastExpr::CastExpr( const CastExpr &other ) : Expression( other ), arg( maybeClone( other.arg ) ) {
}

CastExpr::~CastExpr() {
	delete arg;
}

// CastExpr *CastExpr::clone() const { return 0; }

void CastExpr::print( std::ostream &os, int indent ) const {
	os << "Cast of:" << std::endl << std::string( indent+2, ' ' );
	arg->print(os, indent+2);
	os << std::endl << std::string( indent, ' ' ) << "to:" << std::endl;
	if ( results.empty() ) {
		os << std::string( indent+2, ' ' ) << "nothing" << std::endl;
	} else {
		printAll(results, os, indent+2);
	} // if
	Expression::print( os, indent );
}

UntypedMemberExpr::UntypedMemberExpr( std::string _member, Expression *_aggregate, Expression *_aname ) :
		Expression( _aname ), member(_member), aggregate(_aggregate) {}

UntypedMemberExpr::UntypedMemberExpr( const UntypedMemberExpr &other ) :
		Expression( other ), member( other.member ), aggregate( maybeClone( other.aggregate ) ) {
}

UntypedMemberExpr::~UntypedMemberExpr() {
	delete aggregate;
}

void UntypedMemberExpr::print( std::ostream &os, int indent ) const {
	os << "Untyped Member Expression, with field: " << get_member();

	Expression *agg = get_aggregate();
	os << std::string( indent, ' ' ) << "from aggregate: ";
	if (agg != 0) {
		os << std::string( indent+2, ' ' );
		agg->print(os, indent + 2);
	}
	os << std::string( indent+2, ' ' );
	Expression::print( os, indent );
}


MemberExpr::MemberExpr( DeclarationWithType *_member, Expression *_aggregate, Expression *_aname ) :
		Expression( _aname ), member(_member), aggregate(_aggregate) {
	add_result( member->get_type()->clone() );
	for ( std::list< Type* >::iterator i = get_results().begin(); i != get_results().end(); ++i ) {
		(*i)->set_isLvalue( true );
	} // for
}

MemberExpr::MemberExpr( const MemberExpr &other ) :
		Expression( other ), member( maybeClone( other.member ) ), aggregate( maybeClone( other.aggregate ) ) {
}

MemberExpr::~MemberExpr() {
	delete member;
	delete aggregate;
}

void MemberExpr::print( std::ostream &os, int indent ) const {
	os << "Member Expression, with field: " << std::endl;

	assert( member );
	os << std::string( indent + 2, ' ' );
	member->print( os, indent + 2 );
	os << std::endl;

	Expression *agg = get_aggregate();
	os << std::string( indent, ' ' ) << "from aggregate: " << std::endl;
	if (agg != 0) {
		os << std::string( indent+2, ' ' );
		agg->print(os, indent + 2);
	}
	os << std::string( indent+2, ' ' );
	Expression::print( os, indent );
}


UntypedExpr::UntypedExpr( Expression *_function, Expression *_aname ) : Expression( _aname ), function( _function ) {}

UntypedExpr::UntypedExpr( const UntypedExpr &other ) :
		Expression( other ), function( maybeClone( other.function ) ) {
	cloneAll( other.args, args );
}

UntypedExpr::UntypedExpr( Expression *_function, std::list<Expression *> &_args, Expression *_aname ) :
		Expression( _aname ), function(_function), args(_args) {}

UntypedExpr::~UntypedExpr() {}

void UntypedExpr::print( std::ostream &os, int indent ) const {
	os << "Applying untyped: " << std::endl;
	os << std::string( indent, ' ' );
	function->print(os, indent + 4);
	os << std::string( indent, ' ' ) << "...to: " << std::endl;
	os << std::string( indent, ' ' );
	printArgs(os, indent + 4);
	Expression::print( os, indent );
}

void UntypedExpr::printArgs( std::ostream &os, int indent ) const {
	std::list<Expression *>::const_iterator i;
	for (i = args.begin(); i != args.end(); i++)
		(*i)->print(os, indent);
}

NameExpr::NameExpr( std::string _name, Expression *_aname ) : Expression( _aname ), name(_name) {}

NameExpr::NameExpr( const NameExpr &other ) : Expression( other ), name( other.name ) {
}

NameExpr::~NameExpr() {}

void NameExpr::print( std::ostream &os, int indent ) const {
	os << "Name: " << get_name() << std::endl;
	Expression::print( os, indent );
}

LogicalExpr::LogicalExpr( Expression *arg1_, Expression *arg2_, bool andp, Expression *_aname ) :
		Expression( _aname ), arg1(arg1_), arg2(arg2_), isAnd(andp) {
	add_result( new BasicType( Type::Qualifiers(), BasicType::SignedInt ) );
}

LogicalExpr::LogicalExpr( const LogicalExpr &other ) :
		Expression( other ), arg1( maybeClone( other.arg1 ) ), arg2( maybeClone( other.arg2 ) ), isAnd( other.isAnd ) {
}

LogicalExpr::~LogicalExpr() {
	delete arg1;
	delete arg2;
}

void LogicalExpr::print( std::ostream &os, int indent )const {
	os << std::string( indent, ' ' ) << "Short-circuited operation (" << (isAnd?"and":"or") << ") on: ";
	arg1->print(os);
	os << " and ";
	arg2->print(os);
	os << std::endl;
	Expression::print( os, indent );
}

ConditionalExpr::ConditionalExpr( Expression *arg1_, Expression *arg2_, Expression *arg3_, Expression *_aname ) :
		Expression( _aname ), arg1(arg1_), arg2(arg2_), arg3(arg3_) {}

ConditionalExpr::ConditionalExpr( const ConditionalExpr &other ) :
		Expression( other ), arg1( maybeClone( other.arg1 ) ), arg2( maybeClone( other.arg2 ) ), arg3( maybeClone( other.arg3 ) ) {
}

ConditionalExpr::~ConditionalExpr() {
	delete arg1;
	delete arg2;
	delete arg3;
}

void ConditionalExpr::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) << "Conditional expression on: " << std::endl;
	arg1->print( os, indent+2 );
	os << std::string( indent, ' ' ) << "First alternative:" << std::endl;
	arg2->print( os, indent+2 );
	os << std::string( indent, ' ' ) << "Second alternative:" << std::endl;
	arg3->print( os, indent+2 );
	os << std::endl;
	Expression::print( os, indent );
}

AsmExpr::AsmExpr( const AsmExpr & other ) : inout( maybeClone( other.inout ) ), constraint( maybeClone( other.constraint ) ), operand( maybeClone( other.operand ) ) {}


void AsmExpr::print( std::ostream &os, int indent ) const {
	os << "Asm Expression: " << std::endl;
	if ( inout ) inout->print( os, indent + 2 );
	if ( constraint ) constraint->print( os, indent + 2 );
	if ( operand ) operand->print( os, indent + 2 );
}


ImplicitCopyCtorExpr::ImplicitCopyCtorExpr( ApplicationExpr * callExpr ) : callExpr( callExpr ) {
	assert( callExpr );
	cloneAll( callExpr->get_results(), results );
}

ImplicitCopyCtorExpr::ImplicitCopyCtorExpr( const ImplicitCopyCtorExpr & other ) : Expression( other ), callExpr( maybeClone( other.callExpr ) ) {
	cloneAll( other.copyCtors, copyCtors );
	cloneAll( other.tempDecls, tempDecls );
	cloneAll( other.returnDecls, returnDecls );
	cloneAll( other.dtors, dtors );
}

ImplicitCopyCtorExpr::~ImplicitCopyCtorExpr() {
	delete callExpr;
	deleteAll( copyCtors );
	deleteAll( tempDecls );
	deleteAll( returnDecls );
	deleteAll( dtors );
}

void ImplicitCopyCtorExpr::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) <<  "Implicit Copy Constructor Expression: " << std::endl;
	assert( callExpr );
	callExpr->print( os, indent + 2 );
	os << std::endl << std::string( indent, ' ' ) << "with temporaries:" << std::endl;
	printAll(tempDecls, os, indent+2);
	os << std::endl << std::string( indent, ' ' ) << "with copyCtors:" << std::endl;
	printAll(copyCtors, os, indent+2);
	os << std::endl << std::string( indent, ' ' ) << "with return temporaries:" << std::endl;
	printAll(returnDecls, os, indent+2);
	Expression::print( os, indent );
}

UntypedValofExpr::UntypedValofExpr( const UntypedValofExpr & other ) : Expression( other ), body ( maybeClone( other.body ) ) {}

UntypedValofExpr::~UntypedValofExpr() { delete body; }

void UntypedValofExpr::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) << "Valof Expression: " << std::endl;
	if ( get_body() != 0 )
		get_body()->print( os, indent + 2 );
}


CompoundLiteralExpr::CompoundLiteralExpr( Type * type, Initializer * initializer ) : type( type ), initializer( initializer ) {
	add_result( type->clone() );
}

CompoundLiteralExpr::CompoundLiteralExpr( const CompoundLiteralExpr &other ) : Expression( other ), type( maybeClone( other.type ) ), initializer( maybeClone( other.initializer ) ) {}

CompoundLiteralExpr::~CompoundLiteralExpr() {
	delete initializer;
	delete type;
}

void CompoundLiteralExpr::print( std::ostream &os, int indent ) const {
	os << "Compound Literal Expression: " << std::endl;
	if ( type ) type->print( os, indent + 2 );
	if ( initializer ) initializer->print( os, indent + 2 );
}


std::ostream & operator<<( std::ostream & out, Expression * expr ) {
	expr->print( out );
	return out;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
