//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Tuples.h --
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 15:04:02 2015
// Update Count     : 2
//

#ifndef _TUPLES_H_
#define _TUPLES_H_

#include <string>
#include <vector>
#include "ResolvExpr/AlternativeFinder.h"

#include "SynTree/Expression.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"

namespace Tuples {
	// TupleAssignment.cc
	void handleTupleAssignment( ResolvExpr::AlternativeFinder & currentFinder, UntypedExpr * assign, const std::list<ResolvExpr::AltList> & possibilities );

	// TupleExpansion.cc
	void expandTuples( std::list< Declaration * > & translationUnit );

  void expandUniqueExpr( std::list< Declaration * > & translationUnit );

  /// returns VoidType if any of the expressions have Voidtype, otherwise TupleType of the Expression result types
  Type * makeTupleType( const std::list< Expression * > & exprs );

  bool maybeImpure( Expression * expr );


	/// helper function used by explode
	template< typename OutputIterator >
	void explodeUnique( Expression * expr, const ResolvExpr::Alternative & alt, OutputIterator out ) {
		Type * res = expr->get_result();
		if ( AddressExpr * addrExpr = dynamic_cast< AddressExpr * >( expr ) ) {
			ResolvExpr::AltList alts;
			explodeUnique( addrExpr->get_arg(), alt, back_inserter( alts ) );
			for ( ResolvExpr::Alternative & alt : alts ) {
				// distribute '&' over all components
				alt.expr = new AddressExpr( alt.expr );
				*out++ = alt;
			}
		} else if ( TupleType * tupleType = dynamic_cast< TupleType * > ( res ) ) {
			if ( TupleExpr * tupleExpr = dynamic_cast< TupleExpr * >( expr ) ) {
				// can open tuple expr and dump its exploded components
				for ( Expression * expr : tupleExpr->get_exprs() ) {
					explodeUnique( expr, alt, out );
				}
			} else {
				// tuple type, but not tuple expr - recursively index into its components
				Expression * arg = expr->clone();
				if ( Tuples::maybeImpure( arg ) ) {
					// expressions which may contain side effects require a single unique instance of the expression
					arg = new UniqueExpr( arg );
				}
				for ( unsigned int i = 0; i < tupleType->size(); i++ ) {
					TupleIndexExpr * idx = new TupleIndexExpr( arg->clone(), i );
					explodeUnique( idx, alt, out );
					delete idx;
				}
				delete arg;
			}
		} else {
			// atomic (non-tuple) type - output a clone of the expression in a new alternative
			*out++ = ResolvExpr::Alternative( expr->clone(), alt.env, alt.cost, alt.cvtCost );
		}
	}

	/// expands a tuple-valued alternative into multiple alternatives, each with a non-tuple-type
	template< typename OutputIterator >
	void explode( const ResolvExpr::Alternative &alt, OutputIterator out ) {
		explodeUnique( alt.expr, alt, out );
	}

	// explode list of alternatives
	template< typename AltIterator, typename OutputIterator >
	void explode( AltIterator altBegin, AltIterator altEnd, OutputIterator out ) {
		for ( ; altBegin != altEnd; ++altBegin ) {
			explode( *altBegin, out );
		}
	}

	template< typename OutputIterator >
	void explode( const ResolvExpr::AltList & alts, OutputIterator out ) {
		explode( alts.begin(), alts.end(), out );
	}
} // namespace Tuples

#endif // _TUPLE_ASSIGNMENT_H_

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
