//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// DeclStats.cc --
//
// Author           : Aaron Moss
// Created On       : Wed Jan 31 16:40:00 2016
// Last Modified By : Aaron Moss
// Last Modified On : Wed Jan 31 16:40:00 2016
// Update Count     : 1
//

#include "DeclStats.h"

#include <iostream>
#include <string>
#include <unordered_map>
#include <unordered_set>

#include "Common/VectorMap.h"
#include "GenPoly/GenPoly.h"
#include "Parser/LinkageSpec.h"
#include "SynTree/Declaration.h"
#include "SynTree/Visitor.h"

namespace CodeTools {
	
	class DeclStats : public Visitor {
		struct Stats {
			unsigned n_decls;     ///< Total number of declarations
			unsigned mono_decls;  ///< Monomorphic declarations
			unsigned poly_decls;  ///< Polymorphic declarations
			/// Count of declarations with each name
			std::unordered_map<std::string, unsigned> by_name;
			/// Count of declarations with each number of parameters
			VectorMap<unsigned> by_params;
			/// Count of declarations with each number of return types
			VectorMap<unsigned> by_returns;
			/// Count of declarations with each number of polymorphic parameters
			VectorMap<unsigned> n_poly_params;
			/// Count of declarations with each number of polymorphic return types
			VectorMap<unsigned> n_poly_returns;

			Stats() : n_decls(0), mono_decls(0), poly_decls(0), by_name(), by_params(), by_returns(), n_poly_params(), n_poly_returns() {}

			Stats& operator+= (const Stats& o) {
				n_decls += o.n_decls;
				mono_decls += o.mono_decls;
				poly_decls += o.poly_decls;
				
				for ( auto& entry : o.by_name ) {
					by_name[ entry.first ] += entry.second;
				}
				
				by_params.reserve( o.by_params.size() );
				for ( unsigned i = 0; i < o.by_params.size(); ++i ) {
					by_params[i] += o.by_params[i];
				}

				by_returns.reserve( o.by_returns.size() );
				for ( unsigned i = 0; i < o.by_returns.size(); ++i ) {
					by_returns[i] += o.by_returns[i];
				}

				n_poly_params.reserve( o.n_poly_params.size() );
				for ( unsigned i = 0; i < o.n_poly_params.size(); ++i ) {
					n_poly_params[i] += o.n_poly_params[i];
				}

				n_poly_returns.reserve( o.n_poly_returns.size() );
				for ( unsigned i = 0; i < o.n_poly_returns.size(); ++i ) {
					n_poly_returns[i] += o.n_poly_returns[i];
				}

				return *this;
			}
		};

		Stats for_linkage[LinkageSpec::NoOfSpecs];   ///< Stores separate stats per linkage
		std::unordered_set<std::string> seen_names;  ///< Stores manglenames already seen to avoid double-counting
		Stats total;

	public:
		using Visitor::visit;

		virtual void visit( FunctionDecl *decl ) {
			// skip if already seen declaration for this function
			const std::string& mangleName = decl->get_mangleName().empty() ? decl->get_name() : decl->get_mangleName();
			if ( ! seen_names.insert( mangleName ).second ) return;
			
			Stats& stats = for_linkage[ decl->get_linkage() ];

			++stats.n_decls;
			FunctionType* fnTy = decl->get_functionType();
			if ( fnTy->get_forall().empty() ) ++stats.mono_decls; else ++stats.poly_decls;

			++stats.by_name[ decl->get_name() ];

			unsigned n_params = 0;
			unsigned n_poly_params = 0;
			for ( auto pdecl : fnTy->get_parameters() ) {
				n_params += pdecl->get_type()->size();
				if ( GenPoly::hasPolyBase( pdecl->get_type() ) ) ++n_poly_params;
			}
			++stats.by_params.at( n_params );
			++stats.n_poly_params.at( n_poly_params );

			unsigned n_returns = 0;
			unsigned n_poly_returns = 0;
			for ( auto rdecl : fnTy->get_returnVals() ) {
				n_returns += rdecl->get_type()->size();
				if ( GenPoly::hasPolyBase( rdecl->get_type() ) ) ++n_poly_returns;
			}
			++stats.by_returns.at( n_returns );
			++stats.n_poly_returns.at( n_poly_returns );
		}

	private:
		template<typename F>
		void printAll( const char* name, F extract ) {
			std::cout << "\"" << name << "\",";
			for ( const auto& stats : for_linkage ) {
				std::cout << "," << extract(stats);
			}
			std::cout << "," << extract(total) << std::endl;
		}

		template<typename F>
		void printAllMap( const char* name, F extract ) {
			for ( const auto& entry : extract(total) ) {
				const auto& key = entry.first;
				std::cout << "\"" << name << "\"," << key;
				for ( const auto& stats : for_linkage ) {
					const auto& map = extract(stats);
					auto it = map.find( key );
					if ( it == map.end() ) std::cout << ",0";
					else std::cout << "," << it->second;
				}
				std::cout  << "," << entry.second << std::endl;
			}
		}

		template<typename F>
		void printAllHisto( const char* name, F extract ) {
			VectorMap<unsigned> histos[LinkageSpec::NoOfSpecs];
			VectorMap<unsigned> thisto;

			for ( const auto& entry : extract(total) ) { ++thisto.at( entry.second ); }

			for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
				// can't be a higher count in one of the sub-histograms than the total
				histos[i].reserve( thisto.size() );

				for ( const auto& entry : extract(for_linkage[i]) ) { ++histos[i][entry.second]; }
			}

			for ( unsigned i = 0; i < thisto.size(); ++i ) {
				std::cout << "\"" << name << "\"," << i;
				for ( const auto& histo : histos ) {
					std::cout << "," << histo[i];
				}
				std::cout << "," << thisto[i] << std::endl;
			}
		}
		
	public:
		void print() {
			for ( auto& stats : for_linkage ) {
				total += stats;
			}

			std::cout << ",,\"intrinsic\",\"Cforall\",\"C\",\"autogen\",\"builtin\",\"TOTAL\"" << std::endl;

			printAll("mono_decls", [](const Stats& stats) { return stats.mono_decls; });
			printAll("poly_decls", [](const Stats& stats) { return stats.poly_decls; });
			printAll("n_decls", [](const Stats& stats) { return stats.n_decls; });
			printAll("unique_names", [](const Stats& stats) { return stats.by_name.size(); });

			printAllHisto("overloads", [](const Stats& stats) { return stats.by_name; });
			printAllMap("n_poly_params", [](const Stats& stats) { return stats.n_poly_params; });
			printAllMap("n_params", [](const Stats& stats) { return stats.by_params; });
			printAllMap("n_poly_returns", [](const Stats& stats) { return stats.n_poly_returns; });
			printAllMap("n_returns", [](const Stats& stats) { return stats.by_returns; });
		}
	};

	void printDeclStats( std::list< Declaration * > &translationUnit ) {
		DeclStats stats;
		acceptAll( translationUnit, stats );
		stats.print();
	}
	
} // namespace CodeTools

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
