//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// fstream.c --
//
// Author           : Peter A. Buhr
// Created On       : Wed May 27 17:56:53 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Mar  7 14:48:09 2017
// Update Count     : 192
//

#include "fstream"

extern "C" {
#include <stdio.h>										// vfprintf, vfscanf
#include <stdlib.h>										// exit
#include <stdarg.h>										// varargs
#include <string.h>										// strlen
#include <stdbool.h>									// true/false
#include <float.h>										// DBL_DIG, LDBL_DIG
#include <complex.h>									// creal, cimag
}

#define IO_MSG "I/O error: "

_Bool sepPrt( ofstream * os ) { return os->sepOnOff; }
void sepOn( ofstream * os ) { os->sepOnOff = 1; }
void sepOff( ofstream * os ) { os->sepOnOff = 0; }
void sepReset( ofstream * os ) { os->sepOnOff = os->sepDefault; }
void sepReset( ofstream * os, _Bool reset ) { os->sepDefault = reset; os->sepOnOff = os->sepDefault; }
const char * sepGet( ofstream * os ) { return &(os->separator[0]); }

void sepSet( ofstream * os, const char * s ) {
	strncpy( &(os->separator[0]), s, separateSize - 1 );
	os->separator[separateSize - 1] = '\0';
} // sepSet

_Bool sepDisable( ofstream *os ) {
	_Bool temp = os->sepDefault;
	os->sepDefault = false;
	sepReset( os );
	return temp;
} // sepDisable

_Bool sepEnable( ofstream *os ) {
	_Bool temp = os->sepDefault;
	os->sepDefault = true;
	sepReset( os );
	return temp;
} // sepEnable

int fail( ofstream * os ) {
	return ferror( (FILE *)(os->file) );
} // fail

int flush( ofstream * os ) {
	return fflush( (FILE *)(os->file) );
} // flush

void open( ofstream * os, const char * name, const char * mode ) {
	FILE *file = fopen( name, mode );
	if ( file == 0 ) {									// do not change unless successful
		fprintf( stderr, IO_MSG "open output file \"%s\", ", name );
		perror( 0 );
		exit( EXIT_FAILURE );
	} // if
	os->file = file;
	sepOff( os );
	sepSet( os, " " );
} // open

void close( ofstream * os ) {
	if ( (FILE *)(os->file) == stdout || (FILE *)(os->file) == stderr ) return;

	if ( fclose( (FILE *)(os->file) ) == EOF ) {
		perror( IO_MSG "close output" );
	} // if
} // close

ofstream * write( ofstream * os, const char * data, unsigned long int size ) {
	if ( fail( os ) ) {
		fprintf( stderr, "attempt write I/O on failed stream\n" );
		exit( EXIT_FAILURE );
	} // if

	if ( fwrite( data, 1, size, (FILE *)(os->file) ) != size ) {
		perror( IO_MSG "write" );
		exit( EXIT_FAILURE );
	} // if
	return os;
} // write

int prtfmt( ofstream * os, const char fmt[], ... ) {
	va_list args;
	va_start( args, fmt );
	int len = vfprintf( (FILE *)(os->file), fmt, args );
	if ( len == EOF ) {
		if ( ferror( (FILE *)(os->file) ) ) {
			fprintf( stderr, "invalid write\n" );
			exit( EXIT_FAILURE );
		} // if
	} // if
	va_end( args );

	sepReset( os );										// reset separator
	return len;
} // prtfmt


static ofstream soutFile = { (FILE *)(&_IO_2_1_stdout_), 1, 0, { ' ', '\0' } };
ofstream *sout = &soutFile;
static ofstream serrFile = { (FILE *)(&_IO_2_1_stderr_), 1, 0, { ' ', '\0' } };
ofstream *serr = &serrFile;


//---------------------------------------


int fail( ifstream * is ) {
	return ferror( (FILE *)(is->file) );
} // fail

int eof( ifstream * is ) {
	return feof( (FILE *)(is->file) );
} // eof

void open( ifstream * is, const char * name, const char * mode ) {
	FILE *t = fopen( name, mode );
	if ( t == 0 ) {										// do not change unless successful
		fprintf( stderr, IO_MSG "open input file \"%s\", ", name );
		perror( 0 );
		exit( EXIT_FAILURE );
	} // if
	is->file = t;
} // open

void close( ifstream * is ) {
	if ( (FILE *)(is->file) == stdin ) return;

	if ( fclose( (FILE *)(is->file) ) == EOF ) {
		perror( IO_MSG "close input" );
	} // if
} // close

ifstream * read( ifstream * is, char * data, unsigned long int size ) {
	if ( fail( is ) ) {
		fprintf( stderr, "attempt read I/O on failed stream\n" );
		exit( EXIT_FAILURE );
	} // if

	if ( fread( data, size, 1, (FILE *)(is->file) ) == 0 ) {
		perror( IO_MSG "read" );
		exit( EXIT_FAILURE );
	} // if
	return is;
} // read

ifstream *ungetc( ifstream * is, char c ) {
	if ( fail( is ) ) {
		fprintf( stderr, "attempt ungetc I/O on failed stream\n" );
		exit( EXIT_FAILURE );
	} // if

	if ( ungetc( c, (FILE *)(is->file) ) == EOF ) {
		perror( IO_MSG "ungetc" );
		exit( EXIT_FAILURE );
	} // if
	return is;
} // ungetc

int scanfmt( ifstream * is, const char fmt[], ... ) {
	va_list args;

	va_start( args, fmt );
	int len = vfscanf( (FILE *)(is->file), fmt, args );
	if ( len == EOF ) {
		if ( ferror( (FILE *)(is->file) ) ) {
			fprintf( stderr, "invalid read\n" );
			exit( EXIT_FAILURE );
		} // if
	} // if
	va_end( args );
	return len;
} // prtfmt


static ifstream sinFile = { (FILE *)(&_IO_2_1_stdin_) };
ifstream *sin = &sinFile;

// Local Variables: //
// tab-width: 4 //
// End: //
