//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CopyParams.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue May 19 07:33:31 2015
// Update Count     : 1
//

#include <cassert>                 // for assert
#include <list>                    // for list, _List_iterator, _List_const_...
#include <map>                     // for map, _Rb_tree_const_iterator, map<...
#include <set>                     // for set, set<>::const_iterator
#include <string>                  // for string, operator==
#include <utility>                 // for pair

#include "Common/SemanticError.h"  // for SemanticError
#include "Common/UniqueName.h"     // for UniqueName
#include "SynTree/Declaration.h"   // for DeclarationWithType, TypeDecl, Fun...
#include "SynTree/Expression.h"    // for VariableExpr, ApplicationExpr, Add...
#include "SynTree/Label.h"         // for Label, noLabels
#include "SynTree/Statement.h"     // for CompoundStmt, DeclStmt, ExprStmt
#include "SynTree/SynTree.h"       // for UniqueId
#include "SynTree/Type.h"          // for FunctionType, TypeInstType, Type
#include "SynTree/Visitor.h"       // for acceptAll, Visitor

namespace GenPoly {
	class CopyParams : public Visitor {
	  public:
		CopyParams();

		virtual void visit( FunctionDecl *funcDecl );
		virtual void visit( AddressExpr *addrExpr );

	  private:
		std::set< UniqueId > modVars;
		UniqueName namer;
	};

	/// creates local copies of polymorphic function parameters
	void copyParams( std::list< Declaration* > &translationUnit ) {
		CopyParams copier;
		acceptAll( translationUnit, copier );
	}

	CopyParams::CopyParams() : namer( "_cp" ) {}

	void CopyParams::visit( FunctionDecl *funcDecl ) {
		if ( funcDecl->statements ) {
			funcDecl->statements->accept( *this );

			if ( ! modVars.empty() ) {
				std::map< std::string, DeclarationWithType* > assignOps;
				// xxx - this needs to use constructors, not assignment operators
				// assume the assignment operator is the first assert param after any "type" parameter
				for ( Type::ForallList::const_iterator tyVar = funcDecl->type->forall.begin(); tyVar != funcDecl->type->forall.end(); ++tyVar ) {
					if ( (*tyVar)->get_kind() == TypeDecl::Any ) {
						assert( !(*tyVar)->assertions.empty() );
						assert( (*tyVar)->assertions.front()->name == "?=?" );
						assignOps[ (*tyVar)->name ] = (*tyVar)->assertions.front();
					} // if
				} // for
				for ( std::list< DeclarationWithType* >::iterator param = funcDecl->type->parameters.begin(); param != funcDecl->type->parameters.end(); ++param ) {
					std::set< UniqueId >::const_iterator var = modVars.find( (*param)->get_uniqueId() );
					if ( var != modVars.end() ) {
						TypeInstType *typeInst = dynamic_cast< TypeInstType* >( (*param)->get_type() );
						assertf( typeInst, "Found in modVars for function %s: unique id: %u, %s", funcDecl->name.c_str(), (*param)->get_uniqueId(), toString( *param ).c_str() );
						std::map< std::string, DeclarationWithType* >::const_iterator assignOp = assignOps.find( typeInst->name );
						if ( assignOp != assignOps.end() ) {
							DeclarationWithType *oldParam = *param;
							*param = (*param)->clone();
							(*param)->set_mangleName( namer.newName( (*param)->get_mangleName() ) );
							ApplicationExpr *assign = new ApplicationExpr( new VariableExpr( assignOp->second ) );
							assign->args.push_back( new VariableExpr( oldParam ) );
							assign->args.push_back( new VariableExpr( *param ) );
							funcDecl->statements->push_front( new ExprStmt( noLabels, assign ) );
							funcDecl->statements->push_front( new DeclStmt( noLabels, oldParam ) );
						} // if
						modVars.erase( var );
					} // if
				} // for
			} // if
		} // if
	}

	// this test is insufficient because it is possible for values to be modified by being passed to other polymorphic
	// routines (e.g., assignment operators) without having their addresses explicitly taken. Some thought is needed to
	// make sure that all of the correct cases are identified where copies are necessary.
	//
	// As a temporary measure, for correctness at the expense of performance, ignore the modVars list entirely and copy
	// every parameter of TypeInstType* when visiting the FunctionDecl.
	void CopyParams::visit( AddressExpr *addrExpr ) {
		if ( VariableExpr *varExpr = dynamic_cast< VariableExpr* >( addrExpr->arg ) ) {
			if ( dynamic_cast< TypeInstType* >( varExpr->var->get_type() ) ) {
				assertf( varExpr->var->get_uniqueId(), "Address-taken variable does not have a unique ID: %s", toString( varExpr->var ).c_str() );
				modVars.insert( varExpr->var->get_uniqueId() );
				std::cerr << "added unique id: " << varExpr->var->get_uniqueId() << " var: " << varExpr->var << std::endl;
			} // if
		} // if
	}
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
