//
// Cforall Version 1.0.0 Copyright (C) 2019 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Convert.cpp -- Convert between the new and old syntax trees.
//
// Author           : Thierry Delisle
// Created On       : Thu May 09 15::37::05 2019
// Last Modified By : Andrew Beach
// Last Modified On : Tue May 21 15:30:00 2019
// Update Count     : 5
//

#include "Convert.hpp"

#include <unordered_map>

#include "AST/Attribute.hpp"
#include "AST/Decl.hpp"
#include "AST/Expr.hpp"
#include "AST/Init.hpp"
#include "AST/Stmt.hpp"
#include "AST/TypeSubstitution.hpp"

#include "SynTree/Attribute.h"
#include "SynTree/Declaration.h"
#include "SynTree/TypeSubstitution.h"

//================================================================================================
// Utilities
template<template <class...> class C>
struct to {
	template<typename T>
	static auto from( T && v ) -> C< typename T::value_type > {
		C< typename T::value_type > l;
		std::move(std::begin(v), std::end(v), std::back_inserter(l));
		return l;
	}
};

//================================================================================================
class ConverterNewToOld : public ast::Visitor {
	BaseSyntaxNode * node = nullptr;
	std::unordered_map< ast::Node *, BaseSyntaxNode * > cache;

	template<typename T>
	struct Getter {
		ConverterNewToOld & visitor;

		template<typename U, enum ast::Node::ref_type R>
		T * accept1( const ast::ptr_base<U, R> & ptr ) {
			if ( ! ptr ) return nullptr;
			ptr->accept( visitor );
			T * ret = strict_dynamic_cast< T * >( visitor.node );
			visitor.node = nullptr;
			return ret;
		}

		template<typename U>
		std::list< T * > acceptL( const U & container ) {
			std::list< T * > ret;
			for (auto ptr : container ) {
				ret.emplace_back( accept1( ptr ) );
			}
			return ret;
		}
	};

    template<typename T>
    Getter<T> get() {
        return Getter<T>{ *this };
    }

	Label makeLabel(Statement * labelled, const ast::Label& label) {
		return Label(
			label.name,
			labelled,
			get<Attribute>().acceptL(label.attributes)
		);
	}

	template<template <class...> class C>
	std::list<Label> makeLabelL(Statement * labelled, const C<ast::Label>& labels) {
		std::list<Label> ret;
		for (auto label : labels) {
			ret.push_back( makeLabel(labelled, label) );
		}
		return ret;
	}

	/// get new qualifiers from old type
	Type::Qualifiers cv( const ast::Type * ty ) { return { ty->qualifiers.val }; }

	template<typename NewT, typename OldT>
	NewT * cached( const OldT & old ) {
		auto it = cache.find( old.get() );
		if ( it == cache.end() ) {
			// doesn't update cache, that should be handled by the accept function
			return get< NewT >().accept1( old );
		} else {
			return strict_dynamic_cast< NewT * >( it->second );
		}
	}

public:
	Declaration * decl( const ast::Decl * declNode ) {
		return get<Declaration>().accept1( ast::ptr<ast::Decl>( declNode ) );
	}

private:
	void declPostamble( Declaration * decl, const ast::Decl * node ) {
		decl->location = node->location;
		// name comes from constructor
		// linkage comes from constructor
		decl->extension = node->extension;
		decl->uniqueId = node->uniqueId;
		// storageClasses comes from constructor
		this->node = decl;
	}

	const ast::DeclWithType * declWithTypePostamble (
			DeclarationWithType * decl, const ast::DeclWithType * node ) {
		declPostamble( decl, node );
		decl->mangleName = node->mangleName;
		decl->scopeLevel = node->scopeLevel;
		decl->asmName = get<Expression>().accept1( node->asmName );
		// attributes comes from constructor
		decl->isDeleted = node->isDeleted;
		// fs comes from constructor
		return nullptr;
	}

	const ast::DeclWithType * visit( const ast::ObjectDecl * node ) override final {
		auto decl = new ObjectDecl(
			node->name,
			Type::StorageClasses( node->storage.val ),
			LinkageSpec::Spec( node->linkage.val ),
			get<Expression>().accept1( node->bitfieldWidth ),
			get<Type>().accept1( node->type ),
			get<Initializer>().accept1( node->init ),
			get<Attribute>().acceptL( node->attributes ),
			Type::FuncSpecifiers( node->funcSpec.val )
		);
		return declWithTypePostamble( decl, node );
	}

	const ast::DeclWithType * visit( const ast::FunctionDecl * node ) override final {
		auto decl = new FunctionDecl(
			node->name,
			Type::StorageClasses( node->storage.val ),
			LinkageSpec::Spec( node->linkage.val ),
			get<FunctionType>().accept1( node->type ),
			get<CompoundStmt>().accept1( node->stmts ),
			get<Attribute>().acceptL( node->attributes ),
			Type::FuncSpecifiers( node->funcSpec.val )
		);
		decl->withExprs = get<Expression>().acceptL( node->withExprs );
		return declWithTypePostamble( decl, node );
	}

	// NamedTypeDecl
	const ast::Decl * namedTypePostamble( NamedTypeDecl * decl, const ast::NamedTypeDecl * node ) {
		declPostamble( decl, node );
		// base comes from constructor
		decl->parameters = get<TypeDecl>().acceptL( node->params );
		decl->assertions = get<DeclarationWithType>().acceptL( node->assertions );
		return nullptr;
	}

	const ast::Decl * visit( const ast::TypeDecl * node ) override final {
		TypeDecl::Kind kind;
		switch (node->kind) {
		case ast::TypeVar::Dtype:
			kind = TypeDecl::Dtype;
			break;
		case ast::TypeVar::Ftype:
			kind = TypeDecl::Ftype;
			break;
		case ast::TypeVar::Ttype:
			kind = TypeDecl::Ttype;
			break;
		default:
			assertf(false, "Invalid ast::TypeVar::Kind: %d\n", node->kind);
		};
		auto decl = new TypeDecl(
			node->name,
			Type::StorageClasses( node->storage.val ),
			get<Type>().accept1( node->base ),
			kind,
			node->sized,
			get<Type>().accept1( node->init )
		);
		return namedTypePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::TypedefDecl * node ) override final {
		auto decl = new TypedefDecl(
			node->name,
			node->location,
			Type::StorageClasses( node->storage.val ),
            get<Type>().accept1( node->base ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return namedTypePostamble( decl, node );
	}

	const ast::Decl * aggregatePostamble( AggregateDecl * decl, const ast::AggregateDecl * node ) {
		decl->members = get<Declaration>().acceptL( node->members );
		decl->parameters = get<TypeDecl>().acceptL( node->params );
		decl->body = node->body;
		// attributes come from constructor
		// TODO: Need caching for: decl->parent = node->parent;
		return nullptr;
	}

	const ast::Decl * visit( const ast::StructDecl * node ) override final {
		auto decl = new StructDecl(
			node->name,
			node->kind,
			get<Attribute>().acceptL( node->attributes ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::UnionDecl * node ) override final {
		auto decl = new UnionDecl(
			node->name,
			get<Attribute>().acceptL( node->attributes ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::EnumDecl * node ) override final {
		auto decl = new EnumDecl(
			node->name,
			get<Attribute>().acceptL( node->attributes ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::TraitDecl * node ) override final {
		auto decl = new TraitDecl(
			node->name,
			{},
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::AsmDecl * visit( const ast::AsmDecl * node ) override final {
		auto decl = new AsmDecl( get<AsmStmt>().accept1( node->stmt ) );
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::StaticAssertDecl * visit( const ast::StaticAssertDecl * node ) override final {
		auto decl = new StaticAssertDecl(
			get<Expression>().accept1( node->cond ),
			get<ConstantExpr>().accept1( node->msg )
		);
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::Stmt * stmtPostamble( Statement * stmt, const ast::Stmt * node ) {
		stmt->location = node->location;
		stmt->labels = makeLabelL( stmt, node->labels );
		this->node = stmt;
		return nullptr;
	}

	const ast::CompoundStmt * visit( const ast::CompoundStmt * node ) override final {
		auto stmt = new CompoundStmt( get<Statement>().acceptL( node->kids ) );
		stmtPostamble( stmt, node );
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ExprStmt * node ) override final {
		auto stmt = new ExprStmt( get<Expression>().accept1( node->expr ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::AsmStmt * node ) override final {
		auto stmt = new AsmStmt(
			node->isVolatile,
			get<Expression>().accept1( node->instruction ),
			get<Expression>().acceptL( node->output ),
			get<Expression>().acceptL( node->input ),
			get<ConstantExpr>().acceptL( node->clobber ),
			makeLabelL( nullptr, node->gotoLabels ) // What are these labelling?
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::DirectiveStmt * node ) override final {
		auto stmt = new DirectiveStmt( node->directive );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::IfStmt * node ) override final {
		auto stmt = new IfStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().accept1( node->thenPart ),
			get<Statement>().accept1( node->elsePart ),
			get<Statement>().acceptL( node->inits )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::SwitchStmt * node ) override final {
		auto stmt = new SwitchStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().acceptL( node->stmts )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::CaseStmt * node ) override final {
		auto stmt = new CaseStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().acceptL( node->stmts ),
			node->isDefault()
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::WhileStmt * node ) override final {
		auto inits = get<Statement>().acceptL( node->inits );
		auto stmt = new WhileStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().accept1( node->body ),
			inits,
			node->isDoWhile
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ForStmt * node ) override final {
		auto stmt = new ForStmt(
			get<Statement>().acceptL( node->inits ),
			get<Expression>().accept1( node->cond ),
			get<Expression>().accept1( node->inc ),
			get<Statement>().accept1( node->body )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::BranchStmt * node ) override final {
		BranchStmt * stmt;
		if (node->computedTarget) {
			stmt = new BranchStmt( get<Expression>().accept1( node->computedTarget ),
				BranchStmt::Goto );
		} else {
			BranchStmt::Type type;
			switch (node->kind) {
			#define CASE(n) \
			case ast::BranchStmt::n: \
				type = BranchStmt::n; \
				break
			CASE(Goto);
			CASE(Break);
			CASE(Continue);
			CASE(FallThrough);
			CASE(FallThroughDefault);
			#undef CASE
			default:
				assertf(false, "Invalid ast::BranchStmt::Kind: %d\n", node->kind);
			}

			// The labels here are also weird.
			stmt = new BranchStmt( makeLabel( nullptr, node->originalTarget ), type );
			stmt->target = makeLabel( stmt, node->target );
		}
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ReturnStmt * node ) override final {
		auto stmt = new ReturnStmt( get<Expression>().accept1( node->expr ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ThrowStmt * node ) override final {
		ThrowStmt::Kind kind;
		switch (node->kind) {
		case ast::ThrowStmt::Terminate:
			kind = ThrowStmt::Terminate;
			break;
		case ast::ThrowStmt::Resume:
			kind = ThrowStmt::Resume;
			break;
		default:
			assertf(false, "Invalid ast::ThrowStmt::Kind: %d\n", node->kind);
		}
		auto stmt = new ThrowStmt(
			kind,
			get<Expression>().accept1( node->expr ),
			get<Expression>().accept1( node->target )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::TryStmt * node ) override final {
		auto handlers = get<CatchStmt>().acceptL( node->handlers );
		auto stmt = new TryStmt(
			get<CompoundStmt>().accept1( node->body ),
			handlers,
			get<FinallyStmt>().accept1( node->finally )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::CatchStmt * node ) override final {
		CatchStmt::Kind kind;
		switch (node->kind) {
		case ast::CatchStmt::Terminate:
			kind = CatchStmt::Terminate;
			break;
		case ast::CatchStmt::Resume:
			kind = CatchStmt::Resume;
			break;
		default:
			assertf(false, "Invalid ast::CatchStmt::Kind: %d\n", node->kind);
		}
		auto stmt = new CatchStmt(
			kind,
			get<Declaration>().accept1( node->decl ),
			get<Expression>().accept1( node->cond ),
			get<Statement>().accept1( node->body )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::FinallyStmt * node ) override final {
		auto stmt = new FinallyStmt( get<CompoundStmt>().accept1( node->body ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::WaitForStmt * node ) override final {
		auto stmt = new WaitForStmt;
		stmt->clauses.reserve( node->clauses.size() );
		for ( auto clause : node->clauses ) {
			stmt->clauses.push_back({{
					get<Expression>().accept1( clause.target.func ),
					get<Expression>().acceptL( clause.target.args ),
				},
				get<Statement>().accept1( clause.stmt ),
				get<Expression>().accept1( clause.cond ),
			});
		}
		stmt->timeout = {
			get<Expression>().accept1( node->timeout.time ),
			get<Statement>().accept1( node->timeout.stmt ),
			get<Expression>().accept1( node->timeout.cond ),
		};
		stmt->orelse = {
			get<Statement>().accept1( node->orElse.stmt ),
			get<Expression>().accept1( node->orElse.cond ),
		};
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::WithStmt * node ) override final {
		auto stmt = new WithStmt(
			get<Expression>().acceptL( node->exprs ),
			get<Statement>().accept1( node->stmt )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::NullStmt * visit( const ast::NullStmt * node ) override final {
		auto stmt = new NullStmt();
		stmtPostamble( stmt, node );
		return nullptr;
	}

	const ast::Stmt * visit( const ast::DeclStmt * node ) override final {
		auto stmt = new DeclStmt( get<Declaration>().accept1( node->decl ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ImplicitCtorDtorStmt * node ) override final {
		(void)node;
		return nullptr;
	}

	TypeSubstitution * convertTypeSubstitution(const ast::TypeSubstitution * src) {

		TypeSubstitution *rslt = new TypeSubstitution();

		for (decltype(src->begin()) src_i = src->begin(); src_i != src->end(); src_i++) {
			rslt->add( src_i->first,
			           get<Type>().accept1(src_i->second) );
		}

		for (decltype(src->beginVar()) src_i = src->beginVar(); src_i != src->endVar(); src_i++) {
			rslt->addVar( src_i->first,
			              get<Expression>().accept1(src_i->second) );
		}

		return rslt;
	}

	void convertInferUnion(std::map<UniqueId,ParamEntry> &tgtInferParams,
						   std::vector<UniqueId>         &tgtResnSlots,
						   const ast::Expr::InferUnion   &srcInferred ) {

		assert( tgtInferParams.empty() );
		assert( tgtResnSlots.empty() );

		if ( srcInferred.mode == ast::Expr::InferUnion::Params ) {
			const ast::InferredParams &srcParams = srcInferred.inferParamsConst();
			for (auto srcParam : srcParams) {
				tgtInferParams[srcParam.first] = ParamEntry(
					srcParam.second.decl,
					get<Type>().accept1(srcParam.second.actualType),
					get<Type>().accept1(srcParam.second.formalType),
					get<Expression>().accept1(srcParam.second.expr)
				);
			}
		} else if ( srcInferred.mode == ast::Expr::InferUnion::Slots  ) {
			const ast::ResnSlots &srcSlots = srcInferred.resnSlotsConst();
			for (auto srcSlot : srcSlots) {
				tgtResnSlots.push_back(srcSlot);
			}
		}
	}

	Expression * visitBaseExpr(const ast::Expr * src, Expression * tgt) {

		tgt->location = src->location;

		tgt->result = get<Type>().accept1(src->result);
		tgt->env    = convertTypeSubstitution(src->env);

		tgt->extension = src->extension;
		convertInferUnion(tgt->inferParams, tgt->resnSlots, src->inferred);

		return tgt;
	}

	const ast::Expr * visit( const ast::ApplicationExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new ApplicationExpr(
				get<Expression>().accept1(node->func),
				get<Expression>().acceptL(node->args)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new UntypedExpr(
				get<Expression>().accept1(node->func),
				get<Expression>().acceptL(node->args)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::NameExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new NameExpr(
				node->name
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AddressExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::LabelAddressExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CastExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::KeywordCastExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::VirtualCastExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedMemberExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::MemberExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::VariableExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConstantExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::SizeofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AlignofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedOffsetofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::OffsetofExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::OffsetPackExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::LogicalExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConditionalExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CommaExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TypeExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AsmExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ImplicitCopyCtorExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConstructorExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CompoundLiteralExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::RangeExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedTupleExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleIndexExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleAssignExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::StmtExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UniqueExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedInitExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::InitExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::DeletedExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::DefaultArgExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Expr * visit( const ast::GenericExpr * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::VoidType * node ) override final {
		this->node = new VoidType{ cv( node ) };
		return nullptr;
	}

	const ast::Type * visit( const ast::BasicType * node ) override final {
		this->node = new BasicType{ cv( node ), (BasicType::Kind)(unsigned)node->kind };
		return nullptr;
	}

	const ast::Type * visit( const ast::PointerType * node ) override final {
		this->node = new PointerType{
			cv( node ),
			get<Type>().accept1( node->base ),
			get<Expression>().accept1( node->dimension ),
			node->isVarLen,
			node->isStatic
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::ArrayType * node ) override final {
		this->node = new ArrayType{
			cv( node ),
			get<Type>().accept1( node->base ),
			get<Expression>().accept1( node->dimension ),
			node->isVarLen,
			node->isStatic
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::ReferenceType * node ) override final {
		this->node = new ReferenceType{
			cv( node ),
			get<Type>().accept1( node->base )
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::QualifiedType * node ) override final {
		this->node = new QualifiedType{
			cv( node ),
			get<Type>().accept1( node->parent ),
			get<Type>().accept1( node->child )
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::FunctionType * node ) override final {
		auto ty = new FunctionType { cv( node ), node->isVarArgs };
		ty->returnVals = get<DeclarationWithType>().acceptL( node->returns );
		ty->parameters = get<DeclarationWithType>().acceptL( node->params );
		ty->forall = get<TypeDecl>().acceptL( node->forall );
		this->node = ty;
		return nullptr;
	}

	const ast::Type * visit( const ast::StructInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::UnionInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::EnumInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TraitInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TypeInstType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TupleType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::TypeofType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::VarArgsType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::ZeroType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::OneType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Type * visit( const ast::GlobalScopeType * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Designation * visit( const ast::Designation * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Init * visit( const ast::SingleInit * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Init * visit( const ast::ListInit * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Init * visit( const ast::ConstructorInit * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::Attribute * visit( const ast::Attribute * node ) override final {
		(void)node;
		return nullptr;
	}

	const ast::TypeSubstitution * visit( const ast::TypeSubstitution * node ) override final {
		(void)node;
		return nullptr;
	}
};

std::list< Declaration * > convert( std::list< ast::ptr< ast::Decl > > && translationUnit ) {
	ConverterNewToOld c;
	std::list< Declaration * > decls;
	for(auto d : translationUnit) {
		decls.emplace_back( c.decl( d ) );
		delete d;
	}
	return decls;
}

//================================================================================================

class ConverterOldToNew : public Visitor {
public:
	ast::Decl * decl() {
		return strict_dynamic_cast< ast::Decl * >( node );
	}
private:
	/// conversion output
	ast::Node * node;
	/// cache of nodes that might be referenced by readonly<> for de-duplication
	std::unordered_map< BaseSyntaxNode *, ast::Node * > cache;

	// Local Utilities:

	template<typename NewT, typename OldT>
	NewT * getAccept1( OldT old ) {
		if ( ! old ) return nullptr;
		old->accept(*this);
		return strict_dynamic_cast< NewT * >( node );
	}

#	define GET_ACCEPT_1(child, type) \
		getAccept1< ast::type, decltype( old->child ) >( old->child )

	template<typename NewT, typename OldC>
	std::vector< ast::ptr<NewT> > getAcceptV( OldC& old ) {
		std::vector< ast::ptr<NewT> > ret;
		ret.reserve( old.size() );
		for ( auto a : old ) {
			a->accept( *this );
			ret.emplace_back( strict_dynamic_cast< NewT * >(node) );
		}
		return ret;
	}

#	define GET_ACCEPT_V(child, type) \
		getAcceptV< ast::type, decltype( old->child ) >( old->child )

	ast::Label make_label(Label* old) {
		return ast::Label(
			old->labelled->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute)
		);
	}

	template<template <class...> class C>
	C<ast::Label> make_labels(C<Label> olds) {
		C<ast::Label> ret;
		for (auto oldn : olds) {
			ret.push_back( make_label( &oldn ) );
		}
		return ret;
	}

#	define GET_LABELS_V(labels) \
		to<std::vector>::from( make_labels( std::move( labels ) ) )
	
	static ast::CV::Qualifiers cv( Type * ty ) { return { ty->get_qualifiers().val }; }

	template<typename NewT, typename OldT>
	NewT * cached( OldT * old ) {
		auto it = cache.find( old );
		// doesn't update cache, that should be handled by the accept function
		ast::Node * nw = it == cache.end() ? getAccept1< NewT >( old ) : it->second;
		return strict_dynamic_cast< NewT * >( nw );
	}

	// Now all the visit functions:

	virtual void visit( ObjectDecl * old ) override final {
		auto decl = new ast::ObjectDecl(
			old->location,
			old->name,
			GET_ACCEPT_1(type, Type),
			GET_ACCEPT_1(init, Init),
			{ old->get_storageClasses().val },
			{ old->linkage.val },
			GET_ACCEPT_1(bitfieldWidth, Expr),
			GET_ACCEPT_V(attributes, Attribute),
			{ old->get_funcSpec().val }
		);
		decl->scopeLevel = old->scopeLevel;
		decl->mangleName = old->mangleName;
		decl->isDeleted  = old->isDeleted;
		decl->uniqueId   = old->uniqueId;
		decl->extension  = old->extension;

		this->node = decl;
	}

	virtual void visit( FunctionDecl * ) override final {

	}

	virtual void visit( StructDecl * old ) override final {
		auto decl = new ast::StructDecl(
			old->location,
			old->name,
			old->kind,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( UnionDecl * old ) override final {
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( EnumDecl * old ) override final {
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( TraitDecl * old ) override final {
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( TypeDecl * ) override final {

	}

	virtual void visit( TypedefDecl * old ) override final {
		auto decl = new ast::TypedefDecl(
			old->location,
			old->name,
			{ old->storageClasses.val },
			GET_ACCEPT_1(base, Type),
			{ old->linkage.val }
		);
		decl->assertions = GET_ACCEPT_V(assertions, DeclWithType);
		decl->params     = GET_ACCEPT_V(parameters, TypeDecl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( AsmDecl * ) override final {

	}

	virtual void visit( StaticAssertDecl * ) override final {

	}

	virtual void visit( CompoundStmt * old ) override final {
		auto stmt = new ast::CompoundStmt(
			old->location,
			to<std::list>::from( GET_ACCEPT_V(kids, Stmt) ),
			GET_LABELS_V(old->labels)
		);

		this->node = stmt;
	}

	virtual void visit( ExprStmt * old ) override final {
		this->node = new ast::ExprStmt(
			old->location,
			GET_ACCEPT_1(expr, Expr),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( AsmStmt * old ) override final {
		this->node = new ast::AsmStmt(
			old->location,
			old->voltile,
			GET_ACCEPT_1(instruction, Expr),
			GET_ACCEPT_V(output, Expr),
			GET_ACCEPT_V(input, Expr),
			GET_ACCEPT_V(clobber, ConstantExpr),
			GET_LABELS_V(old->gotolabels),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( DirectiveStmt * old ) override final {
		this->node = new ast::DirectiveStmt(
			old->location,
			old->directive,
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( IfStmt * old ) override final {
		this->node = new ast::IfStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(thenPart, Stmt),
			GET_ACCEPT_1(elsePart, Stmt),
			GET_ACCEPT_V(initialization, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( SwitchStmt * old ) override final {
		this->node = new ast::SwitchStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_V(statements, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( CaseStmt * old ) override final {
		this->node = new ast::CaseStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_V(stmts, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( WhileStmt * old ) override final {
		this->node = new ast::WhileStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_ACCEPT_V(initialization, Stmt),
			old->isDoWhile,
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( ForStmt * old ) override final {
		this->node = new ast::ForStmt(
			old->location,
			GET_ACCEPT_V(initialization, Stmt),
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(increment, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( BranchStmt * old ) override final {
		if (old->computedTarget) {
			this->node = new ast::BranchStmt(
				old->location,
				GET_ACCEPT_1(computedTarget, Expr),
				GET_LABELS_V(old->labels)
			);
		} else {
			ast::BranchStmt::Kind kind;
			switch (old->type) {
			#define CASE(n) \
			case BranchStmt::n: \
				kind = ast::BranchStmt::n; \
				break
			CASE(Goto);
			CASE(Break);
			CASE(Continue);
			CASE(FallThrough);
			CASE(FallThroughDefault);
			#undef CASE
			default:
				assertf(false, "Invalid BranchStmt::Type %d\n", old->type);
			}

			Label label = old->originalTarget;
			auto stmt = new ast::BranchStmt(
				old->location,
				kind,
				make_label(&label),
				GET_LABELS_V(old->labels)
			);
			stmt->target = make_label(&old->target);
			this->node = stmt;
		}
	}

	virtual void visit( ReturnStmt * old ) override final {
		this->node = new ast::ReturnStmt(
			old->location,
			GET_ACCEPT_1(expr, Expr),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( ThrowStmt * old ) override final {
		ast::ThrowStmt::Kind kind;
		switch (old->kind) {
		case ThrowStmt::Terminate:
			kind = ast::ThrowStmt::Terminate;
			break;
		case ThrowStmt::Resume:
			kind = ast::ThrowStmt::Resume;
			break;
		default:
			assertf(false, "Invalid ThrowStmt::Kind %d\n", old->kind);
		}

		this->node = new ast::ThrowStmt(
			old->location,
			kind,
			GET_ACCEPT_1(expr, Expr),
			GET_ACCEPT_1(target, Expr),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( TryStmt * old ) override final {
		this->node = new ast::TryStmt(
			old->location,
			GET_ACCEPT_1(block, CompoundStmt),
			GET_ACCEPT_V(handlers, CatchStmt),
			GET_ACCEPT_1(finallyBlock, FinallyStmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( CatchStmt * old ) override final {
		ast::CatchStmt::Kind kind;
		switch (old->kind) {
		case CatchStmt::Terminate:
			kind = ast::CatchStmt::Terminate;
			break;
		case CatchStmt::Resume:
			kind = ast::CatchStmt::Resume;
			break;
		default:
			assertf(false, "Invalid CatchStmt::Kind %d\n", old->kind);
		}

		this->node = new ast::CatchStmt(
			old->location,
			kind,
			GET_ACCEPT_1(decl, Decl),
			GET_ACCEPT_1(cond, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( FinallyStmt * old ) override final {
		this->node = new ast::FinallyStmt(
			old->location,
			GET_ACCEPT_1(block, CompoundStmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( WaitForStmt * old ) override final {
		ast::WaitForStmt * stmt = new ast::WaitForStmt(
			old->location,
			GET_LABELS_V(old->labels)
		);

		stmt->clauses.reserve( old->clauses.size() );
		for (size_t i = 0 ; i < old->clauses.size() ; ++i) {
			stmt->clauses.push_back({
				ast::WaitForStmt::Target{
					GET_ACCEPT_1(clauses[i].target.function, Expr),
					GET_ACCEPT_V(clauses[i].target.arguments, Expr)
				},
				GET_ACCEPT_1(clauses[i].statement, Stmt),
				GET_ACCEPT_1(clauses[i].condition, Expr)
			});
		}
		stmt->timeout = {
			GET_ACCEPT_1(timeout.time, Expr),
			GET_ACCEPT_1(timeout.statement, Stmt),
			GET_ACCEPT_1(timeout.condition, Expr),
		};
		stmt->orElse = {
			GET_ACCEPT_1(timeout.statement, Stmt),
			GET_ACCEPT_1(timeout.condition, Expr),
		};

		this->node = stmt;
	}

	virtual void visit( WithStmt * old ) override final {
		this->node = new ast::WithStmt(
			old->location,
			GET_ACCEPT_V(exprs, Expr),
			GET_ACCEPT_1(stmt, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( NullStmt * old ) override final {
		this->node = new ast::NullStmt(
			old->location,
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( DeclStmt * old ) override final {
		this->node = new ast::DeclStmt(
			old->location,
			GET_ACCEPT_1(decl, Decl),
			GET_LABELS_V(old->labels)
		);
	}

	virtual void visit( ImplicitCtorDtorStmt * old ) override final {
		this->node = new ast::ImplicitCtorDtorStmt(
			old->location,
			GET_ACCEPT_1(callStmt, Stmt),
			GET_LABELS_V(old->labels)
		);
	}

	ast::TypeSubstitution * convertTypeSubstitution(const TypeSubstitution * old) {

		ast::TypeSubstitution *rslt = new ast::TypeSubstitution();

		for (decltype(old->begin()) old_i = old->begin(); old_i != old->end(); old_i++) {
			rslt->add( old_i->first,
			           getAccept1<ast::Type>(old_i->second) );
		}

		for (decltype(old->beginVar()) old_i = old->beginVar(); old_i != old->endVar(); old_i++) {
			rslt->addVar( old_i->first,
			              getAccept1<ast::Expr>(old_i->second) );
		}

		return rslt;
	}

	void convertInferUnion(ast::Expr::InferUnion               &newInferred,
						   const std::map<UniqueId,ParamEntry> &oldInferParams,
						   const std::vector<UniqueId>         &oldResnSlots) {

		assert( oldInferParams.empty() || oldResnSlots.empty() );
		assert( newInferred.mode == ast::Expr::InferUnion::Empty );

		if ( !oldInferParams.empty() ) {
			ast::InferredParams &tgt = newInferred.inferParams();
			for (auto old : oldInferParams) {
				tgt[old.first] = ast::ParamEntry(
					old.second.decl,
					getAccept1<ast::Type>(old.second.actualType),
					getAccept1<ast::Type>(old.second.formalType),
					getAccept1<ast::Expr>(old.second.expr)
				);
			}
		} else if ( !oldResnSlots.empty() ) {
			ast::ResnSlots &tgt = newInferred.resnSlots();
			for (auto old : oldResnSlots) {
				tgt.push_back(old);
			}
		}
	}

	ast::Expr * visitBaseExpr(Expression * old, ast::Expr * nw) {

		nw->result = GET_ACCEPT_1(result, Type);
		nw->env    = convertTypeSubstitution(old->env);

		nw->extension = old->extension;
		convertInferUnion(nw->inferred, old->inferParams, old->resnSlots);

		return nw;
	}

	virtual void visit( ApplicationExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::ApplicationExpr(
				old->location,
				GET_ACCEPT_1(function, Expr),
				GET_ACCEPT_V(args, Expr)
			)
		);
	}

	virtual void visit( UntypedExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::UntypedExpr(
				old->location,
				GET_ACCEPT_1(function, Expr),
				GET_ACCEPT_V(args, Expr)
			)
		);
	}

	virtual void visit( NameExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::NameExpr(
				old->location,
				old->get_name()
			)
		);
	}

	virtual void visit( CastExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::CastExpr(
				old->location,
				nullptr, // cast's "to" type is expr's result type; converted in visitBaseExpr
				old->isGenerated ? ast::GeneratedCast : ast::ExplicitCast
			)
		);
	}

	virtual void visit( KeywordCastExpr * ) override final {

	}

	virtual void visit( VirtualCastExpr * ) override final {

	}

	virtual void visit( AddressExpr * ) override final {

	}

	virtual void visit( LabelAddressExpr * ) override final {

	}

	virtual void visit( UntypedMemberExpr * ) override final {

	}

	virtual void visit( MemberExpr * ) override final {

	}

	virtual void visit( VariableExpr * ) override final {

	}

	virtual void visit( ConstantExpr * ) override final {

	}

	virtual void visit( SizeofExpr * ) override final {

	}

	virtual void visit( AlignofExpr * ) override final {

	}

	virtual void visit( UntypedOffsetofExpr * ) override final {

	}

	virtual void visit( OffsetofExpr * ) override final {

	}

	virtual void visit( OffsetPackExpr * ) override final {

	}

	virtual void visit( LogicalExpr * ) override final {

	}

	virtual void visit( ConditionalExpr * ) override final {

	}

	virtual void visit( CommaExpr * ) override final {

	}

	virtual void visit( TypeExpr * ) override final {

	}

	virtual void visit( AsmExpr * ) override final {

	}

	virtual void visit( ImplicitCopyCtorExpr * ) override final {

	}

	virtual void visit( ConstructorExpr *  ) override final {

	}

	virtual void visit( CompoundLiteralExpr * ) override final {

	}

	virtual void visit( RangeExpr * ) override final {

	}

	virtual void visit( UntypedTupleExpr * ) override final {

	}

	virtual void visit( TupleExpr * ) override final {

	}

	virtual void visit( TupleIndexExpr * ) override final {

	}

	virtual void visit( TupleAssignExpr * ) override final {

	}

	virtual void visit( StmtExpr *  ) override final {

	}

	virtual void visit( UniqueExpr *  ) override final {

	}

	virtual void visit( UntypedInitExpr *  ) override final {

	}

	virtual void visit( InitExpr *  ) override final {

	}

	virtual void visit( DeletedExpr * ) override final {

	}

	virtual void visit( DefaultArgExpr * ) override final {

	}

	virtual void visit( GenericExpr * ) override final {

	}

	virtual void visit( VoidType * old ) override final {
		this->node = new ast::VoidType{ cv( old ) };
	}

	virtual void visit( BasicType * old ) override final {
		this->node = new ast::BasicType{ (ast::BasicType::Kind)(unsigned)old->kind, cv( old ) };
	}

	virtual void visit( PointerType * old ) override final {
		this->node = new ast::PointerType{
			GET_ACCEPT_1( base, Type ),
			GET_ACCEPT_1( dimension, Expr ),
			(ast::LengthFlag)old->isVarLen,
			(ast::DimensionFlag)old->isStatic,
			cv( old )
		};
	}

	virtual void visit( ArrayType * old ) override final {
		this->node = new ast::ArrayType{
			GET_ACCEPT_1( base, Type ),
			GET_ACCEPT_1( dimension, Expr ),
			(ast::LengthFlag)old->isVarLen,
			(ast::DimensionFlag)old->isStatic,
			cv( old )
		};
	}

	virtual void visit( ReferenceType * old ) override final {
		this->node = new ast::ReferenceType{
			GET_ACCEPT_1( base, Type ),
			cv( old )
		};
	}

	virtual void visit( QualifiedType * old ) override final {
		this->node = new ast::QualifiedType{
			GET_ACCEPT_1( parent, Type ),
			GET_ACCEPT_1( child, Type ),
			cv( old )
		};
	}

	virtual void visit( FunctionType * old ) override final {
		auto ty = new ast::FunctionType {
			(ast::ArgumentFlag)old->isVarArgs,
			cv( old )
		};
		ty->returns = GET_ACCEPT_V( returnVals, DeclWithType );
		ty->params = GET_ACCEPT_V( parameters, DeclWithType );
		ty->forall = GET_ACCEPT_V( forall, TypeDecl );
		this->node = ty;
	}

	void postvisit( ReferenceToType * old, ast::ReferenceToType * ty ) {
		ty->forall = GET_ACCEPT_V( forall, TypeDecl );
		ty->params = GET_ACCEPT_V( parameters, Expr );
		ty->hoistType = old->hoistType;
	}

	virtual void visit( StructInstType * old ) override final {
		auto ty = new ast::StructInstType{
			cached< ast::StructDecl >( old->baseStruct ),
			cv( old ),
			GET_ACCEPT_V( attributes, Attribute )
		};
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( UnionInstType * old ) override final {
		auto ty = new ast::UnionInstType{
			cached< ast::UnionDecl >( old->baseUnion ),
			cv( old ),
			GET_ACCEPT_V( attributes, Attribute )
		};
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( EnumInstType * old ) override final {
		auto ty = new ast::EnumInstType{
			cached< ast::EnumDecl >( old->baseEnum ),
			cv( old ),
			GET_ACCEPT_V( attributes, Attribute )
		};
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( TraitInstType * old ) override final {
		auto ty = new ast::TraitInstType{
			cached< ast::TraitDecl >( old->baseTrait ),
			cv( old ),
			GET_ACCEPT_V( attributes, Attribute )
		};
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( TypeInstType * old ) override final {
		ast::TypeInstType * ty;
		if ( old->baseType ) {
			ty = new ast::TypeInstType{
				old->name,
				cached< ast::TypeDecl >( old->baseType ),
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		} else {
			ty = new ast::TypeInstType{
				old->name,
				old->isFtype ? ast::TypeVar::Ftype : ast::TypeVar::Dtype,
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		}
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( TupleType * ) override final {

	}

	virtual void visit( TypeofType * ) override final {

	}

	virtual void visit( AttrType * ) override final {

	}

	virtual void visit( VarArgsType * ) override final {

	}

	virtual void visit( ZeroType * ) override final {

	}

	virtual void visit( OneType * ) override final {

	}

	virtual void visit( GlobalScopeType * ) override final {

	}

	virtual void visit( Designation * ) override final {

	}

	virtual void visit( SingleInit * ) override final {

	}

	virtual void visit( ListInit * ) override final {

	}

	virtual void visit( ConstructorInit * ) override final {

	}

	virtual void visit( Constant * ) override final {

	}

	virtual void visit( Attribute * ) override final {

	}

	virtual void visit( AttrExpr * ) override final {

		assert( 0 );
	}
};

#undef GET_LABELS_V
#undef GET_ACCEPT_V
#undef GET_ACCEPT_1

std::list< ast::ptr< ast::Decl > > convert( const std::list< Declaration * > && translationUnit ) {
	ConverterOldToNew c;
	std::list< ast::ptr< ast::Decl > > decls;
	for(auto d : translationUnit) {
		d->accept( c );
		decls.emplace_back( c.decl() );
		delete d;
	}
	return decls;
}
