#include "memory_pool.h"

#include <stdlib>

#include "object_header.h"

const size_t gc_pool_header_size = (size_t)(  &(((gc_memory_pool*)NULL)->start_p) );

void ctor(gc_memory_pool *const this, size_t size, gc_memory_pool* next, gc_memory_pool* mirror, uint8_t type)
{
	this->mirror = mirror;
	this->next = next;
	this->type_code = type;

	card_table_t* new = (card_table_t*)malloc(sizeof(card_table_t));
	this->cards = new;
	ctor(this->cards);

	this->end_p = ((uint8_t*)this) + size;
	this->free_p = this->start_p;

	check(gc_pool_of(this) == this);
	check(this->cards);
	gc_reset_pool(this);
}

void dtor(gc_memory_pool *const this)
{
	dtor(this->cards);
	free(this->cards);
}

void gc_reset_pool(gc_memory_pool *const this)
{
	this->free_p = this->start_p;
	#if _DEBUG
		memset(this->start_p, 0xCD, gc_pool_total_size(this));
	#endif

	check(this->cards);
	reset(this->cards);

	check(gc_pool_size_left(this) == gc_pool_total_size(this));
}

void* gc_pool_allocate(gc_memory_pool *const this, size_t size, bool zero)
{
	void* ret = this->free_p;

	this->free_p += size;

	if (zero) memset(ret, 0x00, size);

	check(this->cards);
	register_object(this->cards, ret);

	return ret;
}

void ctor(
		gc_pool_object_iterator* const this,
		struct gc_object_header* start_object
		#if _DEBUG
			, intptr_t pool_start
			, intptr_t pool_end
		#endif
	)
{
	this->object = start_object;
	#if _DEBUG
		this->lower_limit = pool_start;
		this->upper_limit = pool_end;
	#endif

	check( ((intptr_t)start_object) >= lower_limit );
	check( ((intptr_t)start_object) <= upper_limit );
}

gc_pool_object_iterator gc_pool_iterator_for(gc_memory_pool* const this, void* member)
{
	size_t card = card_of(member);
	intptr_t member_add = (intptr_t)member;
	void* start_obj;
	intptr_t start_obj_add;

	do
	{
		check(card < CARDS_COUNT);
		start_obj = object_at(this->cards, card);
		check(card != 0 || start_obj);
		card--;
		start_obj_add = (intptr_t)start_obj;
	}
	while(start_obj_add > member_add || start_obj_add != 0);

	check(start_obj);
	
	struct gc_object_header* start_obj_typed = (struct gc_object_header*)start_obj;

	gc_pool_object_iterator it;
	ctor( &it,
		start_obj_typed,
		#if _DEBUG
			, (intptr_t)this->start_p
			, (intptr_t)this->free_p
		#endif
	);
	return it;
}

bool ?!=?(const gc_pool_object_iterator lhs, const gc_pool_object_iterator rhs)
{
	return lhs.object != rhs.object;
}

gc_pool_object_iterator begin(gc_memory_pool* const this)
{
	struct gc_object_header* start_obj = (struct gc_object_header*)this->start_p;
	gc_pool_object_iterator it;
	ctor( &it,
		start_obj,
		#if _DEBUG
			, (intptr_t)this->start_p
			, (intptr_t)this->free_p
		#endif
	);
	return it;
}

gc_pool_object_iterator end(gc_memory_pool* const this)
{
	gc_pool_object_iterator it;
	ctor( &it,
		(struct gc_object_header*)this->free_p,
		#if _DEBUG
			, (intptr_t)this->start_p
			, (intptr_t)this->free_p
		#endif
	);
	return it;
}

gc_pool_object_iterator* ++?(gc_pool_object_iterator* it)
{
	struct gc_object_header* object = it->object;
	intptr_t next_ptr = ((intptr_t)object) + object->size;
	check(next_ptr > lower_limit);
	check(next_ptr <= upper_limit);

	struct gc_object_header* next_obj = ((struct gc_object_header*)next_ptr);
	check(next_ptr == upper_limit || is_valide(next_obj));

	it->object = next_obj;
	return it;
}

const struct gc_object_header* *?(const gc_pool_object_iterator it)
{
	return it.object;
}

struct gc_object_header* *?(gc_pool_object_iterator it)
{
	return it.object;
}
