//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// AlternativeFinder.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Sat May 16 23:52:08 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Jul  3 17:58:39 2015
// Update Count     : 22
//

#include <list>
#include <iterator>
#include <algorithm>
#include <functional>
#include <cassert>

#include "AlternativeFinder.h"
#include "Alternative.h"
#include "Cost.h"
#include "typeops.h"
#include "Unify.h"
#include "RenameVars.h"
#include "SynTree/Type.h"
#include "SynTree/Declaration.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Visitor.h"
#include "SymTab/Indexer.h"
#include "SymTab/Mangler.h"
#include "SynTree/TypeSubstitution.h"
#include "SymTab/Validate.h"
#include "Designators/Processor.h"
#include "Tuples/TupleAssignment.h"
#include "Tuples/NameMatcher.h"
#include "utility.h"

extern bool resolvep;
#define PRINT( text ) if ( resolvep ) { text } 
//#define DEBUG_COST

namespace ResolvExpr {
	Expression *resolveInVoidContext( Expression *expr, const SymTab::Indexer &indexer, TypeEnvironment &env ) {
		CastExpr *castToVoid = new CastExpr( expr );

		AlternativeFinder finder( indexer, env );
		finder.findWithAdjustment( castToVoid );

		// it's a property of the language that a cast expression has either 1 or 0 interpretations; if it has 0
		// interpretations, an exception has already been thrown.
		assert( finder.get_alternatives().size() == 1 );
		CastExpr *newExpr = dynamic_cast< CastExpr* >( finder.get_alternatives().front().expr );
		assert( newExpr );
		env = finder.get_alternatives().front().env;
		return newExpr->get_arg()->clone();
	}

	namespace {
		void printAlts( const AltList &list, std::ostream &os, int indent = 0 ) {
			for ( AltList::const_iterator i = list.begin(); i != list.end(); ++i ) {
				i->print( os, indent );
				os << std::endl;
			}
		}

		void makeExprList( const AltList &in, std::list< Expression* > &out ) {
			for ( AltList::const_iterator i = in.begin(); i != in.end(); ++i ) {
				out.push_back( i->expr->clone() );
			}
		}

		Cost sumCost( const AltList &in ) {
			Cost total;
			for ( AltList::const_iterator i = in.begin(); i != in.end(); ++i ) {
				total += i->cost;
			}
			return total;
		}

		struct PruneStruct {
			bool isAmbiguous;
			AltList::iterator candidate;
			PruneStruct() {}
			PruneStruct( AltList::iterator candidate ): isAmbiguous( false ), candidate( candidate ) {}
		};

		/// Prunes a list of alternatives down to those that have the minimum conversion cost for a given return type; skips ambiguous interpretations
		template< typename InputIterator, typename OutputIterator >
		void pruneAlternatives( InputIterator begin, InputIterator end, OutputIterator out, const SymTab::Indexer &indexer ) {
			// select the alternatives that have the minimum conversion cost for a particular set of result types
			std::map< std::string, PruneStruct > selected;
			for ( AltList::iterator candidate = begin; candidate != end; ++candidate ) {
				PruneStruct current( candidate );
				std::string mangleName;
				for ( std::list< Type* >::const_iterator retType = candidate->expr->get_results().begin(); retType != candidate->expr->get_results().end(); ++retType ) {
					Type *newType = (*retType)->clone();
					candidate->env.apply( newType );
					mangleName += SymTab::Mangler::mangle( newType );
					delete newType;
				}
				std::map< std::string, PruneStruct >::iterator mapPlace = selected.find( mangleName );
				if ( mapPlace != selected.end() ) {
					if ( candidate->cost < mapPlace->second.candidate->cost ) {
						PRINT(
							std::cout << "cost " << candidate->cost << " beats " << mapPlace->second.candidate->cost << std::endl;
						)
						selected[ mangleName ] = current;
					} else if ( candidate->cost == mapPlace->second.candidate->cost ) {
						PRINT(
							std::cout << "marking ambiguous" << std::endl;
						)
						mapPlace->second.isAmbiguous = true;
					}
				} else {
					selected[ mangleName ] = current;
				}
			}

			PRINT(
				std::cout << "there are " << selected.size() << " alternatives before elimination" << std::endl;
			)

			// accept the alternatives that were unambiguous
			for ( std::map< std::string, PruneStruct >::iterator target = selected.begin(); target != selected.end(); ++target ) {
				if ( ! target->second.isAmbiguous ) {
					Alternative &alt = *target->second.candidate;
					for ( std::list< Type* >::iterator result = alt.expr->get_results().begin(); result != alt.expr->get_results().end(); ++result ) {
						alt.env.applyFree( *result );
					}
					*out++ = alt;
				}
			}

		}

		template< typename InputIterator, typename OutputIterator >
		void findMinCost( InputIterator begin, InputIterator end, OutputIterator out ) {
			AltList alternatives;

			// select the alternatives that have the minimum parameter cost
			Cost minCost = Cost::infinity;
			for ( AltList::iterator i = begin; i != end; ++i ) {
				if ( i->cost < minCost ) {
					minCost = i->cost;
					i->cost = i->cvtCost;
					alternatives.clear();
					alternatives.push_back( *i );
				} else if ( i->cost == minCost ) {
					i->cost = i->cvtCost;
					alternatives.push_back( *i );
				}
			}
			std::copy( alternatives.begin(), alternatives.end(), out );
		}

		template< typename InputIterator >
		void simpleCombineEnvironments( InputIterator begin, InputIterator end, TypeEnvironment &result ) {
			while ( begin != end ) {
				result.simpleCombine( (*begin++).env );
			}
		}

		void renameTypes( Expression *expr ) {
			for ( std::list< Type* >::iterator i = expr->get_results().begin(); i != expr->get_results().end(); ++i ) {
				(*i)->accept( global_renamer );
			}
		}
	}

	template< typename InputIterator, typename OutputIterator >
	void AlternativeFinder::findSubExprs( InputIterator begin, InputIterator end, OutputIterator out ) {
		while ( begin != end ) {
			AlternativeFinder finder( indexer, env );
			finder.findWithAdjustment( *begin );
			// XXX  either this
			//Designators::fixDesignations( finder, (*begin++)->get_argName() );
			// or XXX this
			begin++;
			PRINT(
				std::cout << "findSubExprs" << std::endl;
				printAlts( finder.alternatives, std::cout );
			)
			*out++ = finder;
		}
	}

	AlternativeFinder::AlternativeFinder( const SymTab::Indexer &indexer, const TypeEnvironment &env )
		: indexer( indexer ), env( env ) {
	}

	void AlternativeFinder::find( Expression *expr, bool adjust ) {
		expr->accept( *this );
		if ( alternatives.empty() ) {
			throw SemanticError( "No reasonable alternatives for expression ", expr );
		}
		for ( AltList::iterator i = alternatives.begin(); i != alternatives.end(); ++i ) {
			if ( adjust ) {
				adjustExprTypeList( i->expr->get_results().begin(), i->expr->get_results().end(), i->env, indexer );
			}
		}
		PRINT(
			std::cout << "alternatives before prune:" << std::endl;
			printAlts( alternatives, std::cout );
		)
		AltList::iterator oldBegin = alternatives.begin();
		pruneAlternatives( alternatives.begin(), alternatives.end(), front_inserter( alternatives ), indexer );
		if ( alternatives.begin() == oldBegin ) {
			std::ostringstream stream;
			stream << "Can't choose between alternatives for expression ";
			expr->print( stream );
			stream << "Alternatives are:";
			AltList winners;
			findMinCost( alternatives.begin(), alternatives.end(), back_inserter( winners ) );
			printAlts( winners, stream, 8 );
			throw SemanticError( stream.str() );
		}
		alternatives.erase( oldBegin, alternatives.end() );
		PRINT(
			std::cout << "there are " << alternatives.size() << " alternatives after elimination" << std::endl;
		)
	}

	void AlternativeFinder::findWithAdjustment( Expression *expr ) {
		find( expr, true );
	}

	template< typename StructOrUnionType >
	void AlternativeFinder::addAggMembers( StructOrUnionType *aggInst, Expression *expr, const Cost &newCost, const std::string &name ) {
		std::list< Declaration* > members;
		aggInst->lookup( name, members );
		for ( std::list< Declaration* >::const_iterator i = members.begin(); i != members.end(); ++i ) {
			if ( DeclarationWithType *dwt = dynamic_cast< DeclarationWithType* >( *i ) ) {
				alternatives.push_back( Alternative( new MemberExpr( dwt->clone(), expr->clone() ), env, newCost ) );
				renameTypes( alternatives.back().expr );
			} else {
				assert( false );
			}
		}
	}

	void AlternativeFinder::visit( ApplicationExpr *applicationExpr ) {
		alternatives.push_back( Alternative( applicationExpr->clone(), env, Cost::zero ) );
	}

	Cost computeConversionCost( Alternative &alt, const SymTab::Indexer &indexer ) {
		ApplicationExpr *appExpr = dynamic_cast< ApplicationExpr* >( alt.expr );
		assert( appExpr );
		PointerType *pointer = dynamic_cast< PointerType* >( appExpr->get_function()->get_results().front() );
		assert( pointer );
		FunctionType *function = dynamic_cast< FunctionType* >( pointer->get_base() );
		assert( function );

		Cost convCost( 0, 0, 0 );
		std::list< DeclarationWithType* >& formals = function->get_parameters();
		std::list< DeclarationWithType* >::iterator formal = formals.begin();
		std::list< Expression* >& actuals = appExpr->get_args();
		for ( std::list< Expression* >::iterator actualExpr = actuals.begin(); actualExpr != actuals.end(); ++actualExpr ) {
			PRINT(
				std::cout << "actual expression:" << std::endl;
				(*actualExpr)->print( std::cout, 8 );
				std::cout << "--- results are" << std::endl;
				printAll( (*actualExpr)->get_results(), std::cout, 8 );
			)
			std::list< DeclarationWithType* >::iterator startFormal = formal;
			Cost actualCost;
			for ( std::list< Type* >::iterator actual = (*actualExpr)->get_results().begin(); actual != (*actualExpr)->get_results().end(); ++actual ) {
				if ( formal == formals.end() ) {
					if ( function->get_isVarArgs() ) {
						convCost += Cost( 1, 0, 0 );
						break;
					} else {
						return Cost::infinity;
					}
				}
				PRINT(
					std::cout << std::endl << "converting ";
					(*actual)->print( std::cout, 8 );
					std::cout << std::endl << " to ";
					(*formal)->get_type()->print( std::cout, 8 );
				)
				Cost newCost = conversionCost( *actual, (*formal)->get_type(), indexer, alt.env );
				PRINT(
					std::cout << std::endl << "cost is" << newCost << std::endl;
				)

				if ( newCost == Cost::infinity ) {
					return newCost;
				}
				convCost += newCost;
				actualCost += newCost;

				convCost += Cost( 0, polyCost( (*formal)->get_type(), alt.env, indexer ) + polyCost( *actual, alt.env, indexer ), 0 );

				formal++;
			}
			if ( actualCost != Cost( 0, 0, 0 ) ) {
				std::list< DeclarationWithType* >::iterator startFormalPlusOne = startFormal;
				startFormalPlusOne++;
				if ( formal == startFormalPlusOne ) {
					// not a tuple type
					Type *newType = (*startFormal)->get_type()->clone();
					alt.env.apply( newType );
					*actualExpr = new CastExpr( *actualExpr, newType );
				} else {
					TupleType *newType = new TupleType( Type::Qualifiers() );
					for ( std::list< DeclarationWithType* >::iterator i = startFormal; i != formal; ++i ) {
						newType->get_types().push_back( (*i)->get_type()->clone() );
					}
					alt.env.apply( newType );
					*actualExpr = new CastExpr( *actualExpr, newType );
				}
			}

		}
		if ( formal != formals.end() ) {
			return Cost::infinity;
		}

		for ( InferredParams::const_iterator assert = appExpr->get_inferParams().begin(); assert != appExpr->get_inferParams().end(); ++assert ) {
			PRINT(
				std::cout << std::endl << "converting ";
				assert->second.actualType->print( std::cout, 8 );
				std::cout << std::endl << " to ";
				assert->second.formalType->print( std::cout, 8 );
				)
				Cost newCost = conversionCost( assert->second.actualType, assert->second.formalType, indexer, alt.env );
			PRINT(
				std::cout << std::endl << "cost of conversion is " << newCost << std::endl;
				)
				if ( newCost == Cost::infinity ) {
					return newCost;
				}
			convCost += newCost;

			convCost += Cost( 0, polyCost( assert->second.formalType, alt.env, indexer ) + polyCost( assert->second.actualType, alt.env, indexer ), 0 );
		}

		return convCost;
	}

	/// Adds type variables to the open variable set and marks their assertions
	void makeUnifiableVars( Type *type, OpenVarSet &unifiableVars, AssertionSet &needAssertions ) {
		for ( std::list< TypeDecl* >::const_iterator tyvar = type->get_forall().begin(); tyvar != type->get_forall().end(); ++tyvar ) {
			unifiableVars[ (*tyvar)->get_name() ] = (*tyvar)->get_kind();
			for ( std::list< DeclarationWithType* >::iterator assert = (*tyvar)->get_assertions().begin(); assert != (*tyvar)->get_assertions().end(); ++assert ) {
				needAssertions[ *assert ] = true;
			}
///     needAssertions.insert( needAssertions.end(), (*tyvar)->get_assertions().begin(), (*tyvar)->get_assertions().end() );
		}
	}

	bool AlternativeFinder::instantiateFunction( std::list< DeclarationWithType* >& formals, /*const*/ AltList &actuals, bool isVarArgs, OpenVarSet& openVars, TypeEnvironment &resultEnv, AssertionSet &resultNeed, AssertionSet &resultHave ) {
		simpleCombineEnvironments( actuals.begin(), actuals.end(), resultEnv );
		// make sure we don't widen any existing bindings
		for ( TypeEnvironment::iterator i = resultEnv.begin(); i != resultEnv.end(); ++i ) {
			i->allowWidening = false;
		}
		resultEnv.extractOpenVars( openVars );

		/*
		  Tuples::NameMatcher matcher( formals );
		  try {
		  matcher.match( actuals );
		  } catch ( Tuples::NoMatch &e ) {
		  std::cerr << "Alternative doesn't match: " << e.message << std::endl;
		  }
		*/
		std::list< DeclarationWithType* >::iterator formal = formals.begin();
		for ( AltList::const_iterator actualExpr = actuals.begin(); actualExpr != actuals.end(); ++actualExpr ) {
			for ( std::list< Type* >::iterator actual = actualExpr->expr->get_results().begin(); actual != actualExpr->expr->get_results().end(); ++actual ) {
				if ( formal == formals.end() ) {
					return isVarArgs;
				}
				PRINT(
					std::cerr << "formal type is ";
					(*formal)->get_type()->print( std::cerr );
					std::cerr << std::endl << "actual type is ";
					(*actual)->print( std::cerr );
					std::cerr << std::endl;
				)
				if ( ! unify( (*formal)->get_type(), *actual, resultEnv, resultNeed, resultHave, openVars, indexer ) ) {
					return false;
				}
				formal++;
			}
		}
		// Handling of default values
		while ( formal != formals.end() ) {
			if ( ObjectDecl *od = dynamic_cast<ObjectDecl *>( *formal ) )
				if ( SingleInit *si = dynamic_cast<SingleInit *>( od->get_init() ))
					// so far, only constant expressions are accepted as default values
					if ( ConstantExpr *cnstexpr = dynamic_cast<ConstantExpr *>( si->get_value()) )
						if ( Constant *cnst = dynamic_cast<Constant *>( cnstexpr->get_constant() ) )
							if ( unify( (*formal)->get_type(), cnst->get_type(), resultEnv, resultNeed, resultHave, openVars, indexer ) ) {
								// XXX Don't know if this is right
								actuals.push_back( Alternative( cnstexpr->clone(), env, Cost::zero ) );
								formal++;
								if ( formal == formals.end()) break;
							}
			return false;
		}
		return true;
	}

	static const int recursionLimit = 10;

	void addToIndexer( AssertionSet &assertSet, SymTab::Indexer &indexer ) {
		for ( AssertionSet::iterator i = assertSet.begin(); i != assertSet.end(); ++i ) {
			if ( i->second == true ) {
				i->first->accept( indexer );
			}
		}
	}

	template< typename ForwardIterator, typename OutputIterator >
	void inferRecursive( ForwardIterator begin, ForwardIterator end, const Alternative &newAlt, OpenVarSet &openVars, const SymTab::Indexer &decls, const AssertionSet &newNeed, int level, const SymTab::Indexer &indexer, OutputIterator out ) {
		if ( begin == end ) {
			if ( newNeed.empty() ) {
				*out++ = newAlt;
				return;
			} else if ( level >= recursionLimit ) {
				throw SemanticError( "Too many recursive assertions" );
			} else {
				AssertionSet newerNeed;
				PRINT(
					std::cerr << "recursing with new set:" << std::endl;
					printAssertionSet( newNeed, std::cerr, 8 );
				)
				inferRecursive( newNeed.begin(), newNeed.end(), newAlt, openVars, decls, newerNeed, level+1, indexer, out );
				return;
			}
		}

		ForwardIterator cur = begin++;
		if ( ! cur->second ) {
			inferRecursive( begin, end, newAlt, openVars, decls, newNeed, level, indexer, out );
		}
		DeclarationWithType *curDecl = cur->first;
		PRINT(
			std::cerr << "inferRecursive: assertion is ";
			curDecl->print( std::cerr );
			std::cerr << std::endl;
		)
		std::list< DeclarationWithType* > candidates;
		decls.lookupId( curDecl->get_name(), candidates );
///   if ( candidates.empty() ) { std::cout << "no candidates!" << std::endl; }
		for ( std::list< DeclarationWithType* >::const_iterator candidate = candidates.begin(); candidate != candidates.end(); ++candidate ) {
			PRINT(
				std::cout << "inferRecursive: candidate is ";
				(*candidate)->print( std::cout );
				std::cout << std::endl;
			)
			AssertionSet newHave, newerNeed( newNeed );
			TypeEnvironment newEnv( newAlt.env );
			OpenVarSet newOpenVars( openVars );
			Type *adjType = (*candidate)->get_type()->clone();
			adjustExprType( adjType, newEnv, indexer );
			adjType->accept( global_renamer );
			PRINT(
				std::cerr << "unifying ";
				curDecl->get_type()->print( std::cerr );
				std::cerr << " with ";
				adjType->print( std::cerr );
				std::cerr << std::endl;
			)
			if ( unify( curDecl->get_type(), adjType, newEnv, newerNeed, newHave, newOpenVars, indexer ) ) {
				PRINT(
					std::cerr << "success!" << std::endl;
				)
				SymTab::Indexer newDecls( decls );
				addToIndexer( newHave, newDecls );
				Alternative newerAlt( newAlt );
				newerAlt.env = newEnv;
				assert( (*candidate)->get_uniqueId() );
				Expression *varExpr = new VariableExpr( static_cast< DeclarationWithType* >( Declaration::declFromId( (*candidate)->get_uniqueId() ) ) );
				deleteAll( varExpr->get_results() );
				varExpr->get_results().clear();
				varExpr->get_results().push_front( adjType->clone() );
				PRINT(
					std::cout << "satisfying assertion " << curDecl->get_uniqueId() << " ";
					curDecl->print( std::cout );
					std::cout << " with declaration " << (*candidate)->get_uniqueId() << " ";
					(*candidate)->print( std::cout );
					std::cout << std::endl;
				)
				ApplicationExpr *appExpr = static_cast< ApplicationExpr* >( newerAlt.expr );
				// XXX: this is a memory leak, but adjType can't be deleted because it might contain assertions
				appExpr->get_inferParams()[ curDecl->get_uniqueId() ] = ParamEntry( (*candidate)->get_uniqueId(), adjType->clone(), curDecl->get_type()->clone(), varExpr );
				inferRecursive( begin, end, newerAlt, newOpenVars, newDecls, newerNeed, level, indexer, out );
			} else {
				delete adjType;
			}
		}
	}

	template< typename OutputIterator >
	void AlternativeFinder::inferParameters( const AssertionSet &need, AssertionSet &have, const Alternative &newAlt, OpenVarSet &openVars, OutputIterator out ) {
//	PRINT(
//	    std::cout << "inferParameters: assertions needed are" << std::endl;
//	    printAll( need, std::cout, 8 );
//	    )
		SymTab::Indexer decls( indexer );
		PRINT(
			std::cout << "============= original indexer" << std::endl;
			indexer.print( std::cout );
			std::cout << "============= new indexer" << std::endl;
			decls.print( std::cout );
		)
		addToIndexer( have, decls );
		AssertionSet newNeed;
		inferRecursive( need.begin(), need.end(), newAlt, openVars, decls, newNeed, 0, indexer, out );
//	PRINT(
//	    std::cout << "declaration 14 is ";
//	    Declaration::declFromId
//	    *out++ = newAlt;
//	    )
	}

	template< typename OutputIterator >
	void AlternativeFinder::makeFunctionAlternatives( const Alternative &func, FunctionType *funcType, AltList &actualAlt, OutputIterator out ) {
		OpenVarSet openVars;
		AssertionSet resultNeed, resultHave;
		TypeEnvironment resultEnv;
		makeUnifiableVars( funcType, openVars, resultNeed );
		if ( instantiateFunction( funcType->get_parameters(), actualAlt, funcType->get_isVarArgs(), openVars, resultEnv, resultNeed, resultHave ) ) {
			ApplicationExpr *appExpr = new ApplicationExpr( func.expr->clone() );
			Alternative newAlt( appExpr, resultEnv, sumCost( actualAlt ) );
			makeExprList( actualAlt, appExpr->get_args() );
			PRINT(
				std::cout << "need assertions:" << std::endl;
				printAssertionSet( resultNeed, std::cout, 8 );
			)
			inferParameters( resultNeed, resultHave, newAlt, openVars, out );
		}
	}

	void AlternativeFinder::visit( UntypedExpr *untypedExpr ) {
		bool doneInit = false;
		AlternativeFinder funcOpFinder( indexer, env );

		AlternativeFinder funcFinder( indexer, env ); {
			NameExpr *fname = 0;;
			if ( ( fname = dynamic_cast<NameExpr *>( untypedExpr->get_function()))
				 && ( fname->get_name() == std::string("&&")) ) {
				VoidType v = Type::Qualifiers();		// resolve to type void *
				PointerType pt( Type::Qualifiers(), v.clone() );
				UntypedExpr *vexpr = untypedExpr->clone();
				vexpr->get_results().push_front( pt.clone() );
				alternatives.push_back( Alternative( vexpr, env, Cost()) );
				return;
			}
		}

		funcFinder.findWithAdjustment( untypedExpr->get_function() );
		std::list< AlternativeFinder > argAlternatives;
		findSubExprs( untypedExpr->begin_args(), untypedExpr->end_args(), back_inserter( argAlternatives ) );

		std::list< AltList > possibilities;
		combos( argAlternatives.begin(), argAlternatives.end(), back_inserter( possibilities ) );

		Tuples::TupleAssignSpotter tassign( this );
		if ( tassign.isTupleAssignment( untypedExpr, possibilities ) ) {
			// take care of possible tuple assignments, or discard expression
			return;
		} // else ...

		AltList candidates;

		for ( AltList::const_iterator func = funcFinder.alternatives.begin(); func != funcFinder.alternatives.end(); ++func ) {
			PRINT(
				std::cout << "working on alternative: " << std::endl;
				func->print( std::cout, 8 );
			)
			// check if the type is pointer to function
			PointerType *pointer;
			if ( func->expr->get_results().size() == 1 && ( pointer = dynamic_cast< PointerType* >( func->expr->get_results().front() ) ) ) {
				if ( FunctionType *function = dynamic_cast< FunctionType* >( pointer->get_base() ) ) {
					for ( std::list< AltList >::iterator actualAlt = possibilities.begin(); actualAlt != possibilities.end(); ++actualAlt ) {
						// XXX
						//Designators::check_alternative( function, *actualAlt );
						makeFunctionAlternatives( *func, function, *actualAlt, std::back_inserter( candidates ) );
					}
				} else if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( pointer->get_base() ) ) {
					EqvClass eqvClass;
					if ( func->env.lookup( typeInst->get_name(), eqvClass ) && eqvClass.type ) {
						if ( FunctionType *function = dynamic_cast< FunctionType* >( eqvClass.type ) ) {
							for ( std::list< AltList >::iterator actualAlt = possibilities.begin(); actualAlt != possibilities.end(); ++actualAlt ) {
								makeFunctionAlternatives( *func, function, *actualAlt, std::back_inserter( candidates ) );
							} // for
						} // if
					} // if
				} // if
			} else {
				// seek a function operator that's compatible
				if ( ! doneInit ) {
					doneInit = true;
					NameExpr *opExpr = new NameExpr( "?()" );
					try {
						funcOpFinder.findWithAdjustment( opExpr );
					} catch( SemanticError &e ) {
						// it's ok if there aren't any defined function ops
					}
					PRINT(
						std::cout << "known function ops:" << std::endl;
						printAlts( funcOpFinder.alternatives, std::cout, 8 );
					)
				}

				for ( AltList::const_iterator funcOp = funcOpFinder.alternatives.begin(); funcOp != funcOpFinder.alternatives.end(); ++funcOp ) {
					// check if the type is pointer to function
					PointerType *pointer;
					if ( funcOp->expr->get_results().size() == 1
						 && ( pointer = dynamic_cast< PointerType* >( funcOp->expr->get_results().front() ) ) ) {
						if ( FunctionType *function = dynamic_cast< FunctionType* >( pointer->get_base() ) ) {
							for ( std::list< AltList >::iterator actualAlt = possibilities.begin(); actualAlt != possibilities.end(); ++actualAlt ) {
								AltList currentAlt;
								currentAlt.push_back( *func );
								currentAlt.insert( currentAlt.end(), actualAlt->begin(), actualAlt->end() );
								makeFunctionAlternatives( *funcOp, function, currentAlt, std::back_inserter( candidates ) );
							} // for
						} // if
					} // if
				} // for
			} // if
		} // for

		for ( AltList::iterator withFunc = candidates.begin(); withFunc != candidates.end(); ++withFunc ) {
			Cost cvtCost = computeConversionCost( *withFunc, indexer );

			PRINT(
				ApplicationExpr *appExpr = dynamic_cast< ApplicationExpr* >( withFunc->expr );
				assert( appExpr );
				PointerType *pointer = dynamic_cast< PointerType* >( appExpr->get_function()->get_results().front() );
				assert( pointer );
				FunctionType *function = dynamic_cast< FunctionType* >( pointer->get_base() );
				assert( function );
				std::cout << "Case +++++++++++++" << std::endl;
				std::cout << "formals are:" << std::endl;
				printAll( function->get_parameters(), std::cout, 8 );
				std::cout << "actuals are:" << std::endl;
				printAll( appExpr->get_args(), std::cout, 8 );
				std::cout << "bindings are:" << std::endl;
				withFunc->env.print( std::cout, 8 );
				std::cout << "cost of conversion is:" << cvtCost << std::endl;
			)
			if ( cvtCost != Cost::infinity ) {
				withFunc->cvtCost = cvtCost;
				alternatives.push_back( *withFunc );
			} // if
		} // for
		candidates.clear();
		candidates.splice( candidates.end(), alternatives );

		findMinCost( candidates.begin(), candidates.end(), std::back_inserter( alternatives ) );
	}

	bool isLvalue( Expression *expr ) {
		for ( std::list< Type* >::const_iterator i = expr->get_results().begin(); i != expr->get_results().end(); ++i ) {
			if ( !(*i)->get_isLvalue() ) return false;
		} // for
		return true;
	}

	void AlternativeFinder::visit( AddressExpr *addressExpr ) {
		AlternativeFinder finder( indexer, env );
		finder.find( addressExpr->get_arg() );
		for ( std::list< Alternative >::iterator i = finder.alternatives.begin(); i != finder.alternatives.end(); ++i ) {
			if ( isLvalue( i->expr ) ) {
				alternatives.push_back( Alternative( new AddressExpr( i->expr->clone() ), i->env, i->cost ) );
			} // if
		} // for
	}

	void AlternativeFinder::visit( CastExpr *castExpr ) {
		for ( std::list< Type* >::iterator i = castExpr->get_results().begin(); i != castExpr->get_results().end(); ++i ) {
			SymTab::validateType( *i, &indexer );
			adjustExprType( *i, env, indexer );
		} // for

		AlternativeFinder finder( indexer, env );
		finder.findWithAdjustment( castExpr->get_arg() );

		AltList candidates;
		for ( std::list< Alternative >::iterator i = finder.alternatives.begin(); i != finder.alternatives.end(); ++i ) {
			AssertionSet needAssertions, haveAssertions;
			OpenVarSet openVars;

			// It's possible that a cast can throw away some values in a multiply-valued expression.  (An example is a
			// cast-to-void, which casts from one value to zero.)  Figure out the prefix of the subexpression results
			// that are cast directly.  The candidate is invalid if it has fewer results than there are types to cast
			// to.
			int discardedValues = (*i).expr->get_results().size() - castExpr->get_results().size();
			if ( discardedValues < 0 ) continue;
			std::list< Type* >::iterator candidate_end = (*i).expr->get_results().begin();
			std::advance( candidate_end, castExpr->get_results().size() );
			// unification run for side-effects
			unifyList( castExpr->get_results().begin(), castExpr->get_results().end(),
					   (*i).expr->get_results().begin(), candidate_end,
			           i->env, needAssertions, haveAssertions, openVars, indexer );
			Cost thisCost = castCostList( (*i).expr->get_results().begin(), candidate_end,
										  castExpr->get_results().begin(), castExpr->get_results().end(),
										  indexer, i->env );
			if ( thisCost != Cost::infinity ) {
				// count one safe conversion for each value that is thrown away
				thisCost += Cost( 0, 0, discardedValues );
				CastExpr *newExpr = castExpr->clone();
				newExpr->set_arg( i->expr->clone() );
				candidates.push_back( Alternative( newExpr, i->env, i->cost, thisCost ) );
			} // if
		} // for

		// findMinCost selects the alternatives with the lowest "cost" members, but has the side effect of copying the
		// cvtCost member to the cost member (since the old cost is now irrelevant).  Thus, calling findMinCost twice
		// selects first based on argument cost, then on conversion cost.
		AltList minArgCost;
		findMinCost( candidates.begin(), candidates.end(), std::back_inserter( minArgCost ) );
		findMinCost( minArgCost.begin(), minArgCost.end(), std::back_inserter( alternatives ) );
	}

	void AlternativeFinder::visit( UntypedMemberExpr *memberExpr ) {
		AlternativeFinder funcFinder( indexer, env );
		funcFinder.findWithAdjustment( memberExpr->get_aggregate() );

		for ( AltList::const_iterator agg = funcFinder.alternatives.begin(); agg != funcFinder.alternatives.end(); ++agg ) {
			if ( agg->expr->get_results().size() == 1 ) {
				if ( StructInstType *structInst = dynamic_cast< StructInstType* >( agg->expr->get_results().front() ) ) {
					addAggMembers( structInst, agg->expr, agg->cost, memberExpr->get_member() );
				} else if ( UnionInstType *unionInst = dynamic_cast< UnionInstType* >( agg->expr->get_results().front() ) ) {
					addAggMembers( unionInst, agg->expr, agg->cost, memberExpr->get_member() );
				} // if
			} // if
		} // for
	}

	void AlternativeFinder::visit( MemberExpr *memberExpr ) {
		alternatives.push_back( Alternative( memberExpr->clone(), env, Cost::zero ) );
	}

	void AlternativeFinder::visit( NameExpr *nameExpr ) {
		std::list< DeclarationWithType* > declList;
		indexer.lookupId( nameExpr->get_name(), declList );
		PRINT( std::cerr << "nameExpr is " << nameExpr->get_name() << std::endl; )
		for ( std::list< DeclarationWithType* >::iterator i = declList.begin(); i != declList.end(); ++i ) {
			VariableExpr newExpr( *i, nameExpr->get_argName() );
			alternatives.push_back( Alternative( newExpr.clone(), env, Cost() ) );
			PRINT(
				std::cerr << "decl is ";
				(*i)->print( std::cerr );
				std::cerr << std::endl;
				std::cerr << "newExpr is ";
				newExpr.print( std::cerr );
				std::cerr << std::endl;
			)
			renameTypes( alternatives.back().expr );
			if ( StructInstType *structInst = dynamic_cast< StructInstType* >( (*i)->get_type() ) ) {
				addAggMembers( structInst, &newExpr, Cost( 0, 0, 1 ), "" );
			} else if ( UnionInstType *unionInst = dynamic_cast< UnionInstType* >( (*i)->get_type() ) ) {
				addAggMembers( unionInst, &newExpr, Cost( 0, 0, 1 ), "" );
			} // if
		} // for
	}

	void AlternativeFinder::visit( VariableExpr *variableExpr ) {
		alternatives.push_back( Alternative( variableExpr->clone(), env, Cost::zero ) );
	}

	void AlternativeFinder::visit( ConstantExpr *constantExpr ) {
		alternatives.push_back( Alternative( constantExpr->clone(), env, Cost::zero ) );
	}

	void AlternativeFinder::visit( SizeofExpr *sizeofExpr ) {
		if ( sizeofExpr->get_isType() ) {
			alternatives.push_back( Alternative( sizeofExpr->clone(), env, Cost::zero ) );
		} else {
			// find all alternatives for the argument to sizeof
			AlternativeFinder finder( indexer, env );
			finder.find( sizeofExpr->get_expr() );
			// find the lowest cost alternative among the alternatives, otherwise ambiguous
			AltList winners;
			findMinCost( finder.alternatives.begin(), finder.alternatives.end(), back_inserter( winners ) );
			if ( winners.size() != 1 ) {
				throw SemanticError( "Ambiguous expression in sizeof operand: ", sizeofExpr->get_expr() );
			} // if
			// return the lowest cost alternative for the argument
			Alternative &choice = winners.front();
			alternatives.push_back( Alternative( new SizeofExpr( choice.expr->clone() ), choice.env, Cost::zero ) );
		} // if
	}

	void AlternativeFinder::visit( AlignofExpr *alignofExpr ) {
		if ( alignofExpr->get_isType() ) {
			alternatives.push_back( Alternative( alignofExpr->clone(), env, Cost::zero ) );
		} else {
			// find all alternatives for the argument to sizeof
			AlternativeFinder finder( indexer, env );
			finder.find( alignofExpr->get_expr() );
			// find the lowest cost alternative among the alternatives, otherwise ambiguous
			AltList winners;
			findMinCost( finder.alternatives.begin(), finder.alternatives.end(), back_inserter( winners ) );
			if ( winners.size() != 1 ) {
				throw SemanticError( "Ambiguous expression in alignof operand: ", alignofExpr->get_expr() );
			} // if
			// return the lowest cost alternative for the argument
			Alternative &choice = winners.front();
			alternatives.push_back( Alternative( new AlignofExpr( choice.expr->clone() ), choice.env, Cost::zero ) );
		} // if
	}

	void AlternativeFinder::resolveAttr( DeclarationWithType *funcDecl, FunctionType *function, Type *argType, const TypeEnvironment &env ) {
		// assume no polymorphism
		// assume no implicit conversions
		assert( function->get_parameters().size() == 1 );
		PRINT(
			std::cout << "resolvAttr: funcDecl is ";
			funcDecl->print( std::cout );
			std::cout << " argType is ";
			argType->print( std::cout );
			std::cout << std::endl;
		)
		if ( typesCompatibleIgnoreQualifiers( argType, function->get_parameters().front()->get_type(), indexer, env ) ) {
			alternatives.push_back( Alternative( new AttrExpr( new VariableExpr( funcDecl ), argType->clone() ), env, Cost::zero ) );
			for ( std::list< DeclarationWithType* >::iterator i = function->get_returnVals().begin(); i != function->get_returnVals().end(); ++i ) {
				alternatives.back().expr->get_results().push_back( (*i)->get_type()->clone() );
			} // for
		} // if
	}

	void AlternativeFinder::visit( AttrExpr *attrExpr ) {
		// assume no 'pointer-to-attribute'
		NameExpr *nameExpr = dynamic_cast< NameExpr* >( attrExpr->get_attr() );
		assert( nameExpr );
		std::list< DeclarationWithType* > attrList;
		indexer.lookupId( nameExpr->get_name(), attrList );
		if ( attrExpr->get_isType() || attrExpr->get_expr() ) {
			for ( std::list< DeclarationWithType* >::iterator i = attrList.begin(); i != attrList.end(); ++i ) {
				// check if the type is function
				if ( FunctionType *function = dynamic_cast< FunctionType* >( (*i)->get_type() ) ) {
					// assume exactly one parameter
					if ( function->get_parameters().size() == 1 ) {
						if ( attrExpr->get_isType() ) {
							resolveAttr( *i, function, attrExpr->get_type(), env );
						} else {
							AlternativeFinder finder( indexer, env );
							finder.find( attrExpr->get_expr() );
							for ( AltList::iterator choice = finder.alternatives.begin(); choice != finder.alternatives.end(); ++choice ) {
								if ( choice->expr->get_results().size() == 1 ) {
									resolveAttr(*i, function, choice->expr->get_results().front(), choice->env );
								} // fi
							} // for
						} // if
					} // if
				} // if
			} // for
		} else {
			for ( std::list< DeclarationWithType* >::iterator i = attrList.begin(); i != attrList.end(); ++i ) {
				VariableExpr newExpr( *i );
				alternatives.push_back( Alternative( newExpr.clone(), env, Cost() ) );
				renameTypes( alternatives.back().expr );
			} // for
		} // if
	}

	void AlternativeFinder::visit( LogicalExpr *logicalExpr ) {
		AlternativeFinder firstFinder( indexer, env );
		firstFinder.findWithAdjustment( logicalExpr->get_arg1() );
		for ( AltList::const_iterator first = firstFinder.alternatives.begin(); first != firstFinder.alternatives.end(); ++first ) {
			AlternativeFinder secondFinder( indexer, first->env );
			secondFinder.findWithAdjustment( logicalExpr->get_arg2() );
			for ( AltList::const_iterator second = secondFinder.alternatives.begin(); second != secondFinder.alternatives.end(); ++second ) {
				LogicalExpr *newExpr = new LogicalExpr( first->expr->clone(), second->expr->clone(), logicalExpr->get_isAnd() );
				alternatives.push_back( Alternative( newExpr, second->env, first->cost + second->cost ) );
			}
		}
	}

	void AlternativeFinder::visit( ConditionalExpr *conditionalExpr ) {
		AlternativeFinder firstFinder( indexer, env );
		firstFinder.findWithAdjustment( conditionalExpr->get_arg1() );
		for ( AltList::const_iterator first = firstFinder.alternatives.begin(); first != firstFinder.alternatives.end(); ++first ) {
			AlternativeFinder secondFinder( indexer, first->env );
			secondFinder.findWithAdjustment( conditionalExpr->get_arg2() );
			for ( AltList::const_iterator second = secondFinder.alternatives.begin(); second != secondFinder.alternatives.end(); ++second ) {
				AlternativeFinder thirdFinder( indexer, second->env );
				thirdFinder.findWithAdjustment( conditionalExpr->get_arg3() );
				for ( AltList::const_iterator third = thirdFinder.alternatives.begin(); third != thirdFinder.alternatives.end(); ++third ) {
					OpenVarSet openVars;
					AssertionSet needAssertions, haveAssertions;
					Alternative newAlt( 0, third->env, first->cost + second->cost + third->cost );
					std::list< Type* > commonTypes;
					if ( unifyList( second->expr->get_results().begin(), second->expr->get_results().end(), third->expr->get_results().begin(), third->expr->get_results().end(), newAlt.env, needAssertions, haveAssertions, openVars, indexer, commonTypes ) ) {
						ConditionalExpr *newExpr = new ConditionalExpr( first->expr->clone(), second->expr->clone(), third->expr->clone() );
						std::list< Type* >::const_iterator original = second->expr->get_results().begin();
						std::list< Type* >::const_iterator commonType = commonTypes.begin();
						for ( ; original != second->expr->get_results().end() && commonType != commonTypes.end(); ++original, ++commonType ) {
							if ( *commonType ) {
								newExpr->get_results().push_back( *commonType );
							} else {
								newExpr->get_results().push_back( (*original)->clone() );
							} // if
						} // for
						newAlt.expr = newExpr;
						inferParameters( needAssertions, haveAssertions, newAlt, openVars, back_inserter( alternatives ) );
					} // if
				} // for
			} // for
		} // for
	}

	void AlternativeFinder::visit( CommaExpr *commaExpr ) {
		TypeEnvironment newEnv( env );
		Expression *newFirstArg = resolveInVoidContext( commaExpr->get_arg1(), indexer, newEnv );
		AlternativeFinder secondFinder( indexer, newEnv );
		secondFinder.findWithAdjustment( commaExpr->get_arg2() );
		for ( AltList::const_iterator alt = secondFinder.alternatives.begin(); alt != secondFinder.alternatives.end(); ++alt ) {
			alternatives.push_back( Alternative( new CommaExpr( newFirstArg->clone(), alt->expr->clone() ), alt->env, alt->cost ) );
		} // for
		delete newFirstArg;
	}

	void AlternativeFinder::visit( TupleExpr *tupleExpr ) {
		std::list< AlternativeFinder > subExprAlternatives;
		findSubExprs( tupleExpr->get_exprs().begin(), tupleExpr->get_exprs().end(), back_inserter( subExprAlternatives ) );
		std::list< AltList > possibilities;
		combos( subExprAlternatives.begin(), subExprAlternatives.end(), back_inserter( possibilities ) );
		for ( std::list< AltList >::const_iterator i = possibilities.begin(); i != possibilities.end(); ++i ) {
			TupleExpr *newExpr = new TupleExpr;
			makeExprList( *i, newExpr->get_exprs() );
			for ( std::list< Expression* >::const_iterator resultExpr = newExpr->get_exprs().begin(); resultExpr != newExpr->get_exprs().end(); ++resultExpr ) {
				for ( std::list< Type* >::const_iterator resultType = (*resultExpr)->get_results().begin(); resultType != (*resultExpr)->get_results().end(); ++resultType ) {
					newExpr->get_results().push_back( (*resultType)->clone() );
				} // for
			} // for

			TypeEnvironment compositeEnv;
			simpleCombineEnvironments( i->begin(), i->end(), compositeEnv );
			alternatives.push_back( Alternative( newExpr, compositeEnv, sumCost( *i ) ) );
		} // for
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
