//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TupleAssignment.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 15:02:53 2015
// Update Count     : 2
//

#include <iterator>
#include <iostream>
#include <cassert>
#include "Tuples.h"
#include "GenPoly/DeclMutator.h"
#include "SynTree/Mutator.h"
#include "SynTree/Statement.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SymTab/Mangler.h"
#include "Common/ScopedMap.h"
#include "ResolvExpr/typeops.h"
#include "InitTweak/GenInit.h"

namespace Tuples {
	namespace {
		class MemberTupleExpander final : public Mutator {
		public:
			typedef Mutator Parent;
			using Parent::mutate;

			virtual Expression * mutate( UntypedMemberExpr * memberExpr ) override;
		};

		class UniqueExprExpander final : public GenPoly::DeclMutator {
		public:
			typedef GenPoly::DeclMutator Parent;
			using Parent::mutate;

			virtual Expression * mutate( UniqueExpr * unqExpr ) override;

			std::map< int, Expression * > decls; // not vector, because order added may not be increasing order

			~UniqueExprExpander() {
				for ( std::pair<const int, Expression *> & p : decls ) {
					delete p.second;
				}
			}
		};

		class TupleAssignExpander : public Mutator {
		public:
			typedef Mutator Parent;
			using Parent::mutate;

			virtual Expression * mutate( TupleAssignExpr * tupleExpr );
		};

		class TupleTypeReplacer : public GenPoly::DeclMutator {
		  public:
			typedef GenPoly::DeclMutator Parent;
			using Parent::mutate;

			virtual Type * mutate( TupleType * tupleType ) override;

			virtual CompoundStmt * mutate( CompoundStmt * stmt ) override {
				typeMap.beginScope();
				stmt = Parent::mutate( stmt );
				typeMap.endScope();
				return stmt;
			}
		  private:
			ScopedMap< std::string, StructDecl * > typeMap;
		};

		class TupleIndexExpander final : public Mutator {
		public:
			typedef Mutator Parent;
			using Parent::mutate;

			virtual Expression * mutate( TupleIndexExpr * tupleExpr ) override;
		};

		class TupleExprExpander final : public Mutator {
		public:
			typedef Mutator Parent;
			using Parent::mutate;

			virtual Expression * mutate( TupleExpr * tupleExpr ) override;
		};
	}

	void expandMemberTuples( std::list< Declaration * > & translationUnit ) {
		MemberTupleExpander expander;
		mutateAll( translationUnit, expander );
	}

	void expandUniqueExpr( std::list< Declaration * > & translationUnit ) {
		UniqueExprExpander unqExpander;
		unqExpander.mutateDeclarationList( translationUnit );
	}

	void expandTuples( std::list< Declaration * > & translationUnit ) {
		TupleAssignExpander assnExpander;
		mutateAll( translationUnit, assnExpander );

		TupleTypeReplacer replacer;
		replacer.mutateDeclarationList( translationUnit );

		TupleIndexExpander idxExpander;
		mutateAll( translationUnit, idxExpander );

		TupleExprExpander exprExpander;
		mutateAll( translationUnit, exprExpander );
	}

	namespace {
		/// given a expression representing the member and an expression representing the aggregate,
		/// reconstructs a flattened UntypedMemberExpr with the right precedence
		Expression * reconstructMemberExpr( Expression * member, Expression * aggr ) {
			if ( UntypedMemberExpr * memberExpr = dynamic_cast< UntypedMemberExpr * >( member ) ) {
				// construct a new UntypedMemberExpr with the correct structure , and recursively
				// expand that member expression.
				MemberTupleExpander expander;
				UntypedMemberExpr * newMemberExpr = new UntypedMemberExpr( memberExpr->get_member(), new UntypedMemberExpr( memberExpr->get_aggregate(), aggr->clone() ) );

				memberExpr->set_member(nullptr);
				memberExpr->set_aggregate(nullptr);
				delete memberExpr;
				return newMemberExpr->acceptMutator( expander );
			} else {
				// not a member expression, so there is nothing to do but attach and return
				return new UntypedMemberExpr( member, aggr->clone() );
			}
		}
	}

	Expression * MemberTupleExpander::mutate( UntypedMemberExpr * memberExpr ) {
		if ( UntypedTupleExpr * tupleExpr = dynamic_cast< UntypedTupleExpr * > ( memberExpr->get_member() ) ) {
			Expression * aggr = memberExpr->get_aggregate()->clone()->acceptMutator( *this );
			// aggregate expressions which might be impure must be wrapped in unique expressions
			// xxx - if there's a member-tuple expression nested in the aggregate, this currently generates the wrong code if a UniqueExpr is not used, and it's purely an optimization to remove the UniqueExpr
			// if ( Tuples::maybeImpure( memberExpr->get_aggregate() ) ) aggr = new UniqueExpr( aggr );
			aggr = new UniqueExpr( aggr );
			for ( Expression *& expr : tupleExpr->get_exprs() ) {
				expr = reconstructMemberExpr( expr, aggr );
			}
			delete aggr;
			return tupleExpr;
		} else {
			// there may be a tuple expr buried in the aggregate
			// xxx - this is a memory leak
			return new UntypedMemberExpr( memberExpr->get_member()->clone(), memberExpr->get_aggregate()->acceptMutator( *this ) );
		}
	}

	Expression * UniqueExprExpander::mutate( UniqueExpr * unqExpr ) {
		unqExpr = safe_dynamic_cast< UniqueExpr * > ( Parent::mutate( unqExpr ) );
		const int id = unqExpr->get_id();

		// on first time visiting a unique expr with a particular ID, generate the expression that replaces all UniqueExprs with that ID,
		// and lookup on subsequent hits. This ensures that all unique exprs with the same ID reference the same variable.
		if ( ! decls.count( id ) ) {
			Expression * assignUnq;
			Expression * var = unqExpr->get_var();
			if ( unqExpr->get_object() ) {
				// an object was generated to represent this unique expression -- it should be added to the list of declarations now
				addDeclaration( unqExpr->get_object() );
				unqExpr->set_object( nullptr );
				// steal the expr from the unqExpr
				assignUnq = UntypedExpr::createAssign( unqExpr->get_var()->clone(), unqExpr->get_expr() );
				unqExpr->set_expr( nullptr );
			} else {
				// steal the already generated assignment to var from the unqExpr - this has been generated by FixInit
				Expression * expr = unqExpr->get_expr();
				CommaExpr * commaExpr = safe_dynamic_cast< CommaExpr * >( expr );
				assignUnq = commaExpr->get_arg1();
				commaExpr->set_arg1( nullptr );
			}
			BasicType * boolType = new BasicType( Type::Qualifiers(), BasicType::Bool );
			ObjectDecl * finished = new ObjectDecl( toString( "_unq_expr_finished_", id ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, nullptr, new BasicType( Type::Qualifiers(), BasicType::Bool ), new SingleInit( new ConstantExpr( Constant( boolType->clone(), "0" ) ), noDesignators ) );
			addDeclaration( finished );
			// (finished ? _unq_expr_N : (_unq_expr_N = <unqExpr->get_expr()>, finished = 1, _unq_expr_N))
			// This pattern ensures that each unique expression is evaluated once, regardless of evaluation order of the generated C code.
			Expression * assignFinished = UntypedExpr::createAssign( new VariableExpr(finished), new ConstantExpr( Constant( boolType->clone(), "1" ) ) );
			ConditionalExpr * condExpr = new ConditionalExpr( new VariableExpr( finished ), var->clone(),
				new CommaExpr( new CommaExpr( assignUnq, assignFinished ), var->clone() ) );
			condExpr->set_result( var->get_result()->clone() );
			condExpr->set_env( maybeClone( unqExpr->get_env() ) );
			decls[id] = condExpr;
		}
		delete unqExpr;
		return decls[id]->clone();
	}

	Expression * TupleAssignExpander::mutate( TupleAssignExpr * assnExpr ) {
		assnExpr = safe_dynamic_cast< TupleAssignExpr * >( Parent::mutate( assnExpr ) );
		StmtExpr * ret = assnExpr->get_stmtExpr();
		assnExpr->set_stmtExpr( nullptr );
		// move env to StmtExpr
		ret->set_env( assnExpr->get_env() );
		assnExpr->set_env( nullptr );
		delete assnExpr;
		return ret;
	}

	Type * TupleTypeReplacer::mutate( TupleType * tupleType ) {
		std::string mangleName = SymTab::Mangler::mangleType( tupleType );
		tupleType = safe_dynamic_cast< TupleType * > ( Parent::mutate( tupleType ) );
		if ( ! typeMap.count( mangleName ) ) {
			// generate struct type to replace tuple type
			// xxx - should fix this to only generate one tuple struct for each number of type parameters
			StructDecl * decl = new StructDecl( "_tuple_type_" + mangleName );
			decl->set_body( true );
			for ( size_t i = 0; i < tupleType->size(); ++i ) {
				TypeDecl * tyParam = new TypeDecl( toString("tuple_param_", i), DeclarationNode::NoStorageClass, nullptr, TypeDecl::Any );
				decl->get_members().push_back( new ObjectDecl( toString("field_", i), DeclarationNode::NoStorageClass, LinkageSpec::C, nullptr, new TypeInstType( Type::Qualifiers(), tyParam->get_name(), tyParam ), nullptr ) );
				decl->get_parameters().push_back( tyParam );
			}
			if ( tupleType->size() == 0 ) {
				// empty structs are not standard C. Add a dummy field to empty tuples to silence warnings when a compound literal Tuple0 is created.
				decl->get_members().push_back( new ObjectDecl( "dummy", DeclarationNode::NoStorageClass, LinkageSpec::C, nullptr, new BasicType( Type::Qualifiers(), BasicType::SignedInt ), nullptr ) );
			}
			typeMap[mangleName] = decl;
			addDeclaration( decl );
		}
		Type::Qualifiers qualifiers = tupleType->get_qualifiers();

		StructDecl * decl = typeMap[mangleName];
		StructInstType * newType = new StructInstType( qualifiers, decl );
		for ( Type * t : *tupleType ) {
			newType->get_parameters().push_back( new TypeExpr( t->clone() ) );
		}
		delete tupleType;
		return newType;
	}

	Expression * TupleIndexExpander::mutate( TupleIndexExpr * tupleExpr ) {
		Expression * tuple = maybeMutate( tupleExpr->get_tuple(), *this );
		assert( tuple );
		tupleExpr->set_tuple( nullptr );
		unsigned int idx = tupleExpr->get_index();
		TypeSubstitution * env = tupleExpr->get_env();
		tupleExpr->set_env( nullptr );
		delete tupleExpr;

		StructInstType * type = safe_dynamic_cast< StructInstType * >( tuple->get_result() );
		StructDecl * structDecl = type->get_baseStruct();
		assert( structDecl->get_members().size() > idx );
		Declaration * member = *std::next(structDecl->get_members().begin(), idx);
		MemberExpr * memExpr = new MemberExpr( safe_dynamic_cast< DeclarationWithType * >( member ), tuple );
		memExpr->set_env( env );
		return memExpr;
	}

	Expression * replaceTupleExpr( Type * result, const std::list< Expression * > & exprs, TypeSubstitution * env ) {
		if ( result->isVoid() ) {
			// void result - don't need to produce a value for cascading - just output a chain of comma exprs
			assert( ! exprs.empty() );
			std::list< Expression * >::const_iterator iter = exprs.begin();
			Expression * expr = new CastExpr( *iter++ );
			for ( ; iter != exprs.end(); ++iter ) {
				expr = new CommaExpr( expr, new CastExpr( *iter ) );
			}
			expr->set_env( env );
			return expr;
		} else {
			// typed tuple expression - produce a compound literal which performs each of the expressions
			// as a distinct part of its initializer - the produced compound literal may be used as part of
			// another expression
			std::list< Initializer * > inits;
			for ( Expression * expr : exprs ) {
				inits.push_back( new SingleInit( expr ) );
			}
			Expression * expr = new CompoundLiteralExpr( result, new ListInit( inits ) );
			expr->set_env( env );
			return expr;
		}
	}

	Expression * TupleExprExpander::mutate( TupleExpr * tupleExpr ) {
		// recursively expand sub-tuple-expressions
		tupleExpr = safe_dynamic_cast<TupleExpr *>(Parent::mutate(tupleExpr));
		Type * result = tupleExpr->get_result();
		std::list< Expression * > exprs = tupleExpr->get_exprs();
		assert( result );
		TypeSubstitution * env = tupleExpr->get_env();

		// remove data from shell and delete it
		tupleExpr->set_result( nullptr );
		tupleExpr->get_exprs().clear();
		tupleExpr->set_env( nullptr );
		delete tupleExpr;

		return replaceTupleExpr( result, exprs, env );
	}

	Type * makeTupleType( const std::list< Expression * > & exprs ) {
		// produce the TupleType which aggregates the types of the exprs
		TupleType *tupleType = new TupleType( Type::Qualifiers(true, true, true, true, true, false) );
		Type::Qualifiers &qualifiers = tupleType->get_qualifiers();
		for ( Expression * expr : exprs ) {
			assert( expr->get_result() );
			if ( expr->get_result()->isVoid() ) {
				// if the type of any expr is void, the type of the entire tuple is void
				delete tupleType;
				return new VoidType( Type::Qualifiers() );
			}
			Type * type = expr->get_result()->clone();
			tupleType->get_types().push_back( type );
			// the qualifiers on the tuple type are the qualifiers that exist on all component types
			qualifiers &= type->get_qualifiers();
		} // for
		if ( exprs.empty() ) qualifiers = Type::Qualifiers();
		return tupleType;
	}

	TypeInstType * isTtype( Type * type ) {
		if ( TypeInstType * inst = dynamic_cast< TypeInstType * >( type ) ) {
			if ( inst->get_baseType()->get_kind() == TypeDecl::Ttype ) {
				return inst;
			}
		}
		return nullptr;
	}

	namespace {
		/// determines if impurity (read: side-effects) may exist in a piece of code. Currently gives a very crude approximation, wherein any function call expression means the code may be impure
		class ImpurityDetector : public Visitor {
		public:
			typedef Visitor Parent;
			virtual void visit( ApplicationExpr * appExpr ) { maybeImpure = true;	}
			virtual void visit( UntypedExpr * untypedExpr ) { maybeImpure = true; }
			bool maybeImpure = false;
		};
	} // namespace

	bool maybeImpure( Expression * expr ) {
		ImpurityDetector detector;
		expr->accept( detector );
		return detector.maybeImpure;
	}
} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
