//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Box.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Feb  5 16:45:07 2016
// Update Count     : 286
//

#include <algorithm>
#include <iterator>
#include <list>
#include <map>
#include <set>
#include <stack>
#include <string>
#include <utility>
#include <vector>
#include <cassert>

#include "Box.h"
#include "DeclMutator.h"
#include "PolyMutator.h"
#include "FindFunction.h"
#include "ScopedMap.h"
#include "ScopedSet.h"
#include "ScrubTyVars.h"

#include "Parser/ParseNode.h"

#include "SynTree/Constant.h"
#include "SynTree/Declaration.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Mutator.h"
#include "SynTree/Statement.h"
#include "SynTree/Type.h"
#include "SynTree/TypeSubstitution.h"

#include "ResolvExpr/TypeEnvironment.h"
#include "ResolvExpr/TypeMap.h"
#include "ResolvExpr/typeops.h"

#include "SymTab/Indexer.h"
#include "SymTab/Mangler.h"

#include "Common/SemanticError.h"
#include "Common/UniqueName.h"
#include "Common/utility.h"

#include <ext/functional> // temporary

namespace GenPoly {
	namespace {
		const std::list<Label> noLabels;

		FunctionType *makeAdapterType( FunctionType *adaptee, const TyVarMap &tyVars );

		/// Abstracts type equality for a list of parameter types
		struct TypeList {
			TypeList() : params() {}
			TypeList( const std::list< Type* > &_params ) : params() { cloneAll(_params, params); }
			TypeList( std::list< Type* > &&_params ) : params( _params ) {}

			TypeList( const TypeList &that ) : params() { cloneAll(that.params, params); }
			TypeList( TypeList &&that ) : params( std::move( that.params ) ) {}

			/// Extracts types from a list of TypeExpr*
			TypeList( const std::list< TypeExpr* >& _params ) : params() {
				for ( std::list< TypeExpr* >::const_iterator param = _params.begin(); param != _params.end(); ++param ) {
					params.push_back( (*param)->get_type()->clone() );
				}
			}

			TypeList& operator= ( const TypeList &that ) {
				deleteAll( params );

				params.clear();
				cloneAll( that.params, params );

				return *this;
			}

			TypeList& operator= ( TypeList &&that ) {
				deleteAll( params );

				params = std::move( that.params );

				return *this;
			}

			~TypeList() { deleteAll( params ); }

			bool operator== ( const TypeList& that ) const {
				if ( params.size() != that.params.size() ) return false;

				SymTab::Indexer dummy;
				for ( std::list< Type* >::const_iterator it = params.begin(), jt = that.params.begin(); it != params.end(); ++it, ++jt ) {
					if ( ! ResolvExpr::typesCompatible( *it, *jt, dummy ) ) return false;
				}
				return true;
			}

			std::list< Type* > params;  ///< Instantiation parameters
		};

		/// Maps a key and a TypeList to the some value, accounting for scope
		template< typename Key, typename Value >
		class InstantiationMap {
			/// Wraps value for a specific (Key, TypeList) combination
			typedef std::pair< TypeList, Value* > Instantiation;
			/// List of TypeLists paired with their appropriate values
			typedef std::vector< Instantiation > ValueList;
			/// Underlying map type; maps keys to a linear list of corresponding TypeLists and values
			typedef ScopedMap< Key*, ValueList > InnerMap;

			InnerMap instantiations;  ///< instantiations

		public:
			/// Starts a new scope
			void beginScope() { instantiations.beginScope(); }

			/// Ends a scope
			void endScope() { instantiations.endScope(); }

			/// Gets the value for the (key, typeList) pair, returns NULL on none such.
			Value *lookup( Key *key, const std::list< TypeExpr* >& params ) const {
 				TypeList typeList( params );
 				
				// scan scopes for matches to the key
				for ( typename InnerMap::const_iterator insts = instantiations.find( key ); insts != instantiations.end(); insts = instantiations.findNext( insts, key ) ) {
					for ( typename ValueList::const_reverse_iterator inst = insts->second.rbegin(); inst != insts->second.rend(); ++inst ) {
						if ( inst->first == typeList ) return inst->second;
					}
				}
				// no matching instantiations found
				return 0;
			}

			/// Adds a value for a (key, typeList) pair to the current scope
			void insert( Key *key, const std::list< TypeExpr* > &params, Value *value ) {
				instantiations[ key ].push_back( Instantiation( TypeList( params ), value ) );
			}
		};

		/// Adds layout-generation functions to polymorphic types
		class LayoutFunctionBuilder : public DeclMutator {
			unsigned int functionNesting;  // current level of nested functions
		public:
			LayoutFunctionBuilder() : functionNesting( 0 ) {}

			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual Declaration *mutate( StructDecl *structDecl );
			virtual Declaration *mutate( UnionDecl *unionDecl );
		};
		
		/// Replaces polymorphic return types with out-parameters, replaces calls to polymorphic functions with adapter calls as needed, and adds appropriate type variables to the function call
		class Pass1 : public PolyMutator {
		  public:
			Pass1();
			virtual Expression *mutate( ApplicationExpr *appExpr );
			virtual Expression *mutate( AddressExpr *addrExpr );
			virtual Expression *mutate( UntypedExpr *expr );
			virtual DeclarationWithType* mutate( FunctionDecl *functionDecl );
			virtual TypeDecl *mutate( TypeDecl *typeDecl );
			virtual Expression *mutate( CommaExpr *commaExpr );
			virtual Expression *mutate( ConditionalExpr *condExpr );
			virtual Statement * mutate( ReturnStmt *returnStmt );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type * mutate( FunctionType *functionType );

			virtual void doBeginScope();
			virtual void doEndScope();
		  private:
			/// Pass the extra type parameters from polymorphic generic arguments or return types into a function application
			void passArgTypeVars( ApplicationExpr *appExpr, Type *parmType, Type *argBaseType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars, std::set< std::string > &seenTypes );
			/// passes extra type parameters into a polymorphic function application
			void passTypeVars( ApplicationExpr *appExpr, ReferenceToType *polyRetType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars );
			/// wraps a function application with a new temporary for the out-parameter return value
			Expression *addRetParam( ApplicationExpr *appExpr, FunctionType *function, Type *retType, std::list< Expression *>::iterator &arg );
			/// Replaces all the type parameters of a generic type with their concrete equivalents under the current environment
			void replaceParametersWithConcrete( ApplicationExpr *appExpr, std::list< Expression* >& params );
			/// Replaces a polymorphic type with its concrete equivalant under the current environment (returns itself if concrete).
			/// If `doClone` is set to false, will not clone interior types
			Type *replaceWithConcrete( ApplicationExpr *appExpr, Type *type, bool doClone = true );
			/// wraps a function application returning a polymorphic type with a new temporary for the out-parameter return value
			Expression *addPolyRetParam( ApplicationExpr *appExpr, FunctionType *function, ReferenceToType *polyType, std::list< Expression *>::iterator &arg );
			Expression *applyAdapter( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars );
			void boxParam( Type *formal, Expression *&arg, const TyVarMap &exprTyVars );
			void boxParams( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars );
			void addInferredParams( ApplicationExpr *appExpr, FunctionType *functionType, std::list< Expression *>::iterator &arg, const TyVarMap &tyVars );
			/// Stores assignment operators from assertion list in local map of assignment operations
			void findAssignOps( const std::list< TypeDecl *> &forall );
			void passAdapters( ApplicationExpr *appExpr, FunctionType *functionType, const TyVarMap &exprTyVars );
			FunctionDecl *makeAdapter( FunctionType *adaptee, FunctionType *realType, const std::string &mangleName, const TyVarMap &tyVars );
			/// Replaces intrinsic operator functions with their arithmetic desugaring
			Expression *handleIntrinsics( ApplicationExpr *appExpr );
			/// Inserts a new temporary variable into the current scope with an auto-generated name
			ObjectDecl *makeTemporary( Type *type );

			ScopedMap< std::string, DeclarationWithType *> assignOps;    ///< Currently known type variable assignment operators
			ResolvExpr::TypeMap< DeclarationWithType > scopedAssignOps;  ///< Currently known assignment operators
			ScopedMap< std::string, DeclarationWithType* > adapters;     ///< Set of adapter functions in the current scope
			
			DeclarationWithType *retval;
			bool useRetval;
			UniqueName tempNamer;
		};

		/// * Moves polymorphic returns in function types to pointer-type parameters
		/// * adds type size and assertion parameters to parameter lists
		class Pass2 : public PolyMutator {
		  public:
			template< typename DeclClass >
			DeclClass *handleDecl( DeclClass *decl, Type *type );
			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual ObjectDecl *mutate( ObjectDecl *objectDecl );
			virtual TypeDecl *mutate( TypeDecl *typeDecl );
			virtual TypedefDecl *mutate( TypedefDecl *typedefDecl );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type *mutate( FunctionType *funcType );
			
		  private:
			void addAdapters( FunctionType *functionType );

			std::map< UniqueId, std::string > adapterName;
		};

		/// Mutator pass that replaces concrete instantiations of generic types with actual struct declarations, scoped appropriately
		class GenericInstantiator : public DeclMutator {
			/// Map of (generic type, parameter list) pairs to concrete type instantiations
			InstantiationMap< AggregateDecl, AggregateDecl > instantiations;
			/// Namer for concrete types
			UniqueName typeNamer;

		public:
			GenericInstantiator() : DeclMutator(), instantiations(), typeNamer("_conc_") {}

			virtual Type* mutate( StructInstType *inst );
			virtual Type* mutate( UnionInstType *inst );

	// 		virtual Expression* mutate( MemberExpr *memberExpr );

			virtual void doBeginScope();
			virtual void doEndScope();
		private:
			/// Wrap instantiation lookup for structs
			StructDecl* lookup( StructInstType *inst, const std::list< TypeExpr* > &typeSubs ) { return (StructDecl*)instantiations.lookup( inst->get_baseStruct(), typeSubs ); }
			/// Wrap instantiation lookup for unions
			UnionDecl* lookup( UnionInstType *inst, const std::list< TypeExpr* > &typeSubs ) { return (UnionDecl*)instantiations.lookup( inst->get_baseUnion(), typeSubs ); }
			/// Wrap instantiation insertion for structs
			void insert( StructInstType *inst, const std::list< TypeExpr* > &typeSubs, StructDecl *decl ) { instantiations.insert( inst->get_baseStruct(), typeSubs, decl ); }
			/// Wrap instantiation insertion for unions
			void insert( UnionInstType *inst, const std::list< TypeExpr* > &typeSubs, UnionDecl *decl ) { instantiations.insert( inst->get_baseUnion(), typeSubs, decl ); }
		};

		/// Replaces member and size/align/offsetof expressions on polymorphic generic types with calculated expressions.
		/// * Replaces member expressions for polymorphic types with calculated add-field-offset-and-dereference
		/// * Calculates polymorphic offsetof expressions from offset array
		/// * Inserts dynamic calculation of polymorphic type layouts where needed
		class PolyGenericCalculator : public PolyMutator {
		public:
			template< typename DeclClass >
			DeclClass *handleDecl( DeclClass *decl, Type *type );
			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual ObjectDecl *mutate( ObjectDecl *objectDecl );
			virtual TypedefDecl *mutate( TypedefDecl *objectDecl );
			virtual TypeDecl *mutate( TypeDecl *objectDecl );
			virtual Statement *mutate( DeclStmt *declStmt );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type *mutate( FunctionType *funcType );
			virtual Expression *mutate( MemberExpr *memberExpr );
			virtual Expression *mutate( SizeofExpr *sizeofExpr );
			virtual Expression *mutate( AlignofExpr *alignofExpr );
			virtual Expression *mutate( OffsetofExpr *offsetofExpr );
			virtual Expression *mutate( OffsetPackExpr *offsetPackExpr );

			virtual void doBeginScope();
			virtual void doEndScope();

		private:
			/// Makes a new variable in the current scope with the given name, type & optional initializer
			ObjectDecl *makeVar( const std::string &name, Type *type, Initializer *init = 0 );
			/// returns true if the type has a dynamic layout; such a layout will be stored in appropriately-named local variables when the function returns
			bool findGeneric( Type *ty );
			/// adds type parameters to the layout call; will generate the appropriate parameters if needed
			void addOtypeParamsToLayoutCall( UntypedExpr *layoutCall, const std::list< Type* > &otypeParams );
			
			ScopedSet< std::string > knownLayouts;          ///< Set of generic type layouts known in the current scope, indexed by sizeofName
			ScopedSet< std::string > knownOffsets;          ///< Set of non-generic types for which the offset array exists in the current scope, indexed by offsetofName
		};

		/// Replaces initialization of polymorphic values with alloca, declaration of dtype/ftype with appropriate void expression, and sizeof expressions of polymorphic types with the proper variable
		class Pass3 : public PolyMutator {
		  public:
			template< typename DeclClass >
			DeclClass *handleDecl( DeclClass *decl, Type *type );
			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual ObjectDecl *mutate( ObjectDecl *objectDecl );
			virtual TypedefDecl *mutate( TypedefDecl *objectDecl );
			virtual TypeDecl *mutate( TypeDecl *objectDecl );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type *mutate( FunctionType *funcType );
		  private:
		};

	} // anonymous namespace

	/// version of mutateAll with special handling for translation unit so you can check the end of the prelude when debugging
	template< typename MutatorType >
	inline void mutateTranslationUnit( std::list< Declaration* > &translationUnit, MutatorType &mutator ) {
		bool seenIntrinsic = false;
		SemanticError errors;
		for ( typename std::list< Declaration* >::iterator i = translationUnit.begin(); i != translationUnit.end(); ++i ) {
			try {
				if ( *i ) {
					if ( (*i)->get_linkage() == LinkageSpec::Intrinsic ) {
						seenIntrinsic = true;
					} else if ( seenIntrinsic ) {
						seenIntrinsic = false; // break on this line when debugging for end of prelude
					}

					*i = dynamic_cast< Declaration* >( (*i)->acceptMutator( mutator ) );
					assert( *i );
				} // if
			} catch( SemanticError &e ) {
				errors.append( e );
			} // try
		} // for
		if ( ! errors.isEmpty() ) {
			throw errors;
		} // if
	}

	void box( std::list< Declaration *>& translationUnit ) {
		LayoutFunctionBuilder layoutBuilder;
		Pass1 pass1;
		Pass2 pass2;
		GenericInstantiator instantiator;
		PolyGenericCalculator polyCalculator;
		Pass3 pass3;
		
		layoutBuilder.mutateDeclarationList( translationUnit );
		mutateTranslationUnit/*All*/( translationUnit, pass1 );
		mutateTranslationUnit/*All*/( translationUnit, pass2 );
		instantiator.mutateDeclarationList( translationUnit );
		mutateTranslationUnit/*All*/( translationUnit, polyCalculator );
		mutateTranslationUnit/*All*/( translationUnit, pass3 );
	}

	////////////////////////////////// LayoutFunctionBuilder ////////////////////////////////////////////

	DeclarationWithType *LayoutFunctionBuilder::mutate( FunctionDecl *functionDecl ) {
		functionDecl->set_functionType( maybeMutate( functionDecl->get_functionType(), *this ) );
		mutateAll( functionDecl->get_oldDecls(), *this );
		++functionNesting;
		functionDecl->set_statements( maybeMutate( functionDecl->get_statements(), *this ) );
		--functionNesting;
		return functionDecl;
	}
	
	/// Get a list of type declarations that will affect a layout function
	std::list< TypeDecl* > takeOtypeOnly( std::list< TypeDecl* > &decls ) {
		std::list< TypeDecl * > otypeDecls;

		for ( std::list< TypeDecl* >::const_iterator decl = decls.begin(); decl != decls.end(); ++decl ) {
			if ( (*decl)->get_kind() == TypeDecl::Any ) {
				otypeDecls.push_back( *decl );
			}
		}
		
		return otypeDecls;
	}

	/// Adds parameters for otype layout to a function type
	void addOtypeParams( FunctionType *layoutFnType, std::list< TypeDecl* > &otypeParams ) {
		BasicType sizeAlignType( Type::Qualifiers(), BasicType::LongUnsignedInt );
		
		for ( std::list< TypeDecl* >::const_iterator param = otypeParams.begin(); param != otypeParams.end(); ++param ) {
			TypeInstType paramType( Type::Qualifiers(), (*param)->get_name(), *param );
			std::string paramName = mangleType( &paramType );
			layoutFnType->get_parameters().push_back( new ObjectDecl( sizeofName( paramName ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignType.clone(), 0 ) );
			layoutFnType->get_parameters().push_back( new ObjectDecl( alignofName( paramName ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignType.clone(), 0 ) );
		}
	}

	/// Builds a layout function declaration
	FunctionDecl *buildLayoutFunctionDecl( AggregateDecl *typeDecl, unsigned int functionNesting, FunctionType *layoutFnType ) {
		// Routines at global scope marked "static" to prevent multiple definitions is separate translation units
		// because each unit generates copies of the default routines for each aggregate.
		FunctionDecl *layoutDecl = new FunctionDecl(
			layoutofName( typeDecl ), functionNesting > 0 ? DeclarationNode::NoStorageClass : DeclarationNode::Static, LinkageSpec::AutoGen, layoutFnType, new CompoundStmt( noLabels ), true, false );
		layoutDecl->fixUniqueId();
		return layoutDecl;
	}

	/// Makes a unary operation
	Expression *makeOp( const std::string &name, Expression *arg ) {
		UntypedExpr *expr = new UntypedExpr( new NameExpr( name ) );
		expr->get_args().push_back( arg );
		return expr;
	}

	/// Makes a binary operation
	Expression *makeOp( const std::string &name, Expression *lhs, Expression *rhs ) {
		UntypedExpr *expr = new UntypedExpr( new NameExpr( name ) );
		expr->get_args().push_back( lhs );
		expr->get_args().push_back( rhs );
		return expr;
	}

	/// Returns the dereference of a local pointer variable
	Expression *derefVar( ObjectDecl *var ) {
		return makeOp( "*?", new VariableExpr( var ) );
	}

	/// makes an if-statement with a single-expression if-block and no then block
	Statement *makeCond( Expression *cond, Expression *ifPart ) {
		return new IfStmt( noLabels, cond, new ExprStmt( noLabels, ifPart ), 0 );
	}

	/// makes a statement that assigns rhs to lhs if lhs < rhs
	Statement *makeAssignMax( Expression *lhs, Expression *rhs ) {
		return makeCond( makeOp( "?<?", lhs, rhs ), makeOp( "?=?", lhs->clone(), rhs->clone() ) );
	}

	/// makes a statement that aligns lhs to rhs (rhs should be an integer power of two)
	Statement *makeAlignTo( Expression *lhs, Expression *rhs ) {
		// check that the lhs is zeroed out to the level of rhs
		Expression *ifCond = makeOp( "?&?", lhs, makeOp( "?-?", rhs, new ConstantExpr( Constant( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), "1" ) ) ) );
		// if not aligned, increment to alignment
		Expression *ifExpr = makeOp( "?+=?", lhs->clone(), makeOp( "?-?", rhs->clone(), ifCond->clone() ) );
		return makeCond( ifCond, ifExpr );
	}
	
	/// adds an expression to a compound statement
	void addExpr( CompoundStmt *stmts, Expression *expr ) {
		stmts->get_kids().push_back( new ExprStmt( noLabels, expr ) );
	}

	/// adds a statement to a compound statement
	void addStmt( CompoundStmt *stmts, Statement *stmt ) {
		stmts->get_kids().push_back( stmt );
	}
	
	Declaration *LayoutFunctionBuilder::mutate( StructDecl *structDecl ) {
		// do not generate layout function for "empty" tag structs
		if ( structDecl->get_members().empty() ) return structDecl;

		// get parameters that can change layout, exiting early if none
		std::list< TypeDecl* > otypeParams = takeOtypeOnly( structDecl->get_parameters() );
		if ( otypeParams.empty() ) return structDecl;

		// build layout function signature
		FunctionType *layoutFnType = new FunctionType( Type::Qualifiers(), false );
		BasicType *sizeAlignType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );
		PointerType *sizeAlignOutType = new PointerType( Type::Qualifiers(), sizeAlignType );
		
		ObjectDecl *sizeParam = new ObjectDecl( sizeofName( structDecl->get_name() ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignOutType, 0 );
		layoutFnType->get_parameters().push_back( sizeParam );
		ObjectDecl *alignParam = new ObjectDecl( alignofName( structDecl->get_name() ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignOutType->clone(), 0 );
		layoutFnType->get_parameters().push_back( alignParam );
		ObjectDecl *offsetParam = new ObjectDecl( offsetofName( structDecl->get_name() ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignOutType->clone(), 0 );
		layoutFnType->get_parameters().push_back( offsetParam );
		addOtypeParams( layoutFnType, otypeParams );

		// build function decl
		FunctionDecl *layoutDecl = buildLayoutFunctionDecl( structDecl, functionNesting, layoutFnType );

		// calculate struct layout in function body

		// initialize size and alignment to 0 and 1 (will have at least one member to re-edit size
		addExpr( layoutDecl->get_statements(), makeOp( "?=?", derefVar( sizeParam ), new ConstantExpr( Constant( sizeAlignType->clone(), "0" ) ) ) );
		addExpr( layoutDecl->get_statements(), makeOp( "?=?", derefVar( alignParam ), new ConstantExpr( Constant( sizeAlignType->clone(), "1" ) ) ) );
		unsigned long n_members = 0;
		bool firstMember = true;
		for ( std::list< Declaration* >::const_iterator member = structDecl->get_members().begin(); member != structDecl->get_members().end(); ++member ) {
			DeclarationWithType *dwt = dynamic_cast< DeclarationWithType * >( *member );
			assert( dwt );
			Type *memberType = dwt->get_type();

			if ( firstMember ) {
				firstMember = false;
			} else {
				// make sure all members after the first (automatically aligned at 0) are properly padded for alignment
				addStmt( layoutDecl->get_statements(), makeAlignTo( derefVar( sizeParam ), new AlignofExpr( memberType->clone() ) ) );
			}
			
			// place current size in the current offset index
			addExpr( layoutDecl->get_statements(), makeOp( "?=?", makeOp( "?[?]", new VariableExpr( offsetParam ), new ConstantExpr( Constant::from( n_members ) ) ),
			                                                      derefVar( sizeParam ) ) );
			++n_members;

			// add member size to current size
			addExpr( layoutDecl->get_statements(), makeOp( "?+=?", derefVar( sizeParam ), new SizeofExpr( memberType->clone() ) ) );
			
			// take max of member alignment and global alignment
			addStmt( layoutDecl->get_statements(), makeAssignMax( derefVar( alignParam ), new AlignofExpr( memberType->clone() ) ) );
		}
		// make sure the type is end-padded to a multiple of its alignment
		addStmt( layoutDecl->get_statements(), makeAlignTo( derefVar( sizeParam ), derefVar( alignParam ) ) );

		addDeclarationAfter( layoutDecl );
		return structDecl;
	}
	
	Declaration *LayoutFunctionBuilder::mutate( UnionDecl *unionDecl ) {
		// do not generate layout function for "empty" tag unions
		if ( unionDecl->get_members().empty() ) return unionDecl;
		
		// get parameters that can change layout, exiting early if none
		std::list< TypeDecl* > otypeParams = takeOtypeOnly( unionDecl->get_parameters() );
		if ( otypeParams.empty() ) return unionDecl;

		// build layout function signature
		FunctionType *layoutFnType = new FunctionType( Type::Qualifiers(), false );
		BasicType *sizeAlignType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );
		PointerType *sizeAlignOutType = new PointerType( Type::Qualifiers(), sizeAlignType );
		
		ObjectDecl *sizeParam = new ObjectDecl( sizeofName( unionDecl->get_name() ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignOutType, 0 );
		layoutFnType->get_parameters().push_back( sizeParam );
		ObjectDecl *alignParam = new ObjectDecl( alignofName( unionDecl->get_name() ), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, sizeAlignOutType->clone(), 0 );
		layoutFnType->get_parameters().push_back( alignParam );
		addOtypeParams( layoutFnType, otypeParams );

		// build function decl
		FunctionDecl *layoutDecl = buildLayoutFunctionDecl( unionDecl, functionNesting, layoutFnType );

		// calculate union layout in function body
		addExpr( layoutDecl->get_statements(), makeOp( "?=?", derefVar( sizeParam ), new ConstantExpr( Constant( sizeAlignType->clone(), "1" ) ) ) );
		addExpr( layoutDecl->get_statements(), makeOp( "?=?", derefVar( alignParam ), new ConstantExpr( Constant( sizeAlignType->clone(), "1" ) ) ) );
		for ( std::list< Declaration* >::const_iterator member = unionDecl->get_members().begin(); member != unionDecl->get_members().end(); ++member ) {
			DeclarationWithType *dwt = dynamic_cast< DeclarationWithType * >( *member );
			assert( dwt );
			Type *memberType = dwt->get_type();
			
			// take max member size and global size
			addStmt( layoutDecl->get_statements(), makeAssignMax( derefVar( sizeParam ), new SizeofExpr( memberType->clone() ) ) );
			
			// take max of member alignment and global alignment
			addStmt( layoutDecl->get_statements(), makeAssignMax( derefVar( alignParam ), new AlignofExpr( memberType->clone() ) ) );
		}
		// make sure the type is end-padded to a multiple of its alignment
		addStmt( layoutDecl->get_statements(), makeAlignTo( derefVar( sizeParam ), derefVar( alignParam ) ) );

		addDeclarationAfter( layoutDecl );
		return unionDecl;
	}
	
	////////////////////////////////////////// Pass1 ////////////////////////////////////////////////////

	namespace {
		std::string makePolyMonoSuffix( FunctionType * function, const TyVarMap &tyVars ) {
			std::stringstream name;

			// NOTE: this function previously used isPolyObj, which failed to produce
			// the correct thing in some situations. It's not clear to me why this wasn't working.

			// if the return type or a parameter type involved polymorphic types, then the adapter will need
			// to take those polymorphic types as pointers. Therefore, there can be two different functions
			// with the same mangled name, so we need to further mangle the names.
			for ( std::list< DeclarationWithType *>::iterator retval = function->get_returnVals().begin(); retval != function->get_returnVals().end(); ++retval ) {
				if ( isPolyType( (*retval)->get_type(), tyVars ) ) {
					name << "P";
				} else {
					name << "M";
				}
			}
			name << "_";
			std::list< DeclarationWithType *> &paramList = function->get_parameters();
			for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
				if ( isPolyType( (*arg)->get_type(), tyVars ) ) {
					name << "P";
				} else {
					name << "M";
				}
			} // for
			return name.str();
		}

		std::string mangleAdapterName( FunctionType * function, const TyVarMap &tyVars ) {
			return SymTab::Mangler::mangle( function ) + makePolyMonoSuffix( function, tyVars );
		}

		std::string makeAdapterName( const std::string &mangleName ) {
			return "_adapter" + mangleName;
		}

		Pass1::Pass1() : useRetval( false ), tempNamer( "_temp" ) {}

		/// Returns T if the given declaration is (*?=?)(T *, T) for some TypeInstType T (return not checked, but maybe should be), NULL otherwise
		TypeInstType *isTypeInstAssignment( DeclarationWithType *decl ) {
			if ( decl->get_name() == "?=?" ) {
				if ( FunctionType *funType = getFunctionType( decl->get_type() ) ) {
					if ( funType->get_parameters().size() == 2 ) {
						if ( PointerType *pointer = dynamic_cast< PointerType *>( funType->get_parameters().front()->get_type() ) ) {
							if ( TypeInstType *refType = dynamic_cast< TypeInstType *>( pointer->get_base() ) ) {
								if ( TypeInstType *refType2 = dynamic_cast< TypeInstType *>( funType->get_parameters().back()->get_type() ) ) {
									if ( refType->get_name() == refType2->get_name() ) {
										return refType;
									} // if
								} // if
							} // if
						} // if
					} // if
				} // if
			} // if
			return 0;
		}
		
		/// returns T if the given declaration is: (*?=?)(T *, T) for some type T (return not checked, but maybe should be), NULL otherwise
		/// Only picks assignments where neither parameter is cv-qualified
		Type *isAssignment( DeclarationWithType *decl ) {
			if ( decl->get_name() == "?=?" ) {
				if ( FunctionType *funType = getFunctionType( decl->get_type() ) ) {
					if ( funType->get_parameters().size() == 2 ) {
						Type::Qualifiers defaultQualifiers;
						Type *paramType1 = funType->get_parameters().front()->get_type();
						if ( paramType1->get_qualifiers() != defaultQualifiers ) return 0;
						Type *paramType2 = funType->get_parameters().back()->get_type();
						if ( paramType2->get_qualifiers() != defaultQualifiers ) return 0;
						
						if ( PointerType *pointerType = dynamic_cast< PointerType* >( paramType1 ) ) {
							Type *baseType1 = pointerType->get_base();
							if ( baseType1->get_qualifiers() != defaultQualifiers ) return 0;
							SymTab::Indexer dummy;
							if ( ResolvExpr::typesCompatible( baseType1, paramType2, dummy ) ) {
								return baseType1;
							} // if
						} // if
					} // if
				} // if
			} // if
			return 0;
		}

		void Pass1::findAssignOps( const std::list< TypeDecl *> &forall ) {
			// what if a nested function uses an assignment operator?
			// assignOps.clear();
			for ( std::list< TypeDecl *>::const_iterator i = forall.begin(); i != forall.end(); ++i ) {
				for ( std::list< DeclarationWithType *>::const_iterator assert = (*i)->get_assertions().begin(); assert != (*i)->get_assertions().end(); ++assert ) {
					std::string typeName;
					if ( TypeInstType *typeInst = isTypeInstAssignment( *assert ) ) {
						assignOps[ typeInst->get_name() ] = *assert;
					} // if
				} // for
			} // for
		}

		DeclarationWithType *Pass1::mutate( FunctionDecl *functionDecl ) {
			// if this is a assignment function, put it in the map for this scope
			if ( Type *assignedType = isAssignment( functionDecl ) ) {
				if ( ! dynamic_cast< TypeInstType* >( assignedType ) ) {
					scopedAssignOps.insert( assignedType, functionDecl );
				}
			}

			if ( functionDecl->get_statements() ) {		// empty routine body ?
				doBeginScope();
				scopeTyVars.beginScope();
				assignOps.beginScope();
				DeclarationWithType *oldRetval = retval;
				bool oldUseRetval = useRetval;

				// process polymorphic return value
				retval = 0;
				if ( isPolyRet( functionDecl->get_functionType() ) && functionDecl->get_linkage() == LinkageSpec::Cforall ) {
					retval = functionDecl->get_functionType()->get_returnVals().front();

					// give names to unnamed return values
					if ( retval->get_name() == "" ) {
						retval->set_name( "_retparm" );
						retval->set_linkage( LinkageSpec::C );
					} // if
				} // if

				FunctionType *functionType = functionDecl->get_functionType();
				makeTyVarMap( functionDecl->get_functionType(), scopeTyVars );
				findAssignOps( functionDecl->get_functionType()->get_forall() );

				std::list< DeclarationWithType *> &paramList = functionType->get_parameters();
				std::list< FunctionType *> functions;
				for ( std::list< TypeDecl *>::iterator tyVar = functionType->get_forall().begin(); tyVar != functionType->get_forall().end(); ++tyVar ) {
					for ( std::list< DeclarationWithType *>::iterator assert = (*tyVar)->get_assertions().begin(); assert != (*tyVar)->get_assertions().end(); ++assert ) {
						findFunction( (*assert)->get_type(), functions, scopeTyVars, needsAdapter );
					} // for
				} // for
				for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
					findFunction( (*arg)->get_type(), functions, scopeTyVars, needsAdapter );
				} // for

				for ( std::list< FunctionType *>::iterator funType = functions.begin(); funType != functions.end(); ++funType ) {
					std::string mangleName = mangleAdapterName( *funType, scopeTyVars );
					if ( adapters.find( mangleName ) == adapters.end() ) {
						std::string adapterName = makeAdapterName( mangleName );
						adapters.insert( std::pair< std::string, DeclarationWithType *>( mangleName, new ObjectDecl( adapterName, DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new PointerType( Type::Qualifiers(), makeAdapterType( *funType, scopeTyVars ) ), 0 ) ) );
					} // if
				} // for

				functionDecl->set_statements( functionDecl->get_statements()->acceptMutator( *this ) );

				scopeTyVars.endScope();
				assignOps.endScope();
				retval = oldRetval;
				useRetval = oldUseRetval;
				doEndScope();
			} // if
			return functionDecl;
		}

		TypeDecl *Pass1::mutate( TypeDecl *typeDecl ) {
			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			return Mutator::mutate( typeDecl );
		}

		Expression *Pass1::mutate( CommaExpr *commaExpr ) {
			bool oldUseRetval = useRetval;
			useRetval = false;
			commaExpr->set_arg1( maybeMutate( commaExpr->get_arg1(), *this ) );
			useRetval = oldUseRetval;
			commaExpr->set_arg2( maybeMutate( commaExpr->get_arg2(), *this ) );
			return commaExpr;
		}

		Expression *Pass1::mutate( ConditionalExpr *condExpr ) {
			bool oldUseRetval = useRetval;
			useRetval = false;
			condExpr->set_arg1( maybeMutate( condExpr->get_arg1(), *this ) );
			useRetval = oldUseRetval;
			condExpr->set_arg2( maybeMutate( condExpr->get_arg2(), *this ) );
			condExpr->set_arg3( maybeMutate( condExpr->get_arg3(), *this ) );
			return condExpr;

		}

		void Pass1::passArgTypeVars( ApplicationExpr *appExpr, Type *parmType, Type *argBaseType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars, std::set< std::string > &seenTypes ) {
			Type *polyType = isPolyType( parmType, exprTyVars );
			if ( polyType && ! dynamic_cast< TypeInstType* >( polyType ) ) {
				std::string typeName = mangleType( polyType );
				if ( seenTypes.count( typeName ) ) return;

				arg = appExpr->get_args().insert( arg, new SizeofExpr( argBaseType->clone() ) );
				arg++;
				arg = appExpr->get_args().insert( arg, new AlignofExpr( argBaseType->clone() ) );
				arg++;
				if ( dynamic_cast< StructInstType* >( polyType ) ) {
					if ( StructInstType *argBaseStructType = dynamic_cast< StructInstType* >( argBaseType ) ) {
						// zero-length arrays are forbidden by C, so don't pass offset for empty struct
						if ( ! argBaseStructType->get_baseStruct()->get_members().empty() ) {
							arg = appExpr->get_args().insert( arg, new OffsetPackExpr( argBaseStructType->clone() ) );
							arg++;
						}
					} else {
						throw SemanticError( "Cannot pass non-struct type for generic struct" );
					}
				}

				seenTypes.insert( typeName );
			}
		}

		void Pass1::passTypeVars( ApplicationExpr *appExpr, ReferenceToType *polyRetType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars ) {
			// pass size/align for type variables
			for ( TyVarMap::const_iterator tyParm = exprTyVars.begin(); tyParm != exprTyVars.end(); ++tyParm ) {
				ResolvExpr::EqvClass eqvClass;
				assert( env );
				if ( tyParm->second == TypeDecl::Any ) {
					Type *concrete = env->lookup( tyParm->first );
					if ( concrete ) {
						arg = appExpr->get_args().insert( arg, new SizeofExpr( concrete->clone() ) );
						arg++;
						arg = appExpr->get_args().insert( arg, new AlignofExpr( concrete->clone() ) );
						arg++;
					} else {
						throw SemanticError( "unbound type variable in application ", appExpr );
					} // if
				} // if
			} // for

			// add size/align for generic types to parameter list
			if ( appExpr->get_function()->get_results().empty() ) return;
			FunctionType *funcType = getFunctionType( appExpr->get_function()->get_results().front() );
			assert( funcType );

			std::list< DeclarationWithType* >::const_iterator fnParm = funcType->get_parameters().begin();
			std::list< Expression* >::const_iterator fnArg = arg;
			std::set< std::string > seenTypes; //< names for generic types we've seen

			// a polymorphic return type may need to be added to the argument list
			if ( polyRetType ) {
				Type *concRetType = replaceWithConcrete( appExpr, polyRetType );
				passArgTypeVars( appExpr, polyRetType, concRetType, arg, exprTyVars, seenTypes );
			}
			
			// add type information args for presently unseen types in parameter list
			for ( ; fnParm != funcType->get_parameters().end() && fnArg != appExpr->get_args().end(); ++fnParm, ++fnArg ) {
				VariableExpr *fnArgBase = getBaseVar( *fnArg );
				if ( ! fnArgBase || fnArgBase->get_results().empty() ) continue;
				passArgTypeVars( appExpr, (*fnParm)->get_type(), fnArgBase->get_results().front(), arg, exprTyVars, seenTypes );
			}
		}

		ObjectDecl *Pass1::makeTemporary( Type *type ) {
			ObjectDecl *newObj = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, type, 0 );
			stmtsToAdd.push_back( new DeclStmt( noLabels, newObj ) );
			return newObj;
		}

		Expression *Pass1::addRetParam( ApplicationExpr *appExpr, FunctionType *function, Type *retType, std::list< Expression *>::iterator &arg ) {
			// ***** Code Removal ***** After introducing a temporary variable for all return expressions, the following code appears superfluous.
			// if ( useRetval ) {
			// 	assert( retval );
			// 	arg = appExpr->get_args().insert( arg, new VariableExpr( retval ) );
			// 	arg++;
			// } else {

			// Create temporary to hold return value of polymorphic function and produce that temporary as a result
			// using a comma expression.  Possibly change comma expression into statement expression "{}" for multiple
			// return values.
			ObjectDecl *newObj = makeTemporary( retType->clone() );
			Expression *paramExpr = new VariableExpr( newObj );

			// If the type of the temporary is not polymorphic, box temporary by taking its address;
			// otherwise the temporary is already boxed and can be used directly.
			if ( ! isPolyType( newObj->get_type(), scopeTyVars, env ) ) {
				paramExpr = new AddressExpr( paramExpr );
			} // if
			arg = appExpr->get_args().insert( arg, paramExpr ); // add argument to function call
			arg++;
			// Build a comma expression to call the function and emulate a normal return.
			CommaExpr *commaExpr = new CommaExpr( appExpr, new VariableExpr( newObj ) );
			commaExpr->set_env( appExpr->get_env() );
			appExpr->set_env( 0 );
			return commaExpr;
			// } // if
			// return appExpr;
		}

		void Pass1::replaceParametersWithConcrete( ApplicationExpr *appExpr, std::list< Expression* >& params ) {
			for ( std::list< Expression* >::iterator param = params.begin(); param != params.end(); ++param ) {
				TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
				assert(paramType && "Aggregate parameters should be type expressions");
				paramType->set_type( replaceWithConcrete( appExpr, paramType->get_type(), false ) );
			}
		}

		Type *Pass1::replaceWithConcrete( ApplicationExpr *appExpr, Type *type, bool doClone ) {
			if ( TypeInstType *typeInst = dynamic_cast< TypeInstType * >( type ) ) {
				Type *concrete = env->lookup( typeInst->get_name() );
				if ( concrete == 0 ) {
					throw SemanticError( "Unbound type variable " + typeInst->get_name() + " in ", appExpr );
				} // if
				return concrete;
			} else if ( StructInstType *structType = dynamic_cast< StructInstType* >( type ) ) {
				if ( doClone ) {
					structType = structType->clone();
				}
				replaceParametersWithConcrete( appExpr, structType->get_parameters() );
				return structType;
			} else if ( UnionInstType *unionType = dynamic_cast< UnionInstType* >( type ) ) {
				if ( doClone ) {
					unionType = unionType->clone();
				}
				replaceParametersWithConcrete( appExpr, unionType->get_parameters() );
				return unionType;
			}
			return type;
		}

		Expression *Pass1::addPolyRetParam( ApplicationExpr *appExpr, FunctionType *function, ReferenceToType *polyType, std::list< Expression *>::iterator &arg ) {
			assert( env );
			Type *concrete = replaceWithConcrete( appExpr, polyType );
			// add out-parameter for return value	
			return addRetParam( appExpr, function, concrete, arg );
		}

		Expression *Pass1::applyAdapter( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &tyVars ) {
			Expression *ret = appExpr;
			if ( ! function->get_returnVals().empty() && isPolyType( function->get_returnVals().front()->get_type(), tyVars ) ) {
				ret = addRetParam( appExpr, function, function->get_returnVals().front()->get_type(), arg );
			} // if
			std::string mangleName = mangleAdapterName( function, tyVars );
			std::string adapterName = makeAdapterName( mangleName );

			// cast adaptee to void (*)(), since it may have any type inside a polymorphic function
			Type * adapteeType = new PointerType( Type::Qualifiers(), new FunctionType( Type::Qualifiers(), true ) );
			appExpr->get_args().push_front( new CastExpr( appExpr->get_function(), adapteeType ) );
			appExpr->set_function( new NameExpr( adapterName ) );

			return ret;
		}

		void Pass1::boxParam( Type *param, Expression *&arg, const TyVarMap &exprTyVars ) {
			assert( ! arg->get_results().empty() );
			if ( isPolyType( param, exprTyVars ) ) {
				if ( isPolyType( arg->get_results().front() ) ) {
					// if the argument's type is polymorphic, we don't need to box again!
					return;
				} else if ( arg->get_results().front()->get_isLvalue() ) {
					// VariableExpr and MemberExpr are lvalues; need to check this isn't coming from the second arg of a comma expression though (not an lvalue)
					if ( CommaExpr *commaArg = dynamic_cast< CommaExpr* >( arg ) ) {
						commaArg->set_arg2( new AddressExpr( commaArg->get_arg2() ) );
					} else {
						arg = new AddressExpr( arg );
					}
				} else {
					// use type computed in unification to declare boxed variables
					Type * newType = param->clone();
					if ( env ) env->apply( newType );
					ObjectDecl *newObj = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, newType, 0 );
					newObj->get_type()->get_qualifiers() = Type::Qualifiers(); // TODO: is this right???
					stmtsToAdd.push_back( new DeclStmt( noLabels, newObj ) );
					UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
					assign->get_args().push_back( new VariableExpr( newObj ) );
					assign->get_args().push_back( arg );
					stmtsToAdd.push_back( new ExprStmt( noLabels, assign ) );
					arg = new AddressExpr( new VariableExpr( newObj ) );
				} // if
			} // if
		}

		/// cast parameters to polymorphic functions so that types are replaced with
		/// void * if they are type parameters in the formal type.
		/// this gets rid of warnings from gcc.
		void addCast( Expression *&actual, Type *formal, const TyVarMap &tyVars ) {
			Type * newType = formal->clone();
			if ( getFunctionType( newType ) ) {
				newType = ScrubTyVars::scrub( newType, tyVars );
				actual = new CastExpr( actual, newType );
			} // if
		}

		void Pass1::boxParams( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars ) {
			for ( std::list< DeclarationWithType *>::const_iterator param = function->get_parameters().begin(); param != function->get_parameters().end(); ++param, ++arg ) {
				assert( arg != appExpr->get_args().end() );
				addCast( *arg, (*param)->get_type(), exprTyVars );
				boxParam( (*param)->get_type(), *arg, exprTyVars );
			} // for
		}

		void Pass1::addInferredParams( ApplicationExpr *appExpr, FunctionType *functionType, std::list< Expression *>::iterator &arg, const TyVarMap &tyVars ) {
			std::list< Expression *>::iterator cur = arg;
			for ( std::list< TypeDecl *>::iterator tyVar = functionType->get_forall().begin(); tyVar != functionType->get_forall().end(); ++tyVar ) {
				for ( std::list< DeclarationWithType *>::iterator assert = (*tyVar)->get_assertions().begin(); assert != (*tyVar)->get_assertions().end(); ++assert ) {
					InferredParams::const_iterator inferParam = appExpr->get_inferParams().find( (*assert)->get_uniqueId() );
					assert( inferParam != appExpr->get_inferParams().end() && "NOTE: Explicit casts of polymorphic functions to compatible monomorphic functions are currently unsupported" );
					Expression *newExpr = inferParam->second.expr->clone();
					addCast( newExpr, (*assert)->get_type(), tyVars );
					boxParam( (*assert)->get_type(), newExpr, tyVars );
					appExpr->get_args().insert( cur, newExpr );
				} // for
			} // for
		}

		void makeRetParm( FunctionType *funcType ) {
			DeclarationWithType *retParm = funcType->get_returnVals().front();

			// make a new parameter that is a pointer to the type of the old return value
			retParm->set_type( new PointerType( Type::Qualifiers(), retParm->get_type() ) );
			funcType->get_parameters().push_front( retParm );

			// we don't need the return value any more
			funcType->get_returnVals().clear();
		}

		FunctionType *makeAdapterType( FunctionType *adaptee, const TyVarMap &tyVars ) {
			// actually make the adapter type
			FunctionType *adapter = adaptee->clone();
			if ( ! adapter->get_returnVals().empty() && isPolyType( adapter->get_returnVals().front()->get_type(), tyVars ) ) {
				makeRetParm( adapter );
			} // if
			adapter->get_parameters().push_front( new ObjectDecl( "", DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new PointerType( Type::Qualifiers(), new FunctionType( Type::Qualifiers(), true ) ), 0 ) );
			return adapter;
		}

		Expression *makeAdapterArg( DeclarationWithType *param, DeclarationWithType *arg, DeclarationWithType *realParam, const TyVarMap &tyVars ) {
			assert( param );
			assert( arg );
			if ( isPolyType( realParam->get_type(), tyVars ) ) {
				if ( ! isPolyType( arg->get_type() ) ) {
					UntypedExpr *deref = new UntypedExpr( new NameExpr( "*?" ) );
					deref->get_args().push_back( new CastExpr( new VariableExpr( param ), new PointerType( Type::Qualifiers(), arg->get_type()->clone() ) ) );
					deref->get_results().push_back( arg->get_type()->clone() );
					return deref;
				} // if
			} // if
			return new VariableExpr( param );
		}

		void addAdapterParams( ApplicationExpr *adapteeApp, std::list< DeclarationWithType *>::iterator arg, std::list< DeclarationWithType *>::iterator param, std::list< DeclarationWithType *>::iterator paramEnd, std::list< DeclarationWithType *>::iterator realParam, const TyVarMap &tyVars ) {
			UniqueName paramNamer( "_p" );
			for ( ; param != paramEnd; ++param, ++arg, ++realParam ) {
				if ( (*param)->get_name() == "" ) {
					(*param)->set_name( paramNamer.newName() );
					(*param)->set_linkage( LinkageSpec::C );
				} // if
				adapteeApp->get_args().push_back( makeAdapterArg( *param, *arg, *realParam, tyVars ) );
			} // for
		}

		FunctionDecl *Pass1::makeAdapter( FunctionType *adaptee, FunctionType *realType, const std::string &mangleName, const TyVarMap &tyVars ) {
			FunctionType *adapterType = makeAdapterType( adaptee, tyVars );
			adapterType = ScrubTyVars::scrub( adapterType, tyVars );
			DeclarationWithType *adapteeDecl = adapterType->get_parameters().front();
			adapteeDecl->set_name( "_adaptee" );
			ApplicationExpr *adapteeApp = new ApplicationExpr( new CastExpr( new VariableExpr( adapteeDecl ), new PointerType( Type::Qualifiers(), realType ) ) );
			Statement *bodyStmt;

			std::list< TypeDecl *>::iterator tyArg = realType->get_forall().begin();
			std::list< TypeDecl *>::iterator tyParam = adapterType->get_forall().begin();
			std::list< TypeDecl *>::iterator realTyParam = adaptee->get_forall().begin();
			for ( ; tyParam != adapterType->get_forall().end(); ++tyArg, ++tyParam, ++realTyParam ) {
				assert( tyArg != realType->get_forall().end() );
				std::list< DeclarationWithType *>::iterator assertArg = (*tyArg)->get_assertions().begin();
				std::list< DeclarationWithType *>::iterator assertParam = (*tyParam)->get_assertions().begin();
				std::list< DeclarationWithType *>::iterator realAssertParam = (*realTyParam)->get_assertions().begin();
				for ( ; assertParam != (*tyParam)->get_assertions().end(); ++assertArg, ++assertParam, ++realAssertParam ) {
					assert( assertArg != (*tyArg)->get_assertions().end() );
					adapteeApp->get_args().push_back( makeAdapterArg( *assertParam, *assertArg, *realAssertParam, tyVars ) );
				} // for
			} // for

			std::list< DeclarationWithType *>::iterator arg = realType->get_parameters().begin();
			std::list< DeclarationWithType *>::iterator param = adapterType->get_parameters().begin();
			std::list< DeclarationWithType *>::iterator realParam = adaptee->get_parameters().begin();
			param++;		// skip adaptee parameter
			if ( realType->get_returnVals().empty() ) {
				addAdapterParams( adapteeApp, arg, param, adapterType->get_parameters().end(), realParam, tyVars );
				bodyStmt = new ExprStmt( noLabels, adapteeApp );
			} else if ( isPolyType( adaptee->get_returnVals().front()->get_type(), tyVars ) ) {
				if ( (*param)->get_name() == "" ) {
					(*param)->set_name( "_ret" );
					(*param)->set_linkage( LinkageSpec::C );
				} // if
				UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
				UntypedExpr *deref = new UntypedExpr( new NameExpr( "*?" ) );
				deref->get_args().push_back( new CastExpr( new VariableExpr( *param++ ), new PointerType( Type::Qualifiers(), realType->get_returnVals().front()->get_type()->clone() ) ) );
				assign->get_args().push_back( deref );
				addAdapterParams( adapteeApp, arg, param, adapterType->get_parameters().end(), realParam, tyVars );
				assign->get_args().push_back( adapteeApp );
				bodyStmt = new ExprStmt( noLabels, assign );
			} else {
				// adapter for a function that returns a monomorphic value
				addAdapterParams( adapteeApp, arg, param, adapterType->get_parameters().end(), realParam, tyVars );
				bodyStmt = new ReturnStmt( noLabels, adapteeApp );
			} // if
			CompoundStmt *adapterBody = new CompoundStmt( noLabels );
			adapterBody->get_kids().push_back( bodyStmt );
			std::string adapterName = makeAdapterName( mangleName );
			return new FunctionDecl( adapterName, DeclarationNode::NoStorageClass, LinkageSpec::C, adapterType, adapterBody, false, false );
		}

		void Pass1::passAdapters( ApplicationExpr * appExpr, FunctionType * functionType, const TyVarMap & exprTyVars ) {
			// collect a list of function types passed as parameters or implicit parameters (assertions)
			std::list< DeclarationWithType *> &paramList = functionType->get_parameters();
			std::list< FunctionType *> functions;
			for ( std::list< TypeDecl *>::iterator tyVar = functionType->get_forall().begin(); tyVar != functionType->get_forall().end(); ++tyVar ) {
				for ( std::list< DeclarationWithType *>::iterator assert = (*tyVar)->get_assertions().begin(); assert != (*tyVar)->get_assertions().end(); ++assert ) {
					findFunction( (*assert)->get_type(), functions, exprTyVars, needsAdapter );
				} // for
			} // for
			for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
				findFunction( (*arg)->get_type(), functions, exprTyVars, needsAdapter );
			} // for

			// parameter function types for which an appropriate adapter has been generated.  we cannot use the types
			// after applying substitutions, since two different parameter types may be unified to the same type
			std::set< std::string > adaptersDone;

			for ( std::list< FunctionType *>::iterator funType = functions.begin(); funType != functions.end(); ++funType ) {
				FunctionType *originalFunction = (*funType)->clone();
				FunctionType *realFunction = (*funType)->clone();
				std::string mangleName = SymTab::Mangler::mangle( realFunction );

				// only attempt to create an adapter or pass one as a parameter if we haven't already done so for this
				// pre-substitution parameter function type.
				if ( adaptersDone.find( mangleName ) == adaptersDone.end() ) {
					adaptersDone.insert( adaptersDone.begin(), mangleName );

					// apply substitution to type variables to figure out what the adapter's type should look like
					assert( env );
					env->apply( realFunction );
					mangleName = SymTab::Mangler::mangle( realFunction );
					mangleName += makePolyMonoSuffix( originalFunction, exprTyVars );

					typedef ScopedMap< std::string, DeclarationWithType* >::iterator AdapterIter;
					AdapterIter adapter = adapters.find( mangleName );
					if ( adapter == adapters.end() ) {
						// adapter has not been created yet in the current scope, so define it
						FunctionDecl *newAdapter = makeAdapter( *funType, realFunction, mangleName, exprTyVars );
						std::pair< AdapterIter, bool > answer = adapters.insert( std::pair< std::string, DeclarationWithType *>( mangleName, newAdapter ) );
						adapter = answer.first;
						stmtsToAdd.push_back( new DeclStmt( noLabels, newAdapter ) );
					} // if
					assert( adapter != adapters.end() );

					// add the appropriate adapter as a parameter
					appExpr->get_args().push_front( new VariableExpr( adapter->second ) );
				} // if
			} // for
		} // passAdapters

		Expression *makeIncrDecrExpr( ApplicationExpr *appExpr, Type *polyType, bool isIncr ) {
			NameExpr *opExpr;
			if ( isIncr ) {
				opExpr = new NameExpr( "?+=?" );
			} else {
				opExpr = new NameExpr( "?-=?" );
			} // if
			UntypedExpr *addAssign = new UntypedExpr( opExpr );
			if ( AddressExpr *address = dynamic_cast< AddressExpr *>( appExpr->get_args().front() ) ) {
				addAssign->get_args().push_back( address->get_arg() );
			} else {
				addAssign->get_args().push_back( appExpr->get_args().front() );
			} // if
			addAssign->get_args().push_back( new NameExpr( sizeofName( mangleType( polyType ) ) ) );
			addAssign->get_results().front() = appExpr->get_results().front()->clone();
			if ( appExpr->get_env() ) {
				addAssign->set_env( appExpr->get_env() );
				appExpr->set_env( 0 );
			} // if
			appExpr->get_args().clear();
			delete appExpr;
			return addAssign;
		}

		Expression *Pass1::handleIntrinsics( ApplicationExpr *appExpr ) {
			if ( VariableExpr *varExpr = dynamic_cast< VariableExpr *>( appExpr->get_function() ) ) {
				if ( varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic ) {
					if ( varExpr->get_var()->get_name() == "?[?]" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 2 );
						Type *baseType1 = isPolyPtr( appExpr->get_args().front()->get_results().front(), scopeTyVars, env );
						Type *baseType2 = isPolyPtr( appExpr->get_args().back()->get_results().front(), scopeTyVars, env );
						assert( ! baseType1 || ! baseType2 ); // the arguments cannot both be polymorphic pointers
						UntypedExpr *ret = 0;
						if ( baseType1 || baseType2 ) { // one of the arguments is a polymorphic pointer
							ret = new UntypedExpr( new NameExpr( "?+?" ) );
						} // if
						if ( baseType1 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().back() );
							multiply->get_args().push_back( new SizeofExpr( baseType1->clone() ) );
							ret->get_args().push_back( appExpr->get_args().front() );
							ret->get_args().push_back( multiply );
						} else if ( baseType2 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().front() );
							multiply->get_args().push_back( new SizeofExpr( baseType2->clone() ) );
							ret->get_args().push_back( multiply );
							ret->get_args().push_back( appExpr->get_args().back() );
						} // if
						if ( baseType1 || baseType2 ) {
							ret->get_results().push_front( appExpr->get_results().front()->clone() );
							if ( appExpr->get_env() ) {
								ret->set_env( appExpr->get_env() );
								appExpr->set_env( 0 );
							} // if
							appExpr->get_args().clear();
							delete appExpr;
							return ret;
						} // if
					} else if ( varExpr->get_var()->get_name() == "*?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( ! appExpr->get_args().empty() );
						if ( isPolyType( appExpr->get_results().front(), scopeTyVars, env ) ) {
							Expression *ret = appExpr->get_args().front();
							delete ret->get_results().front();
							ret->get_results().front() = appExpr->get_results().front()->clone();
							if ( appExpr->get_env() ) {
								ret->set_env( appExpr->get_env() );
								appExpr->set_env( 0 );
							} // if
							appExpr->get_args().clear();
							delete appExpr;
							return ret;
						} // if
					} else if ( varExpr->get_var()->get_name() == "?++" || varExpr->get_var()->get_name() == "?--" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 1 );
						if ( Type *baseType = isPolyPtr( appExpr->get_results().front(), scopeTyVars, env ) ) {
							Type *tempType = appExpr->get_results().front()->clone();
							if ( env ) {
								env->apply( tempType );
							} // if
							ObjectDecl *newObj = makeTemporary( tempType );
							VariableExpr *tempExpr = new VariableExpr( newObj );
							UntypedExpr *assignExpr = new UntypedExpr( new NameExpr( "?=?" ) );
							assignExpr->get_args().push_back( tempExpr->clone() );
							if ( AddressExpr *address = dynamic_cast< AddressExpr *>( appExpr->get_args().front() ) ) {
								assignExpr->get_args().push_back( address->get_arg()->clone() );
							} else {
								assignExpr->get_args().push_back( appExpr->get_args().front()->clone() );
							} // if
							CommaExpr *firstComma = new CommaExpr( assignExpr, makeIncrDecrExpr( appExpr, baseType, varExpr->get_var()->get_name() == "?++" ) );
							return new CommaExpr( firstComma, tempExpr );
						} // if
					} else if ( varExpr->get_var()->get_name() == "++?" || varExpr->get_var()->get_name() == "--?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 1 );
						if ( Type *baseType = isPolyPtr( appExpr->get_results().front(), scopeTyVars, env ) ) {
							return makeIncrDecrExpr( appExpr, baseType, varExpr->get_var()->get_name() == "++?" );
						} // if
					} else if ( varExpr->get_var()->get_name() == "?+?" || varExpr->get_var()->get_name() == "?-?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 2 );
						Type *baseType1 = isPolyPtr( appExpr->get_args().front()->get_results().front(), scopeTyVars, env );
						Type *baseType2 = isPolyPtr( appExpr->get_args().back()->get_results().front(), scopeTyVars, env );
						if ( baseType1 && baseType2 ) {
							UntypedExpr *divide = new UntypedExpr( new NameExpr( "?/?" ) );
							divide->get_args().push_back( appExpr );
							divide->get_args().push_back( new SizeofExpr( baseType1->clone() ) );
							divide->get_results().push_front( appExpr->get_results().front()->clone() );
							if ( appExpr->get_env() ) {
								divide->set_env( appExpr->get_env() );
								appExpr->set_env( 0 );
							} // if
							return divide;
						} else if ( baseType1 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().back() );
							multiply->get_args().push_back( new SizeofExpr( baseType1->clone() ) );
							appExpr->get_args().back() = multiply;
						} else if ( baseType2 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().front() );
							multiply->get_args().push_back( new SizeofExpr( baseType2->clone() ) );
							appExpr->get_args().front() = multiply;
						} // if
					} else if ( varExpr->get_var()->get_name() == "?+=?" || varExpr->get_var()->get_name() == "?-=?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 2 );
						Type *baseType = isPolyPtr( appExpr->get_results().front(), scopeTyVars, env );
						if ( baseType ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().back() );
							multiply->get_args().push_back( new SizeofExpr( baseType->clone() ) );
							appExpr->get_args().back() = multiply;
						} // if
					} // if
					return appExpr;
				} // if
			} // if
			return 0;
		}

		Expression *Pass1::mutate( ApplicationExpr *appExpr ) {
			// std::cerr << "mutate appExpr: ";
			// for ( TyVarMap::iterator i = scopeTyVars.begin(); i != scopeTyVars.end(); ++i ) {
			// 	std::cerr << i->first << " ";
			// }
			// std::cerr << "\n";
			bool oldUseRetval = useRetval;
			useRetval = false;
			appExpr->get_function()->acceptMutator( *this );
			mutateAll( appExpr->get_args(), *this );
			useRetval = oldUseRetval;

			assert( ! appExpr->get_function()->get_results().empty() );
			PointerType *pointer = dynamic_cast< PointerType *>( appExpr->get_function()->get_results().front() );
			assert( pointer );
			FunctionType *function = dynamic_cast< FunctionType *>( pointer->get_base() );
			assert( function );

			if ( Expression *newExpr = handleIntrinsics( appExpr ) ) {
				return newExpr;
			} // if

			Expression *ret = appExpr;

			std::list< Expression *>::iterator arg = appExpr->get_args().begin();
			std::list< Expression *>::iterator paramBegin = appExpr->get_args().begin();

			TyVarMap exprTyVars( (TypeDecl::Kind)-1 );
			makeTyVarMap( function, exprTyVars );
			ReferenceToType *polyRetType = isPolyRet( function );

			if ( polyRetType ) {
				ret = addPolyRetParam( appExpr, function, polyRetType, arg );
			} else if ( needsAdapter( function, scopeTyVars ) ) {
				// std::cerr << "needs adapter: ";
				// for ( TyVarMap::iterator i = scopeTyVars.begin(); i != scopeTyVars.end(); ++i ) {
				// 	std::cerr << i->first << " ";
				// }
				// std::cerr << "\n";
				// change the application so it calls the adapter rather than the passed function
				ret = applyAdapter( appExpr, function, arg, scopeTyVars );
			} // if
			arg = appExpr->get_args().begin();

			passTypeVars( appExpr, polyRetType, arg, exprTyVars );
			addInferredParams( appExpr, function, arg, exprTyVars );

			arg = paramBegin;

			boxParams( appExpr, function, arg, exprTyVars );
			passAdapters( appExpr, function, exprTyVars );

			return ret;
		}

		Expression *Pass1::mutate( UntypedExpr *expr ) {
			if ( ! expr->get_results().empty() && isPolyType( expr->get_results().front(), scopeTyVars, env ) ) {
				if ( NameExpr *name = dynamic_cast< NameExpr *>( expr->get_function() ) ) {
					if ( name->get_name() == "*?" ) {
						Expression *ret = expr->get_args().front();
						expr->get_args().clear();
						delete expr;
						return ret->acceptMutator( *this );
					} // if
				} // if
			} // if
			return PolyMutator::mutate( expr );
		}

		Expression *Pass1::mutate( AddressExpr *addrExpr ) {
			assert( ! addrExpr->get_arg()->get_results().empty() );

			bool needs = false;
			if ( UntypedExpr *expr = dynamic_cast< UntypedExpr *>( addrExpr->get_arg() ) ) {
				if ( ! expr->get_results().empty() && isPolyType( expr->get_results().front(), scopeTyVars, env ) ) {
					if ( NameExpr *name = dynamic_cast< NameExpr *>( expr->get_function() ) ) {
						if ( name->get_name() == "*?" ) {
							if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( expr->get_args().front() ) ) {
								assert( ! appExpr->get_function()->get_results().empty() );
								PointerType *pointer = dynamic_cast< PointerType *>( appExpr->get_function()->get_results().front() );
								assert( pointer );
								FunctionType *function = dynamic_cast< FunctionType *>( pointer->get_base() );
								assert( function );
								needs = needsAdapter( function, scopeTyVars );
							} // if
						} // if
					} // if
				} // if
			} // if
			addrExpr->set_arg( mutateExpression( addrExpr->get_arg() ) );
			if ( isPolyType( addrExpr->get_arg()->get_results().front(), scopeTyVars, env ) || needs ) {
				Expression *ret = addrExpr->get_arg();
				delete ret->get_results().front();
				ret->get_results().front() = addrExpr->get_results().front()->clone();
				addrExpr->set_arg( 0 );
				delete addrExpr;
				return ret;
			} else {
				return addrExpr;
			} // if
		}

		/// Wraps a function declaration in a new pointer-to-function variable expression
		VariableExpr *wrapFunctionDecl( DeclarationWithType *functionDecl ) {
			// line below cloned from FixFunction.cc
			ObjectDecl *functionObj = new ObjectDecl( functionDecl->get_name(), functionDecl->get_storageClass(), functionDecl->get_linkage(), 0,
			                                          new PointerType( Type::Qualifiers(), functionDecl->get_type()->clone() ), 0 );
			functionObj->set_mangleName( functionDecl->get_mangleName() );
			return new VariableExpr( functionObj );
		}
		
		Statement * Pass1::mutate( ReturnStmt *returnStmt ) {
			if ( retval && returnStmt->get_expr() ) {
				assert( ! returnStmt->get_expr()->get_results().empty() );
				// ***** Code Removal ***** After introducing a temporary variable for all return expressions, the following code appears superfluous.
				// if ( returnStmt->get_expr()->get_results().front()->get_isLvalue() ) {
				// by this point, a cast expr on a polymorphic return value is redundant
				while ( CastExpr *castExpr = dynamic_cast< CastExpr *>( returnStmt->get_expr() ) ) {
					returnStmt->set_expr( castExpr->get_arg() );
					returnStmt->get_expr()->set_env( castExpr->get_env() );
					castExpr->set_env( 0 );
					castExpr->set_arg( 0 );
					delete castExpr;
				} //while

				// find assignment operator for (polymorphic) return type
				ApplicationExpr *assignExpr = 0;
				if ( TypeInstType *typeInst = dynamic_cast< TypeInstType *>( retval->get_type() ) ) {
					// find assignment operator for type variable
					ScopedMap< std::string, DeclarationWithType *>::const_iterator assignIter = assignOps.find( typeInst->get_name() );
					if ( assignIter == assignOps.end() ) {
						throw SemanticError( "Attempt to return dtype or ftype object in ", returnStmt->get_expr() );
					} // if
					assignExpr = new ApplicationExpr( new VariableExpr( assignIter->second ) );
				} else if ( ReferenceToType *refType = dynamic_cast< ReferenceToType *>( retval->get_type() ) ) {
					// find assignment operator for generic type
					DeclarationWithType *functionDecl = scopedAssignOps.find( refType );
					if ( ! functionDecl ) {
						throw SemanticError( "Attempt to return dtype or ftype generic object in ", returnStmt->get_expr() );
					}

					// wrap it up in an application expression
					assignExpr = new ApplicationExpr( wrapFunctionDecl( functionDecl ) );
					assignExpr->set_env( env->clone() );

					// find each of its needed secondary assignment operators
					std::list< Expression* > &tyParams = refType->get_parameters();
					std::list< TypeDecl* > &forallParams = functionDecl->get_type()->get_forall();
					std::list< Expression* >::const_iterator tyIt = tyParams.begin();
					std::list< TypeDecl* >::const_iterator forallIt = forallParams.begin();
					for ( ; tyIt != tyParams.end() && forallIt != forallParams.end(); ++tyIt, ++forallIt ) {
						// Add appropriate mapping to assignment expression environment
						TypeExpr *formalTypeExpr = dynamic_cast< TypeExpr* >( *tyIt );
						assert( formalTypeExpr && "type parameters must be type expressions" );
						Type *formalType = formalTypeExpr->get_type();
						assignExpr->get_env()->add( (*forallIt)->get_name(), formalType );

						// skip types with no assign op (ftype/dtype)
						if ( (*forallIt)->get_kind() != TypeDecl::Any ) continue;

						// find assignment operator for formal type
						DeclarationWithType *assertAssign = 0;
						if ( TypeInstType *formalTypeInstType = dynamic_cast< TypeInstType* >( formalType ) ) {
							ScopedMap< std::string, DeclarationWithType *>::const_iterator assertAssignIt = assignOps.find( formalTypeInstType->get_name() );
							if ( assertAssignIt == assignOps.end() ) {
								throw SemanticError( "No assignment operation found for ", formalTypeInstType );
							}
							assertAssign = assertAssignIt->second;
						} else {
							assertAssign = scopedAssignOps.find( formalType );
							if ( ! assertAssign ) {
								throw SemanticError( "No assignment operation found for ", formalType );
							}
						}

						// add inferred parameter for field assignment operator to assignment expression
						std::list< DeclarationWithType* > &asserts = (*forallIt)->get_assertions();
						assert( ! asserts.empty() && "Type param needs assignment operator assertion" );
						DeclarationWithType *actualDecl = asserts.front();
						assignExpr->get_inferParams()[ actualDecl->get_uniqueId() ]
							= ParamEntry( assertAssign->get_uniqueId(), assertAssign->get_type()->clone(), actualDecl->get_type()->clone(), wrapFunctionDecl( assertAssign ) );
					}
				}
				assert( assignExpr );

				// replace return statement with appropriate assignment to out parameter
				Expression *retParm = new NameExpr( retval->get_name() );
				retParm->get_results().push_back( new PointerType( Type::Qualifiers(), retval->get_type()->clone() ) );
				assignExpr->get_args().push_back( retParm );
				assignExpr->get_args().push_back( returnStmt->get_expr() );
				stmtsToAdd.push_back( new ExprStmt( noLabels, mutateExpression( assignExpr ) ) );
				// } else {
				// 	useRetval = true;
				// 	stmtsToAdd.push_back( new ExprStmt( noLabels, mutateExpression( returnStmt->get_expr() ) ) );
				// 	useRetval = false;
				// } // if
				returnStmt->set_expr( 0 );
			} else {
				returnStmt->set_expr( mutateExpression( returnStmt->get_expr() ) );
			} // if
			return returnStmt;
		}

		Type * Pass1::mutate( PointerType *pointerType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars.endScope();
			return ret;
		}

		Type * Pass1::mutate( FunctionType *functionType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( functionType, scopeTyVars );

			Type *ret = Mutator::mutate( functionType );

			scopeTyVars.endScope();
			return ret;
		}

		void Pass1::doBeginScope() {
			adapters.beginScope();
			scopedAssignOps.beginScope();
		}

		void Pass1::doEndScope() {
			adapters.endScope();
			scopedAssignOps.endScope();
		}

////////////////////////////////////////// Pass2 ////////////////////////////////////////////////////

		void Pass2::addAdapters( FunctionType *functionType ) {
			std::list< DeclarationWithType *> &paramList = functionType->get_parameters();
			std::list< FunctionType *> functions;
			for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
				Type *orig = (*arg)->get_type();
				findAndReplaceFunction( orig, functions, scopeTyVars, needsAdapter );
				(*arg)->set_type( orig );
			}
			std::set< std::string > adaptersDone;
			for ( std::list< FunctionType *>::iterator funType = functions.begin(); funType != functions.end(); ++funType ) {
				std::string mangleName = mangleAdapterName( *funType, scopeTyVars );
				if ( adaptersDone.find( mangleName ) == adaptersDone.end() ) {
					std::string adapterName = makeAdapterName( mangleName );
					paramList.push_front( new ObjectDecl( adapterName, DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new PointerType( Type::Qualifiers(), makeAdapterType( *funType, scopeTyVars ) ), 0 ) );
					adaptersDone.insert( adaptersDone.begin(), mangleName );
				}
			}
//  deleteAll( functions );
		}

		template< typename DeclClass >
		DeclClass * Pass2::handleDecl( DeclClass *decl, Type *type ) {
			DeclClass *ret = static_cast< DeclClass *>( Mutator::mutate( decl ) );

			return ret;
		}

		DeclarationWithType * Pass2::mutate( FunctionDecl *functionDecl ) {
			return handleDecl( functionDecl, functionDecl->get_functionType() );
		}

		ObjectDecl * Pass2::mutate( ObjectDecl *objectDecl ) {
			return handleDecl( objectDecl, objectDecl->get_type() );
		}

		TypeDecl * Pass2::mutate( TypeDecl *typeDecl ) {
			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			if ( typeDecl->get_base() ) {
				return handleDecl( typeDecl, typeDecl->get_base() );
			} else {
				return Mutator::mutate( typeDecl );
			}
		}

		TypedefDecl * Pass2::mutate( TypedefDecl *typedefDecl ) {
			return handleDecl( typedefDecl, typedefDecl->get_base() );
		}

		Type * Pass2::mutate( PointerType *pointerType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars.endScope();
			return ret;
		}

		Type *Pass2::mutate( FunctionType *funcType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( funcType, scopeTyVars );

			// move polymorphic return type to parameter list
			if ( isPolyRet( funcType ) ) {
				DeclarationWithType *ret = funcType->get_returnVals().front();
				ret->set_type( new PointerType( Type::Qualifiers(), ret->get_type() ) );
				funcType->get_parameters().push_front( ret );
				funcType->get_returnVals().pop_front();
			}

			// add size/align and assertions for type parameters to parameter list
			std::list< DeclarationWithType *>::iterator last = funcType->get_parameters().begin();
			std::list< DeclarationWithType *> inferredParams;
			ObjectDecl newObj( "", DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), 0 );
			ObjectDecl newPtr( "", DeclarationNode::NoStorageClass, LinkageSpec::C, 0,
			                   new PointerType( Type::Qualifiers(), new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) ), 0 );
			for ( std::list< TypeDecl *>::const_iterator tyParm = funcType->get_forall().begin(); tyParm != funcType->get_forall().end(); ++tyParm ) {
				ObjectDecl *sizeParm, *alignParm;
				// add all size and alignment parameters to parameter list
				if ( (*tyParm)->get_kind() == TypeDecl::Any ) {
					TypeInstType parmType( Type::Qualifiers(), (*tyParm)->get_name(), *tyParm );
					std::string parmName = mangleType( &parmType );

					sizeParm = newObj.clone();
					sizeParm->set_name( sizeofName( parmName ) );
					last = funcType->get_parameters().insert( last, sizeParm );
					++last;

					alignParm = newObj.clone();
					alignParm->set_name( alignofName( parmName ) );
					last = funcType->get_parameters().insert( last, alignParm );
					++last;
				}
				// move all assertions into parameter list
				for ( std::list< DeclarationWithType *>::iterator assert = (*tyParm)->get_assertions().begin(); assert != (*tyParm)->get_assertions().end(); ++assert ) {
//      *assert = (*assert)->acceptMutator( *this );
					inferredParams.push_back( *assert );
				}
				(*tyParm)->get_assertions().clear();
			}

			// add size/align for generic parameter types to parameter list
			std::set< std::string > seenTypes; // sizeofName for generic types we've seen
			for ( std::list< DeclarationWithType* >::const_iterator fnParm = last; fnParm != funcType->get_parameters().end(); ++fnParm ) {
				Type *polyType = isPolyType( (*fnParm)->get_type(), scopeTyVars );
				if ( polyType && ! dynamic_cast< TypeInstType* >( polyType ) ) {
					std::string typeName = mangleType( polyType );
					if ( seenTypes.count( typeName ) ) continue;

					ObjectDecl *sizeParm, *alignParm, *offsetParm;
					sizeParm = newObj.clone();
					sizeParm->set_name( sizeofName( typeName ) );
					last = funcType->get_parameters().insert( last, sizeParm );
					++last;

					alignParm = newObj.clone();
					alignParm->set_name( alignofName( typeName ) );
					last = funcType->get_parameters().insert( last, alignParm );
					++last;

					if ( StructInstType *polyBaseStruct = dynamic_cast< StructInstType* >( polyType ) ) {
						// NOTE zero-length arrays are illegal in C, so empty structs have no offset array
						if ( ! polyBaseStruct->get_baseStruct()->get_members().empty() ) {
							offsetParm = newPtr.clone();
							offsetParm->set_name( offsetofName( typeName ) );
							last = funcType->get_parameters().insert( last, offsetParm );
							++last;
						}
					}

					seenTypes.insert( typeName );
				}
			}

			// splice assertion parameters into parameter list
			funcType->get_parameters().splice( last, inferredParams );
			addAdapters( funcType );
			mutateAll( funcType->get_returnVals(), *this );
			mutateAll( funcType->get_parameters(), *this );

			scopeTyVars.endScope();
			return funcType;
		}

//////////////////////////////////////// GenericInstantiator //////////////////////////////////////////////////

		/// Makes substitutions of params into baseParams; returns true if all parameters substituted for a concrete type
		bool makeSubstitutions( const std::list< TypeDecl* >& baseParams, const std::list< Expression* >& params, std::list< TypeExpr* >& out ) {
			bool allConcrete = true;  // will finish the substitution list even if they're not all concrete

			// substitute concrete types for given parameters, and incomplete types for placeholders
			std::list< TypeDecl* >::const_iterator baseParam = baseParams.begin();
			std::list< Expression* >::const_iterator param = params.begin();
			for ( ; baseParam != baseParams.end() && param != params.end(); ++baseParam, ++param ) {
	// 			switch ( (*baseParam)->get_kind() ) {
	// 			case TypeDecl::Any: {   // any type is a valid substitution here; complete types can be used to instantiate generics
					TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
					assert(paramType && "Aggregate parameters should be type expressions");
					out.push_back( paramType->clone() );
					// check that the substituted type isn't a type variable itself
					if ( dynamic_cast< TypeInstType* >( paramType->get_type() ) ) {
						allConcrete = false;
					}
	// 				break;
	// 			}
	// 			case TypeDecl::Dtype:  // dtype can be consistently replaced with void [only pointers, which become void*]
	// 				out.push_back( new TypeExpr( new VoidType( Type::Qualifiers() ) ) );
	// 				break;
	// 			case TypeDecl::Ftype:  // pointer-to-ftype can be consistently replaced with void (*)(void) [similar to dtype]
	// 				out.push_back( new TypeExpr( new FunctionType( Type::Qualifiers(), false ) ) );
	// 				break;
	// 			}
			}

			// if any parameters left over, not done
			if ( baseParam != baseParams.end() ) return false;
	// 		// if not enough parameters given, substitute remaining incomplete types for placeholders
	// 		for ( ; baseParam != baseParams.end(); ++baseParam ) {
	// 			switch ( (*baseParam)->get_kind() ) {
	// 			case TypeDecl::Any:    // no more substitutions here, fail early
	// 				return false;
	// 			case TypeDecl::Dtype:  // dtype can be consistently replaced with void [only pointers, which become void*]
	// 				out.push_back( new TypeExpr( new VoidType( Type::Qualifiers() ) ) );
	// 				break;
	// 			case TypeDecl::Ftype:  // pointer-to-ftype can be consistently replaced with void (*)(void) [similar to dtype]
	// 				out.push_back( new TypeExpr( new FunctionType( Type::Qualifiers(), false ) ) );
	// 				break;
	// 			}
	// 		}

			return allConcrete;
		}

		/// Substitutes types of members of in according to baseParams => typeSubs, appending the result to out
		void substituteMembers( const std::list< Declaration* >& in, const std::list< TypeDecl* >& baseParams, const std::list< TypeExpr* >& typeSubs,
								std::list< Declaration* >& out ) {
			// substitute types into new members
			TypeSubstitution subs( baseParams.begin(), baseParams.end(), typeSubs.begin() );
			for ( std::list< Declaration* >::const_iterator member = in.begin(); member != in.end(); ++member ) {
				Declaration *newMember = (*member)->clone();
				subs.apply(newMember);
				out.push_back( newMember );
			}
		}

		Type* GenericInstantiator::mutate( StructInstType *inst ) {
			// mutate subtypes
			Type *mutated = Mutator::mutate( inst );
			inst = dynamic_cast< StructInstType* >( mutated );
			if ( ! inst ) return mutated;

			// exit early if no need for further mutation
			if ( inst->get_parameters().empty() ) return inst;
			assert( inst->get_baseParameters() && "Base struct has parameters" );

			// check if type can be concretely instantiated; put substitutions into typeSubs
			std::list< TypeExpr* > typeSubs;
			if ( ! makeSubstitutions( *inst->get_baseParameters(), inst->get_parameters(), typeSubs ) ) {
				deleteAll( typeSubs );
				return inst;
			}

			// make concrete instantiation of generic type
			StructDecl *concDecl = lookup( inst, typeSubs );
			if ( ! concDecl ) {
				// set concDecl to new type, insert type declaration into statements to add
				concDecl = new StructDecl( typeNamer.newName( inst->get_name() ) );
				substituteMembers( inst->get_baseStruct()->get_members(), *inst->get_baseParameters(), typeSubs, 	concDecl->get_members() );
				DeclMutator::addDeclaration( concDecl );
				insert( inst, typeSubs, concDecl );
			}
			StructInstType *newInst = new StructInstType( inst->get_qualifiers(), concDecl->get_name() );
			newInst->set_baseStruct( concDecl );

			deleteAll( typeSubs );
			delete inst;
			return newInst;
		}

		Type* GenericInstantiator::mutate( UnionInstType *inst ) {
			// mutate subtypes
			Type *mutated = Mutator::mutate( inst );
			inst = dynamic_cast< UnionInstType* >( mutated );
			if ( ! inst ) return mutated;

			// exit early if no need for further mutation
			if ( inst->get_parameters().empty() ) return inst;
			assert( inst->get_baseParameters() && "Base union has parameters" );

			// check if type can be concretely instantiated; put substitutions into typeSubs
			std::list< TypeExpr* > typeSubs;
			if ( ! makeSubstitutions( *inst->get_baseParameters(), inst->get_parameters(), typeSubs ) ) {
				deleteAll( typeSubs );
				return inst;
			}

			// make concrete instantiation of generic type
			UnionDecl *concDecl = lookup( inst, typeSubs );
			if ( ! concDecl ) {
				// set concDecl to new type, insert type declaration into statements to add
				concDecl = new UnionDecl( typeNamer.newName( inst->get_name() ) );
				substituteMembers( inst->get_baseUnion()->get_members(), *inst->get_baseParameters(), typeSubs, concDecl->get_members() );
				DeclMutator::addDeclaration( concDecl );
				insert( inst, typeSubs, concDecl );
			}
			UnionInstType *newInst = new UnionInstType( inst->get_qualifiers(), concDecl->get_name() );
			newInst->set_baseUnion( concDecl );

			deleteAll( typeSubs );
			delete inst;
			return newInst;
		}

	// 	/// Gets the base struct or union declaration for a member expression; NULL if not applicable
	// 	AggregateDecl* getMemberBaseDecl( MemberExpr *memberExpr ) {
	// 		// get variable for member aggregate
	// 		VariableExpr *varExpr = dynamic_cast< VariableExpr* >( memberExpr->get_aggregate() );
	// 		if ( ! varExpr ) return NULL;
	//
	// 		// get object for variable
	// 		ObjectDecl *objectDecl = dynamic_cast< ObjectDecl* >( varExpr->get_var() );
	// 		if ( ! objectDecl ) return NULL;
	//
	// 		// get base declaration from object type
	// 		Type *objectType = objectDecl->get_type();
	// 		StructInstType *structType = dynamic_cast< StructInstType* >( objectType );
	// 		if ( structType ) return structType->get_baseStruct();
	// 		UnionInstType *unionType = dynamic_cast< UnionInstType* >( objectType );
	// 		if ( unionType ) return unionType->get_baseUnion();
	//
	// 		return NULL;
	// 	}
	//
	// 	/// Finds the declaration with the given name, returning decls.end() if none such
	// 	std::list< Declaration* >::const_iterator findDeclNamed( const std::list< Declaration* > &decls, const std::string &name ) {
	// 		for( std::list< Declaration* >::const_iterator decl = decls.begin(); decl != decls.end(); ++decl ) {
	// 			if ( (*decl)->get_name() == name ) return decl;
	// 		}
	// 		return decls.end();
	// 	}
	//
	// 	Expression* Instantiate::mutate( MemberExpr *memberExpr ) {
	// 		// mutate, exiting early if no longer MemberExpr
	// 		Expression *expr = Mutator::mutate( memberExpr );
	// 		memberExpr = dynamic_cast< MemberExpr* >( expr );
	// 		if ( ! memberExpr ) return expr;
	//
	// 		// get declaration of member and base declaration of member, exiting early if not found
	// 		AggregateDecl *memberBase = getMemberBaseDecl( memberExpr );
	// 		if ( ! memberBase ) return memberExpr;
	// 		DeclarationWithType *memberDecl = memberExpr->get_member();
	// 		std::list< Declaration* >::const_iterator baseIt = findDeclNamed( memberBase->get_members(), memberDecl->get_name() );
	// 		if ( baseIt == memberBase->get_members().end() ) return memberExpr;
	// 		DeclarationWithType *baseDecl = dynamic_cast< DeclarationWithType* >( *baseIt );
	// 		if ( ! baseDecl ) return memberExpr;
	//
	// 		// check if stated type of the member is not the type of the member's declaration; if so, need a cast
	// 		// this *SHOULD* be safe, I don't think anything but the void-replacements I put in for dtypes would make it past the typechecker
	// 		SymTab::Indexer dummy;
	// 		if ( ResolvExpr::typesCompatible( memberDecl->get_type(), baseDecl->get_type(), dummy ) ) return memberExpr;
	// 		else return new CastExpr( memberExpr, memberDecl->get_type() );
	// 	}

		void GenericInstantiator::doBeginScope() {
			DeclMutator::doBeginScope();
			instantiations.beginScope();
		}

		void GenericInstantiator::doEndScope() {
			DeclMutator::doEndScope();
			instantiations.endScope();
		}

////////////////////////////////////////// PolyGenericCalculator ////////////////////////////////////////////////////

		template< typename DeclClass >
		DeclClass * PolyGenericCalculator::handleDecl( DeclClass *decl, Type *type ) {
			scopeTyVars.beginScope();
			makeTyVarMap( type, scopeTyVars );

			DeclClass *ret = static_cast< DeclClass *>( Mutator::mutate( decl ) );

			scopeTyVars.endScope();
			return ret;
		}

		ObjectDecl * PolyGenericCalculator::mutate( ObjectDecl *objectDecl ) {
			return handleDecl( objectDecl, objectDecl->get_type() );
		}

		DeclarationWithType * PolyGenericCalculator::mutate( FunctionDecl *functionDecl ) {
			return handleDecl( functionDecl, functionDecl->get_functionType() );
		}

		TypedefDecl * PolyGenericCalculator::mutate( TypedefDecl *typedefDecl ) {
			return handleDecl( typedefDecl, typedefDecl->get_base() );
		}

		TypeDecl * PolyGenericCalculator::mutate( TypeDecl *typeDecl ) {
			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			return Mutator::mutate( typeDecl );
		}

		Type * PolyGenericCalculator::mutate( PointerType *pointerType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars.endScope();
			return ret;
		}

		Type * PolyGenericCalculator::mutate( FunctionType *funcType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( funcType, scopeTyVars );

			// make sure that any type information passed into the function is accounted for
			for ( std::list< DeclarationWithType* >::const_iterator fnParm = funcType->get_parameters().begin(); fnParm != funcType->get_parameters().end(); ++fnParm ) {
				// condition here duplicates that in Pass2::mutate( FunctionType* )
				Type *polyType = isPolyType( (*fnParm)->get_type(), scopeTyVars );
				if ( polyType && ! dynamic_cast< TypeInstType* >( polyType ) ) {
					knownLayouts.insert( mangleType( polyType ) );
				}
			}
			
			Type *ret = Mutator::mutate( funcType );

			scopeTyVars.endScope();
			return ret;
		}

		Statement *PolyGenericCalculator::mutate( DeclStmt *declStmt ) {
			if ( ObjectDecl *objectDecl = dynamic_cast< ObjectDecl *>( declStmt->get_decl() ) ) {
				if ( findGeneric( objectDecl->get_type() ) ) {
					// change initialization of a polymorphic value object
					// to allocate storage with alloca
					Type *declType = objectDecl->get_type();
					UntypedExpr *alloc = new UntypedExpr( new NameExpr( "__builtin_alloca" ) );
					alloc->get_args().push_back( new NameExpr( sizeofName( mangleType( declType ) ) ) );

					delete objectDecl->get_init();

					std::list<Expression*> designators;
					objectDecl->set_init( new SingleInit( alloc, designators ) );
				}
			}
			return Mutator::mutate( declStmt );
		}

		/// Finds the member in the base list that matches the given declaration; returns its index, or -1 if not present
		long findMember( DeclarationWithType *memberDecl, std::list< Declaration* > &baseDecls ) {
			long i = 0;
			for(std::list< Declaration* >::const_iterator decl = baseDecls.begin(); decl != baseDecls.end(); ++decl, ++i ) {
				if ( memberDecl->get_name() != (*decl)->get_name() ) continue;

				if ( DeclarationWithType *declWithType = dynamic_cast< DeclarationWithType* >( *decl ) ) {
					if ( memberDecl->get_mangleName().empty() || declWithType->get_mangleName().empty()
					     || memberDecl->get_mangleName() == declWithType->get_mangleName() ) return i;
					else continue;
				} else return i;
			}
			return -1;
		}

		/// Returns an index expression into the offset array for a type
		Expression *makeOffsetIndex( Type *objectType, long i ) {
			std::stringstream offset_namer;
			offset_namer << i;
			ConstantExpr *fieldIndex = new ConstantExpr( Constant( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), offset_namer.str() ) );
			UntypedExpr *fieldOffset = new UntypedExpr( new NameExpr( "?[?]" ) );
			fieldOffset->get_args().push_back( new NameExpr( offsetofName( mangleType( objectType ) ) ) );
			fieldOffset->get_args().push_back( fieldIndex );
			return fieldOffset;
		}

		/// Returns an expression dereferenced n times
		Expression *makeDerefdVar( Expression *derefdVar, long n ) {
			for ( int i = 1; i < n; ++i ) {
				UntypedExpr *derefExpr = new UntypedExpr( new NameExpr( "*?" ) );
				derefExpr->get_args().push_back( derefdVar );
				derefdVar = derefExpr;
			}
			return derefdVar;
		}
		
		Expression *PolyGenericCalculator::mutate( MemberExpr *memberExpr ) {
			// mutate, exiting early if no longer MemberExpr
			Expression *expr = Mutator::mutate( memberExpr );
			memberExpr = dynamic_cast< MemberExpr* >( expr );
			if ( ! memberExpr ) return expr;

			// get declaration for base struct, exiting early if not found
			int varDepth;
			VariableExpr *varExpr = getBaseVar( memberExpr->get_aggregate(), &varDepth );
			if ( ! varExpr ) return memberExpr;
			ObjectDecl *objectDecl = dynamic_cast< ObjectDecl* >( varExpr->get_var() );
			if ( ! objectDecl ) return memberExpr;

			// only mutate member expressions for polymorphic types
			int tyDepth;
			Type *objectType = hasPolyBase( objectDecl->get_type(), scopeTyVars, &tyDepth );
			if ( ! objectType ) return memberExpr;
			findGeneric( objectType ); // ensure layout for this type is available

			Expression *newMemberExpr = 0;
			if ( StructInstType *structType = dynamic_cast< StructInstType* >( objectType ) ) {
				// look up offset index
				long i = findMember( memberExpr->get_member(), structType->get_baseStruct()->get_members() );
				if ( i == -1 ) return memberExpr;

				// replace member expression with pointer to base plus offset
				UntypedExpr *fieldLoc = new UntypedExpr( new NameExpr( "?+?" ) );
				fieldLoc->get_args().push_back( makeDerefdVar( varExpr->clone(), varDepth ) );
				fieldLoc->get_args().push_back( makeOffsetIndex( objectType, i ) );
				newMemberExpr = fieldLoc;
			} else if ( dynamic_cast< UnionInstType* >( objectType ) ) {
				// union members are all at offset zero, so build appropriately-dereferenced variable
				newMemberExpr = makeDerefdVar( varExpr->clone(), varDepth );
			} else return memberExpr;
			assert( newMemberExpr );

			Type *memberType = memberExpr->get_member()->get_type();
			if ( ! isPolyType( memberType, scopeTyVars ) ) {
				// Not all members of a polymorphic type are themselves of polymorphic type; in this case the member expression should be wrapped and dereferenced to form an lvalue
				CastExpr *ptrCastExpr = new CastExpr( newMemberExpr, new PointerType( Type::Qualifiers(), memberType->clone() ) );
				UntypedExpr *derefExpr = new UntypedExpr( new NameExpr( "*?" ) );
				derefExpr->get_args().push_back( ptrCastExpr );
				newMemberExpr = derefExpr;
			}

			delete memberExpr;
			return newMemberExpr;
		}

		ObjectDecl *PolyGenericCalculator::makeVar( const std::string &name, Type *type, Initializer *init ) {
			ObjectDecl *newObj = new ObjectDecl( name, DeclarationNode::NoStorageClass, LinkageSpec::C, 0, type, init );
			stmtsToAdd.push_back( new DeclStmt( noLabels, newObj ) );
			return newObj;
		}

		void PolyGenericCalculator::addOtypeParamsToLayoutCall( UntypedExpr *layoutCall, const std::list< Type* > &otypeParams ) {
			for ( std::list< Type* >::const_iterator param = otypeParams.begin(); param != otypeParams.end(); ++param ) {
				if ( findGeneric( *param ) ) {
					// push size/align vars for a generic parameter back
					std::string paramName = mangleType( *param );
					layoutCall->get_args().push_back( new NameExpr( sizeofName( paramName ) ) );
					layoutCall->get_args().push_back( new NameExpr( alignofName( paramName ) ) );
				} else {
					layoutCall->get_args().push_back( new SizeofExpr( (*param)->clone() ) );
					layoutCall->get_args().push_back( new AlignofExpr( (*param)->clone() ) );
				}
			}
		}

		/// returns true if any of the otype parameters have a dynamic layout and puts all otype parameters in the output list
		bool findGenericParams( std::list< TypeDecl* > &baseParams, std::list< Expression* > &typeParams, std::list< Type* > &out ) {
			bool hasDynamicLayout = false;

			std::list< TypeDecl* >::const_iterator baseParam = baseParams.begin();
			std::list< Expression* >::const_iterator typeParam = typeParams.begin();
			for ( ; baseParam != baseParams.end() && typeParam != typeParams.end(); ++baseParam, ++typeParam ) {
				// skip non-otype parameters
				if ( (*baseParam)->get_kind() != TypeDecl::Any ) continue;
				TypeExpr *typeExpr = dynamic_cast< TypeExpr* >( *typeParam );
				assert( typeExpr && "all otype parameters should be type expressions" );

				Type *type = typeExpr->get_type();
				out.push_back( type );
				if ( isPolyType( type ) ) hasDynamicLayout = true;
			}
			assert( baseParam == baseParams.end() && typeParam == typeParams.end() );

			return hasDynamicLayout;
		}

		bool PolyGenericCalculator::findGeneric( Type *ty ) {
			if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( ty ) ) {
				// duplicate logic from isPolyType()
				if ( env ) {
					if ( Type *newType = env->lookup( typeInst->get_name() ) ) {
						return findGeneric( newType );
					} // if
				} // if
				if ( scopeTyVars.find( typeInst->get_name() ) != scopeTyVars.end() ) {
					// NOTE assumes here that getting put in the scopeTyVars included having the layout variables set
					return true;
				}
				return false;
			} else if ( StructInstType *structTy = dynamic_cast< StructInstType* >( ty ) ) {
				// check if this type already has a layout generated for it
				std::string typeName = mangleType( ty );
				if ( knownLayouts.find( typeName ) != knownLayouts.end() ) return true;

				// check if any of the type parameters have dynamic layout; if none do, this type is (or will be) monomorphized
				std::list< Type* > otypeParams;
				if ( ! findGenericParams( *structTy->get_baseParameters(), structTy->get_parameters(), otypeParams ) ) return false;

				// insert local variables for layout and generate call to layout function
				knownLayouts.insert( typeName );  // done early so as not to interfere with the later addition of parameters to the layout call
				Type *layoutType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );

				int n_members = structTy->get_baseStruct()->get_members().size();
				if ( n_members == 0 ) {
					// all empty structs have the same layout - size 1, align 1
					makeVar( sizeofName( typeName ), layoutType, new SingleInit( new ConstantExpr( Constant::from( (unsigned long)1 ) ) ) );
					makeVar( alignofName( typeName ), layoutType->clone(), new SingleInit( new ConstantExpr( Constant::from( (unsigned long)1 ) ) ) );
					// NOTE zero-length arrays are forbidden in C, so empty structs have no offsetof array
				} else {
					ObjectDecl *sizeVar = makeVar( sizeofName( typeName ), layoutType );
					ObjectDecl *alignVar = makeVar( alignofName( typeName ), layoutType->clone() );
					ObjectDecl *offsetVar = makeVar( offsetofName( typeName ), new ArrayType( Type::Qualifiers(), layoutType->clone(), new ConstantExpr( Constant::from( n_members ) ), false, false ) );

					// generate call to layout function
					UntypedExpr *layoutCall = new UntypedExpr( new NameExpr( layoutofName( structTy->get_baseStruct() ) ) );
					layoutCall->get_args().push_back( new AddressExpr( new VariableExpr( sizeVar ) ) );
					layoutCall->get_args().push_back( new AddressExpr( new VariableExpr( alignVar ) ) );
					layoutCall->get_args().push_back( new VariableExpr( offsetVar ) );
					addOtypeParamsToLayoutCall( layoutCall, otypeParams );

					stmtsToAdd.push_back( new ExprStmt( noLabels, layoutCall ) );
				}

				return true;
			} else if ( UnionInstType *unionTy = dynamic_cast< UnionInstType* >( ty ) ) {
				// check if this type already has a layout generated for it
				std::string typeName = mangleType( ty );
				if ( knownLayouts.find( typeName ) != knownLayouts.end() ) return true;

				// check if any of the type parameters have dynamic layout; if none do, this type is (or will be) monomorphized
				std::list< Type* > otypeParams;
				if ( ! findGenericParams( *unionTy->get_baseParameters(), unionTy->get_parameters(), otypeParams ) ) return false;

				// insert local variables for layout and generate call to layout function
				knownLayouts.insert( typeName );  // done early so as not to interfere with the later addition of parameters to the layout call
				Type *layoutType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );

				ObjectDecl *sizeVar = makeVar( sizeofName( typeName ), layoutType );
				ObjectDecl *alignVar = makeVar( alignofName( typeName ), layoutType->clone() );

				// generate call to layout function
				UntypedExpr *layoutCall = new UntypedExpr( new NameExpr( layoutofName( unionTy->get_baseUnion() ) ) );
				layoutCall->get_args().push_back( new AddressExpr( new VariableExpr( sizeVar ) ) );
				layoutCall->get_args().push_back( new AddressExpr( new VariableExpr( alignVar ) ) );
				addOtypeParamsToLayoutCall( layoutCall, otypeParams );

				stmtsToAdd.push_back( new ExprStmt( noLabels, layoutCall ) );

				return true;
			}

			return false;
		}

		Expression *PolyGenericCalculator::mutate( SizeofExpr *sizeofExpr ) {
			Type *ty = sizeofExpr->get_type();
			if ( findGeneric( ty ) ) {
				Expression *ret = new NameExpr( sizeofName( mangleType( ty ) ) );
				delete sizeofExpr;
				return ret;
			}
			return sizeofExpr;
		}

		Expression *PolyGenericCalculator::mutate( AlignofExpr *alignofExpr ) {
			Type *ty = alignofExpr->get_type();
			if ( findGeneric( ty ) ) {
				Expression *ret = new NameExpr( alignofName( mangleType( ty ) ) );
				delete alignofExpr;
				return ret;
			}
			return alignofExpr;
		}

		Expression *PolyGenericCalculator::mutate( OffsetofExpr *offsetofExpr ) {
			// mutate, exiting early if no longer OffsetofExpr
			Expression *expr = Mutator::mutate( offsetofExpr );
			offsetofExpr = dynamic_cast< OffsetofExpr* >( expr );
			if ( ! offsetofExpr ) return expr;

			// only mutate expressions for polymorphic structs/unions
			Type *ty = offsetofExpr->get_type();
			if ( ! findGeneric( ty ) ) return offsetofExpr;
			
			if ( StructInstType *structType = dynamic_cast< StructInstType* >( ty ) ) {
				// replace offsetof expression by index into offset array
				long i = findMember( offsetofExpr->get_member(), structType->get_baseStruct()->get_members() );
				if ( i == -1 ) return offsetofExpr;

				Expression *offsetInd = makeOffsetIndex( ty, i );
				delete offsetofExpr;
				return offsetInd;
			} else if ( dynamic_cast< UnionInstType* >( ty ) ) {
				// all union members are at offset zero
				delete offsetofExpr;
				return new ConstantExpr( Constant( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), std::string("0") ) );
			} else return offsetofExpr;
		}

		Expression *PolyGenericCalculator::mutate( OffsetPackExpr *offsetPackExpr ) {
			StructInstType *ty = offsetPackExpr->get_type();

			Expression *ret = 0;
			if ( findGeneric( ty ) ) {
				// pull offset back from generated type information
				ret = new NameExpr( offsetofName( mangleType( ty ) ) );
			} else {
				std::string offsetName = offsetofName( mangleType( ty ) );
				if ( knownOffsets.find( offsetName ) != knownOffsets.end() ) {
					// use the already-generated offsets for this type
					ret = new NameExpr( offsetName );
				} else {
					knownOffsets.insert( offsetName );

					std::list< Declaration* > &baseMembers = ty->get_baseStruct()->get_members();
					Type *offsetType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );

					// build initializer list for offset array
					std::list< Initializer* > inits;
					for ( std::list< Declaration* >::const_iterator member = baseMembers.begin(); member != baseMembers.end(); ++member ) {
						DeclarationWithType *memberDecl;
						if ( DeclarationWithType *origMember = dynamic_cast< DeclarationWithType* >( *member ) ) {
							memberDecl = origMember->clone();
						} else {
							memberDecl = new ObjectDecl( (*member)->get_name(), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, offsetType->clone(), 0 );
						}
						inits.push_back( new SingleInit( new OffsetofExpr( ty->clone(), memberDecl ) ) );
					}

					// build the offset array and replace the pack with a reference to it
					ObjectDecl *offsetArray = makeVar( offsetName, new ArrayType( Type::Qualifiers(), offsetType, new ConstantExpr( Constant::from( baseMembers.size() ) ), false, false ),
							new ListInit( inits ) );
					ret = new VariableExpr( offsetArray );
				}
			}

			delete offsetPackExpr;
			return ret;
		}

		void PolyGenericCalculator::doBeginScope() {
			knownLayouts.beginScope();
			knownOffsets.beginScope();
		}

		void PolyGenericCalculator::doEndScope() {
			knownLayouts.endScope();
			knownOffsets.endScope();
		}

////////////////////////////////////////// Pass3 ////////////////////////////////////////////////////

		template< typename DeclClass >
		DeclClass * Pass3::handleDecl( DeclClass *decl, Type *type ) {
			scopeTyVars.beginScope();
			makeTyVarMap( type, scopeTyVars );

			DeclClass *ret = static_cast< DeclClass *>( Mutator::mutate( decl ) );
			ScrubTyVars::scrub( decl, scopeTyVars );

			scopeTyVars.endScope();
			return ret;
		}

		ObjectDecl * Pass3::mutate( ObjectDecl *objectDecl ) {
			return handleDecl( objectDecl, objectDecl->get_type() );
		}

		DeclarationWithType * Pass3::mutate( FunctionDecl *functionDecl ) {
			return handleDecl( functionDecl, functionDecl->get_functionType() );
		}

		TypedefDecl * Pass3::mutate( TypedefDecl *typedefDecl ) {
			return handleDecl( typedefDecl, typedefDecl->get_base() );
		}

		TypeDecl * Pass3::mutate( TypeDecl *typeDecl ) {
//   Initializer *init = 0;
//   std::list< Expression *> designators;
//   scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
//   if ( typeDecl->get_base() ) {
//     init = new SimpleInit( new SizeofExpr( handleDecl( typeDecl, typeDecl->get_base() ) ), designators );
//   }
//   return new ObjectDecl( typeDecl->get_name(), Declaration::Extern, LinkageSpec::C, 0, new BasicType( Type::Qualifiers(), BasicType::UnsignedInt ), init );

			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			return Mutator::mutate( typeDecl );
		}

		Type * Pass3::mutate( PointerType *pointerType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars.endScope();
			return ret;
		}

		Type * Pass3::mutate( FunctionType *functionType ) {
			scopeTyVars.beginScope();
			makeTyVarMap( functionType, scopeTyVars );

			Type *ret = Mutator::mutate( functionType );

			scopeTyVars.endScope();
			return ret;
		}
	} // anonymous namespace
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
