//                              -*- Mode: CFA -*-
//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// kernel.c --
//
// Author           : Thierry Delisle
// Created On       : Tue Jan 17 12:27:26 2016
// Last Modified By : Thierry Delisle
// Last Modified On : --
// Update Count     : 0
//

//Header
#include "kernel"

//C Includes
#include <stddef.h>
extern "C" {
#include <sys/resource.h>
}

//CFA Includes
#include "libhdr.h"
#include "threads"

//Private includes
#define __CFA_INVOKE_PRIVATE__
#include "invoke.h"

processor * systemProcessor;
thread_h * mainThread;

void kernel_startup(void)  __attribute__((constructor(101)));
void kernel_shutdown(void) __attribute__((destructor(101)));

void ?{}(processor * this) {
	this->ctx = NULL;
	this->thread_index = 0;
	this->thread_count = 10;
	this->terminated = false;

	for(int i = 0; i < 10; i++) {
		this->threads[i] = NULL;
	}

	LIB_DEBUG_PRINTF("Processor : ctor for core %p (core spots %d)\n", this, this->thread_count);
}

void ^?{}(processor * this) {

}

//-----------------------------------------------------------------------------
// Processor coroutine
struct processorCtx_t {
	processor * proc;
	coroutine c;
};

DECL_COROUTINE(processorCtx_t)

void ?{}(processorCtx_t * this, processor * proc) {
	(&this->c){};
	this->proc = proc;
}

void CtxInvokeProcessor(processor * proc) {
	processorCtx_t proc_cor_storage = {proc};
	resume( &proc_cor_storage );
}

//-----------------------------------------------------------------------------
// Processor running routines
void main(processorCtx_t * ctx);
thread_h * nextThread(processor * this);
void runThread(processor * this, thread_h * dst);
void spin(processor * this, unsigned int * spin_count);

void main(processorCtx_t * ctx) {
	processor * this = ctx->proc;
	LIB_DEBUG_PRINTF("Kernel : core %p starting\n", this);

	thread_h * readyThread = NULL;
	for( unsigned int spin_count = 0; ! this->terminated; spin_count++ ) {
		
		readyThread = nextThread(this);

		if(readyThread) {
			runThread(this, readyThread);
			spin_count = 0;
		} else {
			spin(this, &spin_count);
		}		
	}

	LIB_DEBUG_PRINTF("Kernel : core %p terminated\n", this);
}

thread_h * nextThread(processor * this) {
	for(int i = 0; i < this->thread_count; i++) {
		this->thread_index = (this->thread_index + 1) % this->thread_count;	
		
		thread_h * thrd = this->threads[this->thread_index];
		if(thrd) return thrd;
	}

	return NULL;
}

void runThread(processor * this, thread_h * dst) {
	coroutine * proc_ctx = get_coroutine(this->ctx);
	coroutine * thrd_ctx = get_coroutine(dst);
	thrd_ctx->last = proc_ctx;

	// context switch to specified coroutine
	// Which is now the current_coroutine
	LIB_DEBUG_PRINTF("Kernel : switching to ctx %p (from %p, current %p)\n", thrd_ctx, proc_ctx, current_coroutine);
	current_coroutine = thrd_ctx;
	CtxSwitch( proc_ctx->stack.context, thrd_ctx->stack.context );
	current_coroutine = proc_ctx;
	LIB_DEBUG_PRINTF("Kernel : returned from ctx %p (to %p, current %p)\n", thrd_ctx, proc_ctx, current_coroutine);

	// when CtxSwitch returns we are back in the processor coroutine
}

void spin(processor * this, unsigned int * spin_count) {
	(*spin_count)++;
}

//-----------------------------------------------------------------------------
// Kernel runner (Temporary)

void scheduler_add( thread_h * thrd ) {
	LIB_DEBUG_PRINTF("Kernel : scheduling %p on core %p (%d spots)\n", thrd, systemProcessor, systemProcessor->thread_count);
	for(int i = 0; i < systemProcessor->thread_count; i++) {
		if(systemProcessor->threads[i] == NULL) {
			systemProcessor->threads[i] = thrd;
			return;
		}
	}
	assert(false);
}

void scheduler_remove( thread_h * thrd ) {
	LIB_DEBUG_PRINTF("Kernel : unscheduling %p from core %p\n", thrd, systemProcessor);
	for(int i = 0; i < systemProcessor->thread_count; i++) {
		if(systemProcessor->threads[i] == thrd) {
			systemProcessor->threads[i] = NULL;
			break;
		}
	}
	for(int i = 0; i < systemProcessor->thread_count; i++) {
		if(systemProcessor->threads[i] != NULL) {
			return;
		}
	}
	LIB_DEBUG_PRINTF("Kernel : terminating core %p\n", systemProcessor);	
	systemProcessor->terminated = true;
}

//-----------------------------------------------------------------------------
// Kernel storage
#define KERNEL_STORAGE(T,X) static char X##_storage[sizeof(T)]

KERNEL_STORAGE(processorCtx_t, systemProcessorCtx);
KERNEL_STORAGE(processor, systemProcessor);
KERNEL_STORAGE(thread_h, mainThread);
KERNEL_STORAGE(machine_context_t, mainThread_context);

//-----------------------------------------------------------------------------
// Main thread construction
struct mainThread_info_t {
	machine_context_t ctx;	
	unsigned int size;		// size of stack
	void *base;				// base of stack
	void *storage;			// pointer to stack
	void *limit;			// stack grows towards stack limit
	void *context;			// address of cfa_context_t
	void *top;				// address of top of storage
};

void ?{}( mainThread_info_t * this ) {
	CtxGet( &this->ctx );
	this->base = this->ctx.FP;
	this->storage = this->ctx.SP;

	rlimit r;
	int ret = getrlimit( RLIMIT_STACK, &r);
	this->size = r.rlim_cur;

	this->limit = (void *)(((intptr_t)this->base) - this->size);
	this->context = &mainThread_context_storage;
	this->top = this->base;
}

void ?{}( coStack_t * this, mainThread_info_t * info) {
	this->size = info->size;
	this->storage = info->storage;
	this->limit = info->limit;
	this->base = info->base;
	this->context = info->context;
	this->top = info->top;
	this->userStack = true;
}

void ?{}( coroutine * this, mainThread_info_t * info) {
	(&this->stack){ info };	
	this->name = "Main Thread";
	this->errno_ = 0;
	this->state = Inactive;
	this->notHalted = true;
}

void ?{}( thread_h * this, mainThread_info_t * info) {
	(&this->c){ info };
}

//-----------------------------------------------------------------------------
// Kernel boot procedures
void kernel_startup(void) {

	// SKULLDUGGERY: the mainThread steals the process main thread 
	// which will then be scheduled by the systemProcessor normally
	LIB_DEBUG_PRINTF("Kernel : Starting\n");	

	mainThread_info_t ctx;
	LIB_DEBUG_PRINTF("Kernel :    base : %p\n", ctx.base );
	LIB_DEBUG_PRINTF("Kernel :     top : %p\n", ctx.top );
	LIB_DEBUG_PRINTF("Kernel :   limit : %p\n", ctx.limit );
	LIB_DEBUG_PRINTF("Kernel :    size : %x\n", ctx.size );
	LIB_DEBUG_PRINTF("Kernel : storage : %p\n", ctx.storage );
	LIB_DEBUG_PRINTF("Kernel : context : %p\n", ctx.context );

	// Start by initializing the main thread
	mainThread = (thread_h *)&mainThread_storage;
	LIB_DEBUG_PRINTF("Kernel : Main thread : %p\n", mainThread );
	mainThread{ &ctx };

	// // Initialize the system processor
	systemProcessor = (processor *)&systemProcessor_storage;
	systemProcessor{};

	// Initialize the system processor ctx
	// (the coroutine that contains the processing control flow)
	systemProcessor->ctx = (processorCtx_t *)&systemProcessorCtx_storage;
	systemProcessor->ctx{ systemProcessor };

	scheduler_add(mainThread);

	current_coroutine = &mainThread->c;

	LIB_DEBUG_PRINTF("Kernel : Starting system processor\n");	
	resume(systemProcessor->ctx);

	LIB_DEBUG_PRINTF("Kernel : Started\n--------------------------------------------------\n\n");
}
void kernel_shutdown(void) {
	LIB_DEBUG_PRINTF("\n--------------------------------------------------\nKernel : Shutting down");

	LIB_DEBUG_PRINTF("Unscheduling main thread\n");
	scheduler_remove(mainThread);

	LIB_DEBUG_PRINTF("Suspending main\n");
	suspend();

	LIB_DEBUG_PRINTF("Kernel : Control return to initial process thread\n");

	^(systemProcessor->ctx){};
	^(systemProcessor){};

	^(mainThread){};

	LIB_DEBUG_PRINTF("Kernel : Shutdown complete\n");	
}

// Local Variables: //
// mode: c //
// tab-width: 4 //
// End: //
