//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Statement.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Jun  5 07:51:04 2015
// Update Count     : 15
//

#include <functional>
#include <algorithm>
#include <iostream>
#include <list>
#include <cassert>

#include "Statement.h"
#include "Expression.h"
#include "Declaration.h"
#include "Common/SemanticError.h"

using std::string;
using std::endl;

Statement::Statement( std::list<Label> _labels ) : labels(_labels ) {}

void Statement::print( std::ostream &, int indent ) {}

Statement::~Statement() {}

ExprStmt::ExprStmt( std::list<Label> _labels, Expression *_expr ) : Statement(_labels ), expr(_expr ) {}

ExprStmt::~ExprStmt() {}

void ExprStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Expression Statement:" << endl;
	expr->print( os, indent + 2 );
} 

const char *BranchStmt::brType[] = { "Goto", "Break", "Continue" };

BranchStmt::BranchStmt( std::list<Label> labels, Label _target, Type _type ) throw ( SemanticError ) :
	Statement( labels ), originalTarget(_target ), target(_target ), type(_type ) {
	//actually this is a syntactic error signaled by the parser
	if ( type == BranchStmt::Goto && target.size() == 0 )
		throw SemanticError("goto without target");
}

BranchStmt::BranchStmt( std::list<Label> labels, Expression *_computedTarget, Type _type ) throw ( SemanticError ) :
	Statement( labels ), computedTarget(_computedTarget ), type(_type ) {
	if ( type != BranchStmt::Goto || computedTarget == 0 )
		throw SemanticError("Computed target not valid in branch statement");
}

void BranchStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Branch (" << brType[type] << ")" << endl ;
}

ReturnStmt::ReturnStmt( std::list<Label> labels, Expression *_expr, bool throwP ) : Statement( labels ), expr( _expr ), isThrow( throwP ) {}

ReturnStmt::~ReturnStmt() {
	delete expr;
}

void ReturnStmt::print( std::ostream &os, int indent ) {
	os << std::string( indent, ' ' ) << string ( isThrow? "Throw":"Return" ) << " Statement, returning: ";
	if ( expr != 0 ) expr->print( os );
	os << endl;
}

IfStmt::IfStmt( std::list<Label> _labels, Expression *_condition, Statement *_thenPart, Statement *_elsePart ):
	Statement(_labels ), condition(_condition ), thenPart(_thenPart ), elsePart(_elsePart ) {}

IfStmt::~IfStmt() {}

void IfStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "If on condition: " << endl ;
	condition->print( os, indent + 4 );

	os << string( indent, ' ' ) << ".... and branches: " << endl;

	thenPart->print( os, indent + 4 );

	if ( elsePart != 0 ) {
		elsePart->print( os, indent + 4 );
	} // if
}

SwitchStmt::SwitchStmt( std::list<Label> _labels, Expression * _condition, std::list<Statement *> &_branches ):
	Statement(_labels ), condition(_condition ), branches(_branches ) {
}

SwitchStmt::~SwitchStmt() {
	delete condition;
	// destroy branches
}

void SwitchStmt::add_case( CaseStmt *c ) {}

void SwitchStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Switch on condition: ";
	condition->print( os );
	os << endl;

	// branches
	std::list<Statement *>::iterator i;
	for ( i = branches.begin(); i != branches.end(); i++)
		(*i )->print( os, indent + 4 );

	//for_each( branches.begin(), branches.end(), mem_fun( bind1st(&Statement::print ), os ));
}

CaseStmt::CaseStmt( std::list<Label> _labels, Expression *_condition, std::list<Statement *> &_statements, bool deflt ) throw ( SemanticError ) : 
	Statement(_labels ), condition(_condition ), stmts(_statements ), _isDefault( deflt ) {
	if ( isDefault() && condition != 0 )
		throw SemanticError("default with conditions");
}

CaseStmt::~CaseStmt() {
	delete condition;
}

CaseStmt * CaseStmt::makeDefault( std::list<Label> labels, std::list<Statement *> branches ) {
	return new CaseStmt( labels, 0, branches, true );
}

void CaseStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' );

	if ( isDefault())
		os << "Default ";
	else {
		os << "Case ";
		condition->print( os );
	} // if

	os << endl;

	std::list<Statement *>::iterator i;
	for ( i = stmts.begin(); i != stmts.end(); i++)
		(*i )->print( os, indent + 4 );
}

//ChooseStmt::ChooseStmt( std::list<Label> labels, Expression *condition, Statement *body ) {}
ChooseStmt::ChooseStmt( std::list<Label> _labels, Expression * _condition, std::list<Statement *> &_branches ):
	Statement(_labels ), condition(_condition ), branches(_branches ) {
}

ChooseStmt::~ChooseStmt() {
	delete condition;
}

void ChooseStmt::add_case( CaseStmt *c ) {}

void ChooseStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Choose on condition: ";
	condition->print( os );
	os << endl;

	// branches
	std::list<Statement *>::iterator i;
	for ( i = branches.begin(); i != branches.end(); i++)
		(*i )->print( os, indent + 4 );

	//for_each( branches.begin(), branches.end(), mem_fun( bind1st(&Statement::print ), os ));
}

void FallthruStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Fall-through statement" << endl;
}

WhileStmt::WhileStmt( std::list<Label> labels, Expression *condition_, Statement *body_, bool isDoWhile_ ):
	Statement( labels ), condition( condition_), body( body_), isDoWhile( isDoWhile_) {
}

WhileStmt::~WhileStmt() {
	delete body;
}

void WhileStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "While on condition: " << endl ;
	condition->print( os, indent + 4 );

	os << string( indent, ' ' ) << ".... with body: " << endl;

	if ( body != 0 ) body->print( os, indent + 4 );
}

ForStmt::ForStmt( std::list<Label> labels, Statement *initialization_, Expression *condition_, Expression *increment_, Statement *body_ ):
	Statement( labels ), initialization( initialization_ ), condition( condition_ ), increment( increment_ ), body( body_ ) {
}

ForStmt::~ForStmt() {
	delete initialization;
	delete condition;
	delete increment;
	delete body;
}

void ForStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Labels: {";
	for (std::list<Label>::iterator it = get_labels().begin(); it != get_labels().end(); ++it) {
		os << *it << ",";
	}
	os << "}" << endl;

	os << string( indent, ' ' ) << "For Statement" << endl ;

	os << string( indent + 2, ' ' ) << "initialization: \n"; 
	if ( initialization != 0 )
		initialization->print( os, indent + 4 );

	os << "\n" << string( indent + 2, ' ' ) << "condition: \n"; 
	if ( condition != 0 )
		condition->print( os, indent + 4 );

	os << "\n" << string( indent + 2, ' ' ) << "increment: \n"; 
	if ( increment != 0 )
		increment->print( os, indent + 4 );

	os << "\n" << string( indent + 2, ' ' ) << "statement block: \n"; 
	if ( body != 0 )
		body->print( os, indent + 4 );

	os << endl;
}

TryStmt::TryStmt( std::list<Label> labels, CompoundStmt *tryBlock, std::list<Statement *> &_handlers, FinallyStmt *_finallyBlock ) :
	Statement( labels ), block( tryBlock ),  handlers( _handlers ), finallyBlock( _finallyBlock ) {
}

TryStmt::TryStmt( const TryStmt &other ) : Statement( other.labels ) {
	block = other.block;
	std::copy( other.handlers.begin(), other.handlers.end(), back_inserter( handlers ) );
	finallyBlock = other.finallyBlock;
}

TryStmt::~TryStmt() {
	delete block;
}

void TryStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Try Statement" << endl;
	os << string( indent + 2, ' ' ) << "with block: " << endl;
	block->print( os, indent + 4 );

	// handlers
	os << string( indent + 2, ' ' ) << "and handlers: " << endl;
	std::list<Statement *>::iterator i;
	for ( i = handlers.begin(); i != handlers.end(); i++)
		(*i )->print( os, indent + 4 );

	// finally block
	if ( finallyBlock != 0 ) {
		os << string( indent + 2, ' ' ) << "Finally block: " << endl;
		finallyBlock->print( os, indent + 4 );
	} // if
}

CatchStmt::CatchStmt( std::list<Label> labels, Declaration *_decl, Statement *_body, bool isCatchRest ) :
	Statement( labels ), decl ( _decl ), body( _body ), catchRest ( isCatchRest ) {
}

CatchStmt::~CatchStmt() {
	delete decl;
	delete body;
}

void CatchStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Catch Statement" << endl;

	os << string( indent, ' ' ) << "... catching" << endl;
	if ( decl ) {
		decl->printShort( os, indent + 4 );
		os << endl;
	} else if ( catchRest )
		os << string( indent + 4 , ' ' ) << "the rest" << endl;
	else
		os << string( indent + 4 , ' ' ) << ">>> Error:  this catch clause must have a declaration <<<" << endl;
}


FinallyStmt::FinallyStmt( std::list<Label> labels, CompoundStmt *_block ) : Statement( labels ), block( _block ) {
	assert( labels.empty() ); // finally statement cannot be labeled
}

FinallyStmt::~FinallyStmt() {
	delete block;
}

void FinallyStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Finally Statement" << endl;
	os << string( indent + 2, ' ' ) << "with block: " << endl;
	block->print( os, indent + 4 );
}

NullStmt::NullStmt( std::list<Label> labels ) : CompoundStmt( labels ) {}
NullStmt::NullStmt() : CompoundStmt( std::list<Label>() ) {}
NullStmt::~NullStmt() {}

void NullStmt::print( std::ostream &os, int indent ) {
	os << string( indent, ' ' ) << "Null Statement" << endl ;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
