//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// RemoveInit.cc -- 
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Mon Nov 16 16:58:36 2015
// Update Count     : 30
//

#include "RemoveInit.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Initializer.h"
#include "SynTree/Mutator.h"

// changes to Validate:
// -check that ctor/dtor has >= 1 argument
// -check that first argument to ctor/dtor has pointer type
// -check that return type is void (0 return types)
// -transform ctor to return its first argument
// -generate ctors and dtors alongside ?=? for aggregate types 

// idea: modify this pass to decide whether an object declaration is
// POD type. 
// - If it is not POD-type, initialization should be changed into 
//   a constructor call. 
// - If it is a POD type, then check that there are no designations. 
//   It is probably easiest to leave the declaration in C-initializer
//   form and resolve as normal, since we don't want to actually incur
//   the cost of a constructor unless we have to.

// change indexer to remove all constructors for a type once a user-defined one appears?

// question: if a destructor is declared before the scope of a variable ends, 
// should it be destructed? Or should we decide this at declaration point?


// alternative (that I think I like better, if there aren't any flaws)
//   --flaw appears to be the exponential blowup in the number of ctors described below
// change into constructor form if no designations
// if not POD type, error out if there are designations
// if there are designations, handle them in the resolver

// ==MAYBE== even possible to rewrite designations not as ?=?, but as ?{} (see initialization.txt)
// there may be some transformation that's required to bring this back into a reasonable form
// for codegen, it'll depend on exactly what the expressions that are fed to the resolver look like
// e.g.

// struct A {
//   struct B { int x; } b;
//   struct C { int x, y, z } c;
//   struct D { int x; } d;
// }
//
// A a = { 1, 2, 3, 4, 5 };
// => struct A a;
//    ?{}(&a.b, (struct B){ 1 } );
//    ?{}(&a.c, (struct C){ 2, 3, 4 } );
//    ?{}(&a.d, (struct D){ 5 } );
// (it obviously shouldn't look like this, but what should it look like??)
//
// (perhaps this?)
// => struct A a;
//    ?{}(&a, (struct B){ 1 }, (struct C){ 2, 3, 4 }, (struct D){ 5 });
// (of course, this requires me to do the grouping found here,
//  and remember that parts might be missing! That said, I'm essentially
//  already doing this in the resolver, so whatever I guess?)
// (note this requires an alternative finder, because these may be
//  function calls, not just simple literals)
// (this is a bit of a recursive problem - in order to know how to group
//  the expressions into a struct to be an argument to a constructor, I need to
//  know what the constructor's signature looks like - but in order to figure out
//  which constructor is being used (and thus what its signature looks like), I need
//  to group the values into a struct type)
// (this seems to imply (to me, anyway) that C initializers can't be represented as 
//  constructors without an exponential blowup in the number of constructors present)

namespace InitTweak {
	namespace {
		const std::list<Label> noLabels;
	}

	void tweak( std::list< Declaration * > translationUnit ) {
		RemoveInit remover;
		mutateAll( translationUnit, remover );
	}

	void RemoveInit::mutateStatementList( std::list< Statement* > &statements ) {
		for ( std::list< Statement* >::iterator i = statements.begin(); i != statements.end(); ++i ) {
			if ( ! stmtsToAddAfter.empty() ) {
				statements.splice( i, stmtsToAddAfter );
			} // if
			*i = (*i)->acceptMutator( *this );
		} // for
		if ( ! stmtsToAddAfter.empty() ) {
			statements.splice( statements.end(), stmtsToAddAfter );
		} // if
	}

	CompoundStmt *RemoveInit::mutate(CompoundStmt *compoundStmt) {
		mutateStatementList( compoundStmt->get_kids() );
		return compoundStmt;
	}

// in the case where an object has an initializer and a polymorphic type, insert an assignment
// immediately after the declaration. This will (seemingly) cause the later phases to do the right
// thing with the assignment
	ObjectDecl *RemoveInit::mutate( ObjectDecl *objDecl ) {
		if (objDecl->get_init() && dynamic_cast<TypeInstType*>(objDecl->get_type())) {
			if (SingleInit * single = dynamic_cast<SingleInit*>(objDecl->get_init())) {
				UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
				assign->get_args().push_back( new AddressExpr (new NameExpr( objDecl->get_name() ) ) );
				assign->get_args().push_back( single->get_value()->clone() );
				stmtsToAddAfter.push_back(new ExprStmt(noLabels, assign));
			} // if
		} // if
		return objDecl;
	}
} // namespace InitTweak

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
