//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Type.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Sep 25 15:16:32 2017
// Update Count     : 38
//
#include "Type.h"

#include "Attribute.h"               // for Attribute
#include "Common/utility.h"          // for cloneAll, deleteAll, printAll
#include "InitTweak/InitTweak.h"     // for getPointerBase
#include "SynTree/BaseSyntaxNode.h"  // for BaseSyntaxNode
#include "SynTree/Declaration.h"     // for TypeDecl

using namespace std;

const char *BasicType::typeNames[BasicType::NUMBER_OF_BASIC_TYPES] = {
	"_Bool",
	"char",
	"signed char",
	"unsigned char",
	"signed short int",
	"unsigned short int",
	"signed int",
	"unsigned int",
	"signed long int",
	"unsigned long int",
	"signed long long int",
	"unsigned long long int",
	"float",
	"double",
	"long double",
	"float _Complex",
	"double _Complex",
	"long double _Complex",
	"float _Imaginary",
	"double _Imaginary",
	"long double _Imaginary",
	"__int128",
	"unsigned __int128",
};

Type::Type( const Qualifiers &tq, const std::list< Attribute * > & attributes ) : tq( tq ), attributes( attributes ) {}

Type::Type( const Type &other ) : BaseSyntaxNode( other ), tq( other.tq ) {
	cloneAll( other.forall, forall );
	cloneAll( other.attributes, attributes );
}

Type::~Type() {
	deleteAll( forall );
	deleteAll( attributes );
}

// These must remain in the same order as the corresponding bit fields.
const char * Type::FuncSpecifiersNames[] = { "inline", "fortran", "_Noreturn" };
const char * Type::StorageClassesNames[] = { "extern", "static", "auto", "register", "_Thread_local" };
const char * Type::QualifiersNames[] = { "const", "restrict", "volatile", "lvalue", "mutex", "_Atomic" };

Type * Type::stripDeclarator() {
	Type * type, * at;
	for ( type = this; (at = InitTweak::getPointerBase( type )); type = at );
	return type;
}

Type * Type::stripReferences() {
	Type * type;
	ReferenceType * ref;
	for ( type = this; (ref = dynamic_cast<ReferenceType *>( type )); type = ref->base );
	return type;
}

int Type::referenceDepth() const { return 0; }

void Type::print( std::ostream &os, Indenter indent ) const {
	if ( ! forall.empty() ) {
		os << "forall" << std::endl;
		printAll( forall, os, indent+1 );
		os << ++indent;
	} // if

	if ( ! attributes.empty() ) {
		os << "with attributes" << endl;
		printAll( attributes, os, indent+1 );
	} // if

	tq.print( os );
}

// Empty Variable declarations:
const Type::FuncSpecifiers noFuncSpecifiers;
const Type::StorageClasses noStorageClasses;
const Type::Qualifiers noQualifiers;

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
