//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CodeGenerator.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Andrew Beach
// Last Modified On : Thu Jun  8 16:00:00 2017
// Update Count     : 485
//

#include <cassert>                   // for assert, assertf
#include <list>                      // for _List_iterator, list, list<>::it...

#include "CodeGenerator.h"
#include "Common/SemanticError.h"    // for SemanticError
#include "Common/UniqueName.h"       // for UniqueName
#include "Common/utility.h"          // for CodeLocation, toString
#include "GenType.h"                 // for genType
#include "InitTweak/InitTweak.h"     // for getPointerBase
#include "OperatorTable.h"           // for OperatorInfo, operatorLookup
#include "Parser/LinkageSpec.h"      // for Spec, Intrinsic
#include "SynTree/Attribute.h"       // for Attribute
#include "SynTree/BaseSyntaxNode.h"  // for BaseSyntaxNode
#include "SynTree/Constant.h"        // for Constant
#include "SynTree/Declaration.h"     // for DeclarationWithType, TypeDecl
#include "SynTree/Expression.h"      // for Expression, UntypedExpr, Applica...
#include "SynTree/Initializer.h"     // for Initializer, ListInit, Designation
#include "SynTree/Label.h"           // for Label, operator<<
#include "SynTree/Statement.h"       // for Statement, AsmStmt, BranchStmt
#include "SynTree/Type.h"            // for Type, Type::StorageClasses, Func...

using namespace std;

namespace CodeGen {
	int CodeGenerator::tabsize = 4;

	// the kinds of statements that would ideally be followed by whitespace
	bool wantSpacing( Statement * stmt) {
		return dynamic_cast< IfStmt * >( stmt ) || dynamic_cast< CompoundStmt * >( stmt ) ||
			dynamic_cast< WhileStmt * >( stmt ) || dynamic_cast< ForStmt * >( stmt ) || dynamic_cast< SwitchStmt *>( stmt );
	}

	void CodeGenerator::extension( Expression * expr ) {
		if ( expr->get_extension() ) {
			output << "__extension__ ";
		} // if
	} // extension

	void CodeGenerator::extension( Declaration * decl ) {
		if ( decl->get_extension() ) {
			output << "__extension__ ";
		} // if
	} // extension

	void CodeGenerator::asmName( DeclarationWithType * decl ) {
		if ( ConstantExpr * asmName = decl->get_asmName() ) {
			output << " asm ( " << asmName->get_constant()->get_value() << " )";
		} // if
	} // extension

	CodeGenerator::LabelPrinter & CodeGenerator::LabelPrinter::operator()( std::list< Label > & l ) {
		labels = &l;
		return *this;
	}

	ostream & operator<<( ostream & output, CodeGenerator::LabelPrinter & printLabels ) {
		std::list< Label > & labs = *printLabels.labels;
		// l.unique(); // assumes a sorted list. Why not use set? Does order matter?
		for ( Label & l : labs ) {
			output << l.get_name() + ": ";
			printLabels.cg.genAttributes( l.get_attributes() );
		} // for
		return output;
	}

	CodeGenerator::LineMarker::LineMarker(
			CodeLocation const & loc, bool toPrint) :
		loc(loc), toPrint(toPrint)
	{}

	CodeGenerator::LineMarker CodeGenerator::lineDirective(
			BaseSyntaxNode const * node) {
		return LineMarker(node->location, lineMarks);
	}

	std::ostream & operator<<(std::ostream & out,
			CodeGenerator::LineMarker const & marker) {
		if (marker.toPrint && marker.loc.isSet()) {
			return out << "\n# " << marker.loc.linenumber << " \""
				<< marker.loc.filename << "\"\n";
		} else if (marker.toPrint) {
			return out << "\n/* Missing CodeLocation */\n";
		} else {
        	return out;
		}
	}

	CodeGenerator::CodeGenerator( std::ostream & os, bool pretty, bool genC, bool lineMarks ) : indent( CodeGenerator::tabsize ), insideFunction( false ), output( os ), printLabels( *this ), pretty( pretty ), genC( genC ), lineMarks( lineMarks ) {}

	string CodeGenerator::mangleName( DeclarationWithType * decl ) {
		if ( pretty ) return decl->get_name();
		if ( decl->get_mangleName() != "" ) {
			// need to incorporate scope level in order to differentiate names for destructors
			return decl->get_scopedMangleName();
		} else {
			return decl->get_name();
		} // if
	}

	void CodeGenerator::genAttributes( list< Attribute * > & attributes ) {
	  if ( attributes.empty() ) return;
		output << "__attribute__ ((";
		for ( list< Attribute * >::iterator attr( attributes.begin() );; ) {
			output << (*attr)->get_name();
			if ( ! (*attr)->get_parameters().empty() ) {
				output << "(";
				genCommaList( (*attr)->get_parameters().begin(), (*attr)->get_parameters().end() );
				output << ")";
			} // if
		  if ( ++attr == attributes.end() ) break;
			output << ",";								// separator
		} // for
		output << ")) ";
	} // CodeGenerator::genAttributes


	// *** Declarations
	void CodeGenerator::visit( FunctionDecl * functionDecl ) {
		extension( functionDecl );
		genAttributes( functionDecl->get_attributes() );

		handleStorageClass( functionDecl );
		functionDecl->get_funcSpec().print( output );

		output << genType( functionDecl->get_functionType(), mangleName( functionDecl ), pretty, genC );

		asmName( functionDecl );

		// acceptAll( functionDecl->get_oldDecls(), *this );
		if ( functionDecl->get_statements() ) {
			functionDecl->get_statements()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( ObjectDecl * objectDecl ) {
		if (objectDecl->get_name().empty() && genC ) {
			// only generate an anonymous name when generating C code, otherwise it clutters the output too much
			static UniqueName name = { "__anonymous_object" };
			objectDecl->set_name( name.newName() );
		}

		extension( objectDecl );
		genAttributes( objectDecl->get_attributes() );

		handleStorageClass( objectDecl );
		output << genType( objectDecl->get_type(), mangleName( objectDecl ), pretty, genC );

		asmName( objectDecl );

		if ( objectDecl->get_init() ) {
			output << " = ";
			objectDecl->get_init()->accept( *this );
		} // if

		if ( objectDecl->get_bitfieldWidth() ) {
			output << ":";
			objectDecl->get_bitfieldWidth()->accept( *this );
		} // if
	}

	void CodeGenerator::handleAggregate( AggregateDecl * aggDecl, const std::string & kind ) {
		genAttributes( aggDecl->get_attributes() );

		if( ! aggDecl->get_parameters().empty() && ! genC ) {
			// assertf( ! genC, "Aggregate type parameters should not reach code generation." );
			output << "forall(";
			genCommaList( aggDecl->get_parameters().begin(), aggDecl->get_parameters().end() );
			output << ")" << endl;
		}

		output << kind;
		if ( aggDecl->get_name() != "" )
			output << aggDecl->get_name();

		if ( aggDecl->has_body() ) {
			std::list< Declaration * > & memb = aggDecl->get_members();
			output << " {" << endl;

			++indent;
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end(); i++ ) {
				output << lineDirective( *i ) << indent;
				(*i)->accept( *this );
				output << ";" << endl;
			} // for

			--indent;

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( StructDecl * structDecl ) {
		extension( structDecl );
		handleAggregate( structDecl, "struct " );
	}

	void CodeGenerator::visit( UnionDecl * unionDecl ) {
		extension( unionDecl );
		handleAggregate( unionDecl, "union " );
	}

	void CodeGenerator::visit( EnumDecl * enumDecl ) {
		extension( enumDecl );
		output << lineDirective ( enumDecl );
		output << "enum ";
		genAttributes( enumDecl->get_attributes() );

		if ( enumDecl->get_name() != "" )
			output << enumDecl->get_name();

		std::list< Declaration* > &memb = enumDecl->get_members();

		if ( ! memb.empty() ) {
			output << " {" << endl;

			++indent;
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end();  i++) {
				ObjectDecl * obj = dynamic_cast< ObjectDecl* >( *i );
				assert( obj );
				output << lineDirective( obj ) << indent << mangleName( obj );
				if ( obj->get_init() ) {
					output << " = ";
					obj->get_init()->accept( *this );
				} // if
				output << "," << endl;
			} // for

			--indent;

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( __attribute__((unused)) TraitDecl * traitDecl ) {}

	void CodeGenerator::visit( TypedefDecl * typeDecl ) {
		assertf( ! genC, "Typedefs are removed and substituted in earlier passes." );
		output << lineDirective( typeDecl );
		output << "typedef ";
		output << genType( typeDecl->get_base(), typeDecl->get_name(), pretty, genC ) << endl;
	}

	void CodeGenerator::visit( TypeDecl * typeDecl ) {
		if ( genC ) {
			// really, we should mutate this into something that isn't a TypeDecl but that requires large-scale changes,
			// still to be done
			extension( typeDecl );
			output << "extern unsigned long " << typeDecl->get_name();
			if ( typeDecl->get_base() ) {
				output << " = sizeof( " << genType( typeDecl->get_base(), "", pretty, genC ) << " )";
			} // if
		} else {
			output << typeDecl->genTypeString() << " " << typeDecl->get_name();
			if ( typeDecl->get_kind() != TypeDecl::Any && typeDecl->get_sized() ) {
				output << " | sized(" << typeDecl->get_name() << ")";
			}
			if ( ! typeDecl->get_assertions().empty() ) {
				output << " | { ";
				genCommaList( typeDecl->get_assertions().begin(), typeDecl->get_assertions().end() );
				output << " }";
			}
		}
	}

	void CodeGenerator::visit( Designation * designation ) {
		std::list< Expression * > designators = designation->get_designators();
		if ( designators.size() == 0 ) return;
		for ( Expression * des : designators ) {
			if ( dynamic_cast< NameExpr * >( des ) || dynamic_cast< VariableExpr * >( des ) ) {
				// if expression is a NameExpr or VariableExpr, then initializing aggregate member
				output << ".";
				des->accept( *this );
			} else {
				// otherwise, it has to be a ConstantExpr or CastExpr, initializing array eleemnt
				output << "[";
				des->accept( *this );
				output << "]";
			} // if
		} // for
		output << " = ";
	}

	void CodeGenerator::visit( SingleInit * init ) {
		init->get_value()->accept( *this );
	}

	void CodeGenerator::visit( ListInit * init ) {
		auto initBegin = init->begin();
		auto initEnd = init->end();
		auto desigBegin = init->get_designations().begin();
		auto desigEnd = init->get_designations().end();

		output << "{ ";
		for ( ; initBegin != initEnd && desigBegin != desigEnd; ) {
			(*desigBegin)->accept( *this );
			(*initBegin)->accept( *this );
			++initBegin, ++desigBegin;
			if ( initBegin != initEnd ) {
				output << ", ";
			}
		}
		output << " }";
		assertf( initBegin == initEnd && desigBegin == desigEnd, "Initializers and designators not the same length. %s", toString( init ).c_str() );
	}

	void CodeGenerator::visit( __attribute__((unused)) ConstructorInit * init ){
		assertf( ! genC, "ConstructorInit nodes should not reach code generation." );
		// xxx - generate something reasonable for constructor/destructor pairs
		output << "<ctorinit>";
	}

	void CodeGenerator::visit( Constant * constant ) {
		output << constant->get_value() ;
	}

	// *** Expressions
	void CodeGenerator::visit( ApplicationExpr * applicationExpr ) {
		extension( applicationExpr );
		if ( VariableExpr * varExpr = dynamic_cast< VariableExpr* >( applicationExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( varExpr->get_var()->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = applicationExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_PREFIXASSIGN:
				  case OT_POSTFIXASSIGN:
				  case OT_INFIXASSIGN:
				  case OT_CTOR:
				  case OT_DTOR:
					{
						assert( arg != applicationExpr->get_args().end() );
						if ( AddressExpr * addrExpr = dynamic_cast< AddressExpr * >( *arg ) ) {
							// remove & from first assignment/ctor argument
							*arg = addrExpr->get_arg();
						} else {
							// no address-of operator, so must be a pointer - add dereference
							// NOTE: if the assertion starts to trigger, check that the application expr isn't being shared.
							// Since its arguments are modified here, this assertion most commonly triggers when the application
							// is visited multiple times.
							UntypedExpr * newExpr = new UntypedExpr( new NameExpr( "*?" ) );
							newExpr->get_args().push_back( *arg );
							Type * type = InitTweak::getPointerBase( (*arg)->get_result() );
							assertf( type, "First argument to a derefence must be a pointer. Ensure that expressions are not being shared." );
							newExpr->set_result( type->clone() );
							*arg = newExpr;
						} // if
						break;
					}

				  default:
					// do nothing
					;
				} // switch

				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( applicationExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;

				  case OT_CALL:
					// there are no intrinsic definitions of the function call operator
					assert( false );
					break;

				  case OT_CTOR:
				  case OT_DTOR:
					if ( applicationExpr->get_args().size() == 1 ) {
						// the expression fed into a single parameter constructor or destructor may contain side
						// effects, so must still output this expression
						output << "(";
						(*arg++)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else if ( applicationExpr->get_args().size() == 2 ) {
						// intrinsic two parameter constructors are essentially bitwise assignment
						output << "(";
						(*arg++)->accept( *this );
						output << opInfo.symbol;
						(*arg)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else {
						// no constructors with 0 or more than 2 parameters
						assert( false );
					} // if
					break;

				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;


				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( applicationExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_CONSTANT:
				  case OT_LABELADDRESS:
					// there are no intrinsic definitions of 0/1 or label addresses as functions
					assert( false );
				} // switch
			} else {
				varExpr->accept( *this );
				output << "(";
				genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
				output << ")";
			} // if
		} else {
			applicationExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
			output << ")";
		} // if
	}

	void CodeGenerator::visit( UntypedExpr * untypedExpr ) {
		extension( untypedExpr );
		if ( NameExpr * nameExpr = dynamic_cast< NameExpr* >( untypedExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = untypedExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( untypedExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;

				  case OT_CALL:
					assert( false );

				  case OT_CTOR:
				  case OT_DTOR:
					if ( untypedExpr->get_args().size() == 1 ) {
						// the expression fed into a single parameter constructor or destructor may contain side
						// effects, so must still output this expression
						output << "(";
						(*arg++)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else if ( untypedExpr->get_args().size() == 2 ) {
						// intrinsic two parameter constructors are essentially bitwise assignment
						output << "(";
						(*arg++)->accept( *this );
						output << opInfo.symbol;
						(*arg)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else {
						// no constructors with 0 or more than 2 parameters
						assert( false );
					} // if
					break;

				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
				  case OT_LABELADDRESS:
					assert( untypedExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( untypedExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;

				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( untypedExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_CONSTANT:
					// there are no intrinsic definitions of 0 or 1 as functions
					assert( false );
				} // switch
			} else {
				if ( nameExpr->get_name() == "..." ) { // case V1 ... V2 or case V1~V2
					assert( untypedExpr->get_args().size() == 2 );
					(*untypedExpr->get_args().begin())->accept( *this );
					output << " ... ";
					(*--untypedExpr->get_args().end())->accept( *this );
				} else {								// builtin routines
					nameExpr->accept( *this );
					output << "(";
					genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
					output << ")";
				} // if
			} // if
		} else {
			untypedExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
			output << ")";
		} // if
	}

	void CodeGenerator::visit( RangeExpr * rangeExpr ) {
		rangeExpr->get_low()->accept( *this );
		output << " ... ";
		rangeExpr->get_high()->accept( *this );
	}

	void CodeGenerator::visit( NameExpr * nameExpr ) {
		extension( nameExpr );
		OperatorInfo opInfo;
		if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
			assert( opInfo.type == OT_CONSTANT );
			output << opInfo.symbol;
		} else {
			output << nameExpr->get_name();
		} // if
	}

	void CodeGenerator::visit( AddressExpr * addressExpr ) {
		extension( addressExpr );
		output << "(&";
		// this hack makes sure that we don't convert "constant_zero" to "0" if we're taking its address
		if ( VariableExpr * variableExpr = dynamic_cast< VariableExpr* >( addressExpr->get_arg() ) ) {
			output << mangleName( variableExpr->get_var() );
		} else {
			addressExpr->get_arg()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( CastExpr * castExpr ) {
		extension( castExpr );
		output << "(";
		if ( castExpr->get_result()->isVoid() ) {
			output << "(void)" ;
		} else if ( ! castExpr->get_result()->get_lvalue() ) {
			// at least one result type of cast, but not an lvalue
			output << "(";
			output << genType( castExpr->get_result(), "", pretty, genC );
			output << ")";
		} else {
			// otherwise, the cast is to an lvalue type, so the cast should be dropped, since the result of a cast is
			// never an lvalue in C
		} // if
		castExpr->get_arg()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( UntypedMemberExpr * memberExpr ) {
		assertf( ! genC, "UntypedMemberExpr should not reach code generation." );
		extension( memberExpr );
		memberExpr->get_aggregate()->accept( *this );
		output << ".";
		memberExpr->get_member()->accept( *this );
	}

	void CodeGenerator::visit( MemberExpr * memberExpr ) {
		extension( memberExpr );
		memberExpr->get_aggregate()->accept( *this );
		output << "." << mangleName( memberExpr->get_member() );
	}

	void CodeGenerator::visit( VariableExpr * variableExpr ) {
		extension( variableExpr );
		OperatorInfo opInfo;
		if ( variableExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( variableExpr->get_var()->get_name(), opInfo ) && opInfo.type == OT_CONSTANT ) {
			output << opInfo.symbol;
		} else {
			output << mangleName( variableExpr->get_var() );
		} // if
	}

	void CodeGenerator::visit( ConstantExpr * constantExpr ) {
		assert( constantExpr->get_constant() );
		extension( constantExpr );
		constantExpr->get_constant()->accept( *this );
	}

	void CodeGenerator::visit( SizeofExpr * sizeofExpr ) {
		extension( sizeofExpr );
		output << "sizeof(";
		if ( sizeofExpr->get_isType() ) {
			output << genType( sizeofExpr->get_type(), "", pretty, genC );
		} else {
			sizeofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( AlignofExpr * alignofExpr ) {
		// use GCC extension to avoid bumping std to C11
		extension( alignofExpr );
		output << "__alignof__(";
		if ( alignofExpr->get_isType() ) {
			output << genType( alignofExpr->get_type(), "", pretty, genC );
		} else {
			alignofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( UntypedOffsetofExpr * offsetofExpr ) {
		assertf( ! genC, "UntypedOffsetofExpr should not reach code generation." );
		output << "offsetof(";
		output << genType( offsetofExpr->get_type(), "", pretty, genC );
		output << ", " << offsetofExpr->get_member();
		output << ")";
	}

	void CodeGenerator::visit( OffsetofExpr * offsetofExpr ) {
		// use GCC builtin
		output << "__builtin_offsetof(";
		output << genType( offsetofExpr->get_type(), "", pretty, genC );
		output << ", " << mangleName( offsetofExpr->get_member() );
		output << ")";
	}

	void CodeGenerator::visit( OffsetPackExpr * offsetPackExpr ) {
		assertf( ! genC, "OffsetPackExpr should not reach code generation." );
		output << "__CFA_offsetpack(" << genType( offsetPackExpr->get_type(), "", pretty, genC ) << ")";
	}

	void CodeGenerator::visit( LogicalExpr * logicalExpr ) {
		extension( logicalExpr );
		output << "(";
		logicalExpr->get_arg1()->accept( *this );
		if ( logicalExpr->get_isAnd() ) {
			output << " && ";
		} else {
			output << " || ";
		} // if
		logicalExpr->get_arg2()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( ConditionalExpr * conditionalExpr ) {
		extension( conditionalExpr );
		output << "(";
		conditionalExpr->get_arg1()->accept( *this );
		output << " ? ";
		conditionalExpr->get_arg2()->accept( *this );
		output << " : ";
		conditionalExpr->get_arg3()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( CommaExpr * commaExpr ) {
		extension( commaExpr );
		output << "(";
		commaExpr->get_arg1()->accept( *this );
		output << " , ";
		commaExpr->get_arg2()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( UntypedTupleExpr * tupleExpr ) {
		assertf( ! genC, "UntypedTupleExpr should not reach code generation." );
		extension( tupleExpr );
		output << "[";
		genCommaList( tupleExpr->get_exprs().begin(), tupleExpr->get_exprs().end() );
		output << "]";
	}

	void CodeGenerator::visit( TupleExpr * tupleExpr ) {
		assertf( ! genC, "TupleExpr should not reach code generation." );
		extension( tupleExpr );
		output << "[";
		genCommaList( tupleExpr->get_exprs().begin(), tupleExpr->get_exprs().end() );
		output << "]";
	}

	void CodeGenerator::visit( TupleIndexExpr * tupleExpr ) {
		assertf( ! genC, "TupleIndexExpr should not reach code generation." );
		extension( tupleExpr );
		tupleExpr->get_tuple()->accept( *this );
		output << "." << tupleExpr->get_index();
	}

	void CodeGenerator::visit( TypeExpr * typeExpr ) {
		// if ( genC ) std::cerr << "typeexpr still exists: " << typeExpr << std::endl;
		// assertf( ! genC, "TypeExpr should not reach code generation." );
		if ( ! genC ) {
			output<< genType( typeExpr->get_type(), "", pretty, genC );
		}
	}

	void CodeGenerator::visit( AsmExpr * asmExpr ) {
		if ( asmExpr->get_inout() ) {
			output << "[ ";
			asmExpr->get_inout()->accept( *this );
			output << " ] ";
		} // if
		asmExpr->get_constraint()->accept( *this );
		output << " ( ";
		asmExpr->get_operand()->accept( *this );
		output << " )";
	}

	void CodeGenerator::visit( CompoundLiteralExpr *compLitExpr ) {
		assert( compLitExpr->get_result() && dynamic_cast< ListInit * > ( compLitExpr->get_initializer() ) );
		output << "(" << genType( compLitExpr->get_result(), "", pretty, genC ) << ")";
		compLitExpr->get_initializer()->accept( *this );
	}

	void CodeGenerator::visit( StmtExpr * stmtExpr ) {
		std::list< Statement * > & stmts = stmtExpr->get_statements()->get_kids();
		output << lineDirective( stmtExpr) << "({" << std::endl;
		++indent;
		unsigned int numStmts = stmts.size();
		unsigned int i = 0;
		for ( Statement * stmt : stmts ) {
			output << lineDirective( stmt ) << indent;
            output << printLabels( stmt->get_labels() );
			if ( i+1 == numStmts ) {
				// last statement in a statement expression needs to be handled specially -
				// cannot cast to void, otherwise the expression statement has no value
				if ( ExprStmt * exprStmt = dynamic_cast< ExprStmt * >( stmt ) ) {
					exprStmt->get_expr()->accept( *this );
					output << ";" << endl;
					++i;
					break;
				}
			}
			stmt->accept( *this );
			output << endl;
			if ( wantSpacing( stmt ) ) {
				output << endl;
			} // if
			++i;
		}
		--indent;
		output << indent << "})";
	}

	// *** Statements
	void CodeGenerator::visit( CompoundStmt * compoundStmt ) {
		std::list<Statement*> ks = compoundStmt->get_kids();
		output << "{" << endl;

		++indent;

		for ( std::list<Statement *>::iterator i = ks.begin(); i != ks.end();  i++ ) {
			output << indent << printLabels( (*i)->get_labels() );
			(*i)->accept( *this );

			output << endl;
			if ( wantSpacing( *i ) ) {
				output << endl;
			} // if
		} // for
		--indent;

		output << indent << "}";
	}

	void CodeGenerator::visit( ExprStmt * exprStmt ) {
		assert( exprStmt );
		Expression * expr = exprStmt->get_expr();
		if ( genC ) {
			// cast the top-level expression to void to reduce gcc warnings.
			expr = new CastExpr( expr );
		}
		expr->accept( *this );
		output << ";";
	}

	void CodeGenerator::visit( AsmStmt * asmStmt ) {
		output << "asm ";
		if ( asmStmt->get_voltile() ) output << "volatile ";
		if ( ! asmStmt->get_gotolabels().empty()  ) output << "goto ";
		output << "( ";
		if ( asmStmt->get_instruction() ) asmStmt->get_instruction()->accept( *this );
		output << " : ";
		genCommaList( asmStmt->get_output().begin(), asmStmt->get_output().end() );
		output << " : ";
		genCommaList( asmStmt->get_input().begin(), asmStmt->get_input().end() );
		output << " : ";
		genCommaList( asmStmt->get_clobber().begin(), asmStmt->get_clobber().end() );
		if ( ! asmStmt->get_gotolabels().empty() ) {
			output << " : ";
			for ( std::list<Label>::iterator begin = asmStmt->get_gotolabels().begin();; ) {
				output << *begin++;
				if ( begin == asmStmt->get_gotolabels().end() ) break;
				output << ", ";
			} // for
		} // if
		output << " );" ;
	}

	void CodeGenerator::visit( AsmDecl * asmDecl ) {
		output << "asm ";
		AsmStmt * asmStmt = asmDecl->get_stmt();
		output << "( ";
		if ( asmStmt->get_instruction() ) asmStmt->get_instruction()->accept( *this );
		output << " )" ;
	}

	void CodeGenerator::visit( IfStmt * ifStmt ) {
		output << lineDirective( ifStmt );
		output << "if ( ";
		ifStmt->get_condition()->accept( *this );
		output << " ) ";

		ifStmt->get_thenPart()->accept( *this );

		if ( ifStmt->get_elsePart() != 0) {
			output << " else ";
			ifStmt->get_elsePart()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( SwitchStmt * switchStmt ) {
		output << lineDirective( switchStmt );
		output << "switch ( " ;
		switchStmt->get_condition()->accept( *this );
		output << " ) ";

		output << "{" << std::endl;
		++indent;
		acceptAll( switchStmt->get_statements(), *this );
		--indent;
		output << indent << "}";
	}

	void CodeGenerator::visit( CaseStmt * caseStmt ) {
		output << lineDirective( caseStmt );
		output << indent;
		if ( caseStmt->isDefault()) {
			output << "default";
		} else {
			output << "case ";
			caseStmt->get_condition()->accept( *this );
		} // if
		output << ":\n";

		std::list<Statement *> sts = caseStmt->get_statements();

		++indent;
		for ( std::list<Statement *>::iterator i = sts.begin(); i != sts.end();  i++) {
			output << indent << printLabels( (*i)->get_labels() )  ;
			(*i)->accept( *this );
			output << endl;
		} // for
		--indent;
	}

	void CodeGenerator::visit( BranchStmt * branchStmt ) {
		switch ( branchStmt->get_type()) {
		  case BranchStmt::Goto:
			if ( ! branchStmt->get_target().empty() )
				output << "goto " << branchStmt->get_target();
			else {
				if ( branchStmt->get_computedTarget() != 0 ) {
					output << "goto *";
					branchStmt->get_computedTarget()->accept( *this );
				} // if
			} // if
			break;
		  case BranchStmt::Break:
			output << "break";
			break;
		  case BranchStmt::Continue:
			output << "continue";
			break;
		} // switch
		output << ";";
	}

	void CodeGenerator::visit( ReturnStmt * returnStmt ) {
		output << "return ";
		maybeAccept( returnStmt->get_expr(), *this );
		output << ";";
	}

	void CodeGenerator::visit( ThrowStmt * throwStmt ) {
		assertf( ! genC, "Throw statements should not reach code generation." );

		output << ((throwStmt->get_kind() == ThrowStmt::Terminate) ?
		           "throw" : "throwResume");
		if (throwStmt->get_expr()) {
			output << " ";
			throwStmt->get_expr()->accept( *this );
		}
		if (throwStmt->get_target()) {
			output << " _At ";
			throwStmt->get_target()->accept( *this );
		}
		output << ";";
	}

	void CodeGenerator::visit( WhileStmt * whileStmt ) {
		if ( whileStmt->get_isDoWhile() ) {
			output << "do" ;
		} else {
			output << "while (" ;
			whileStmt->get_condition()->accept( *this );
			output << ")";
		} // if
		output << " ";

		output << CodeGenerator::printLabels( whileStmt->get_body()->get_labels() );
		whileStmt->get_body()->accept( *this );

		output << indent;

		if ( whileStmt->get_isDoWhile() ) {
			output << " while (" ;
			whileStmt->get_condition()->accept( *this );
			output << ");";
		} // if
	}

	void CodeGenerator::visit( ForStmt * forStmt ) {
		// initialization is always hoisted, so don't bother doing anything with that
		output << "for (;";

		if ( forStmt->get_condition() != 0 ) {
			forStmt->get_condition()->accept( *this );
		} // if
		output << ";";

		if ( forStmt->get_increment() != 0 ) {
			// cast the top-level expression to void to reduce gcc warnings.
			Expression * expr = new CastExpr( forStmt->get_increment() );
			expr->accept( *this );
		} // if
		output << ") ";

		if ( forStmt->get_body() != 0 ) {
			output << CodeGenerator::printLabels( forStmt->get_body()->get_labels() );
			forStmt->get_body()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( __attribute__((unused)) NullStmt * nullStmt ) {
		//output << indent << CodeGenerator::printLabels( nullStmt->get_labels() );
		output << "/* null statement */ ;";
	}

	void CodeGenerator::visit( DeclStmt * declStmt ) {
		declStmt->get_decl()->accept( *this );

		if ( doSemicolon( declStmt->get_decl() ) ) {
			output << ";";
		} // if
	}

	void CodeGenerator::handleStorageClass( DeclarationWithType * decl ) {
		if ( decl->get_storageClasses().any() ) {
			decl->get_storageClasses().print( output );
		} // if
	} // CodeGenerator::handleStorageClass

	std::string genName( DeclarationWithType * decl ) {
		CodeGen::OperatorInfo opInfo;
		if ( operatorLookup( decl->get_name(), opInfo ) ) {
			return opInfo.outputName;
		} else {
			return decl->get_name();
		} // if
	}
} // namespace CodeGen

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
