//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// PtrsCastable.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 11:48:00 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Mar  2 17:36:18 2016
// Update Count     : 8
//

#include "AST/Decl.hpp"
#include "AST/Pass.hpp"
#include "AST/Type.hpp"
#include "AST/TypeEnvironment.hpp"
#include "Common/PassVisitor.h"
#include "ResolvExpr/TypeEnvironment.h"  // for EqvClass, TypeEnvironment
#include "SymTab/Indexer.h"              // for Indexer
#include "SynTree/Declaration.h"         // for TypeDecl, TypeDecl::Kind::Ftype
#include "SynTree/Type.h"                // for TypeInstType, Type, BasicType
#include "SynTree/Visitor.h"             // for Visitor
#include "typeops.h"                     // for ptrsAssignable

namespace ResolvExpr {
	struct PtrsCastable_old : public WithShortCircuiting  {
	  public:
		PtrsCastable_old( Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer );

		int get_result() const { return result; }

		void previsit( Type * ) { visit_children = false; }

		void postvisit( VoidType * voidType );
		void postvisit( BasicType * basicType );
		void postvisit( PointerType * pointerType );
		void postvisit( ArrayType * arrayType );
		void postvisit( FunctionType * functionType );
		void postvisit( StructInstType * inst );
		void postvisit( UnionInstType * inst );
		void postvisit( EnumInstType * inst );
		void postvisit( TraitInstType * inst );
		void postvisit( TypeInstType * inst );
		void postvisit( TupleType * tupleType );
		void postvisit( VarArgsType * varArgsType );
		void postvisit( ZeroType * zeroType );
		void postvisit( OneType * oneType );
	  private:
		Type *dest;
		int result;
		const TypeEnvironment &env;
		const SymTab::Indexer &indexer;
	};

	namespace {
		int objectCast( Type *src, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
			if ( dynamic_cast< FunctionType* >( src ) ) {
				return -1;
			} else if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( src ) ) {
				if ( NamedTypeDecl *ntDecl = indexer.lookupType( typeInst->get_name() ) ) {
					if ( TypeDecl *tyDecl = dynamic_cast< TypeDecl* >( ntDecl ) ) {
						if ( tyDecl->get_kind() == TypeDecl::Ftype ) {
							return -1;
						} // if
					} //if
				} else if ( const EqvClass *eqvClass = env.lookup( typeInst->get_name() ) ) {
					if ( eqvClass->data.kind == TypeDecl::Ftype ) {
						return -1;
					} // if
				} // if
			} //if
			return 1;
		}
		int functionCast( Type *src, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
			return -1 * objectCast( src, env, indexer );  // reverse the sense of objectCast
		}
	}

	int ptrsCastable( Type *src, Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
		if ( TypeInstType *destAsTypeInst = dynamic_cast< TypeInstType* >( dest ) ) {
			if ( const EqvClass *eqvClass = env.lookup( destAsTypeInst->get_name() ) ) {
				// xxx - should this be ptrsCastable?
				return ptrsAssignable( src, eqvClass->type, env );
			} // if
		} // if
		if ( dynamic_cast< VoidType* >( dest ) ) {
			return objectCast( src, env, indexer );
		} else {
			PassVisitor<PtrsCastable_old> ptrs( dest, env, indexer );
			src->accept( ptrs );
			return ptrs.pass.get_result();
		} // if
	}

	PtrsCastable_old::PtrsCastable_old( Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer )
		: dest( dest ), result( 0 ), env( env ), indexer( indexer )	{
	}

	void PtrsCastable_old::postvisit( VoidType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( BasicType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( PointerType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( ArrayType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( FunctionType * ) {
		// result = -1;
		result = functionCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( StructInstType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( UnionInstType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( EnumInstType * ) {
		if ( dynamic_cast< EnumInstType* >( dest ) ) {
			result = 1;
		} else if ( BasicType *bt = dynamic_cast< BasicType* >( dest ) ) {
			if ( bt->get_kind() == BasicType::SignedInt ) {
				result = 0;
			} else {
				result = 1;
			}
		} else {
			result = objectCast( dest, env, indexer );
		}
	}

	void PtrsCastable_old::postvisit( TraitInstType * ) {}

	void PtrsCastable_old::postvisit(TypeInstType *inst ) {
		//result = objectCast( inst, env, indexer ) > 0 && objectCast( dest, env, indexer ) > 0 ? 1 : -1;
		result = objectCast( inst, env, indexer ) == objectCast( dest, env, indexer ) ? 1 : -1;
	}

	void PtrsCastable_old::postvisit( TupleType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( VarArgsType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( ZeroType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable_old::postvisit( OneType * ) {
		result = objectCast( dest, env, indexer );
	}

namespace {
	// can this type be cast to an object (1 for yes, -1 for no)
	int objectCast( 
		const ast::Type * src, const ast::TypeEnvironment & env, const ast::SymbolTable & symtab 
	) {
		if ( dynamic_cast< const ast::FunctionType * >( src ) ) {
			return -1;
		} else if ( auto inst = dynamic_cast< const ast::TypeInstType * >( src ) ) {
			if ( const ast::NamedTypeDecl * named = symtab.lookupType( inst->name ) ) {
				if ( auto tyDecl = dynamic_cast< const ast::TypeDecl * >( named ) ) {
					if ( tyDecl->kind == ast::TypeVar::Ftype ) {
						return -1;
					}
				}
			} else if ( const ast::EqvClass * eqvClass = env.lookup( inst->name ) ) {
				if ( eqvClass->data.kind == ast::TypeVar::Ftype ) {
					return -1;
				}
			}
		}

		return 1;
	}

	// can this type be cast to a function (inverse of objectCast)
	int functionCast( 
		const ast::Type * src, const ast::TypeEnvironment & env, const ast::SymbolTable & symtab 
	) {
		return -1 * objectCast( src, env, symtab );
	}

	class PtrsCastable_new : public ast::WithShortCircuiting {
		const ast::Type * dst;
		const ast::TypeEnvironment & env;
		const ast::SymbolTable & symtab;
	public:
		int result;

		PtrsCastable_new( 
			const ast::Type * d, const ast::TypeEnvironment & e, const ast::SymbolTable & syms )
		: dst( d ), env( e ), symtab( syms ), result( 0 ) {}

		void previsit( const ast::Type * ) { visit_children = false; }

		void postvisit( const ast::VoidType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::BasicType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::PointerType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::ArrayType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::FunctionType * ) {
			result = functionCast( dst, env, symtab );
		}

		void postvisit( const ast::StructInstType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::UnionInstType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::EnumInstType * ) {
			if ( dynamic_cast< const ast::EnumInstType * >( dst ) ) {
				result = 1;
			} else if ( auto bt = dynamic_cast< const ast::BasicType * >( dst ) ) {
				if ( bt->kind == ast::BasicType::SignedInt ) {
					result = 0;
				} else {
					result = 1;
				}
			} else {
				result = objectCast( dst, env, symtab );
			}
		}

		void postvisit( const ast::TraitInstType * ) {}

		void postvisit( const ast::TypeInstType * inst ) {
			// check trait and destination type are both object or both function
			result = objectCast( inst, env, symtab ) == objectCast( dst, env, symtab ) ? 1 : -1;
		}

		void postvisit( const ast::TupleType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::VarArgsType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::ZeroType * ) {
			result = objectCast( dst, env, symtab );
		}

		void postvisit( const ast::OneType * ) {
			result = objectCast( dst, env, symtab );
		}

	};
} // anonymous namespace

int ptrsCastable( 
	const ast::Type * src, const ast::Type * dst, const ast::SymbolTable & symtab, 
	const ast::TypeEnvironment & env 
) {
	if ( auto inst = dynamic_cast< const ast::TypeInstType * >( dst ) ) {
		if ( const ast::EqvClass * eqvClass = env.lookup( inst->name ) ) {
			return ptrsAssignable( src, eqvClass->bound, env );
		}
	}

	if ( dynamic_cast< const ast::VoidType * >( dst ) ) {
		return objectCast( src, env, symtab );
	} else {
		ast::Pass< PtrsCastable_new > ptrs{ dst, env, symtab };
		src->accept( ptrs );
		return ptrs.pass.result;
	}
}

} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
