//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TupleAssignment.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 15:02:53 2015
// Update Count     : 2
//

#include <iterator>
#include <iostream>
#include <cassert>
#include "Tuples.h"
#include "GenPoly/DeclMutator.h"
#include "SynTree/Mutator.h"
#include "SynTree/Statement.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SymTab/Mangler.h"
#include "Common/ScopedMap.h"

namespace Tuples {
	namespace {
		class UniqueExprExpander : public GenPoly::DeclMutator {
		public:
			typedef GenPoly::DeclMutator Parent;
			virtual Expression * mutate( UniqueExpr * unqExpr );
			std::map< Expression *, ObjectDecl * > decls;
		};

		class TupleAssignExpander : public Mutator {
		public:
			typedef Mutator Parent;
			virtual Expression * mutate( TupleAssignExpr * tupleExpr );
		};

		class TupleTypeReplacer : public GenPoly::DeclMutator {
		  public:
			typedef GenPoly::DeclMutator Parent;

			virtual Type * mutate( TupleType * tupleType );

			virtual CompoundStmt * mutate( CompoundStmt * stmt ) {
				typeMap.beginScope();
				stmt = Parent::mutate( stmt );
				typeMap.endScope();
				return stmt;
			}
		  private:
			ScopedMap< std::string, StructDecl * > typeMap;
		};

		class TupleIndexExpander : public Mutator {
		public:
			typedef Mutator Parent;
			virtual Expression * mutate( TupleIndexExpr * tupleExpr );
		};

		class TupleExprExpander : public Mutator {
		public:
			typedef Mutator Parent;
			virtual Expression * mutate( TupleExpr * tupleExpr );
		};
	}

	void expandTuples( std::list< Declaration * > & translationUnit ) {
		UniqueExprExpander unqExpander;
		unqExpander.mutateDeclarationList( translationUnit );

		TupleAssignExpander assnExpander;
		mutateAll( translationUnit, assnExpander );

		TupleTypeReplacer replacer;
		replacer.mutateDeclarationList( translationUnit );

		TupleIndexExpander idxExpander;
		mutateAll( translationUnit, idxExpander );

		TupleExprExpander exprExpander;
		mutateAll( translationUnit, exprExpander );
	}

	Expression * UniqueExprExpander::mutate( UniqueExpr * unqExpr ) {
		static UniqueName tempNamer( "_unq_expr_" );
		unqExpr = safe_dynamic_cast< UniqueExpr * > ( Parent::mutate( unqExpr ) );
		if ( ! decls.count( unqExpr->get_expr() ) ) {
			// xxx - it's possible (likely?) that expressions can appear in the wrong order because of this. Need to ensure they're placed in the correct location.
			ObjectDecl * decl = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, nullptr, unqExpr->get_result()->clone(), new SingleInit( unqExpr->get_expr()->clone() ) );
			decls[unqExpr->get_expr()] = decl;
			addDeclaration( decl );
		}
		return new VariableExpr( decls[unqExpr->get_expr()] );
	}

	Expression * TupleAssignExpander::mutate( TupleAssignExpr * assnExpr ) {
		// xxx - Parent::mutate?
		CompoundStmt * compoundStmt = new CompoundStmt( noLabels );
		std::list< Statement * > & stmts = compoundStmt->get_kids();
		for ( ObjectDecl * obj : assnExpr->get_tempDecls() ) {
			stmts.push_back( new DeclStmt( noLabels, obj ) );
		}
		TupleExpr * tupleExpr = new TupleExpr( assnExpr->get_assigns() );
		assert( tupleExpr->get_result() );
		stmts.push_back( new ExprStmt( noLabels, tupleExpr ) );
		assnExpr->get_tempDecls().clear();
		assnExpr->get_assigns().clear();
		delete assnExpr;
		return new StmtExpr( compoundStmt );
	}

	Type * TupleTypeReplacer::mutate( TupleType * tupleType ) {
		std::string mangleName = SymTab::Mangler::mangleType( tupleType );
		TupleType * newType = safe_dynamic_cast< TupleType * > ( Parent::mutate( tupleType ) );
		if ( ! typeMap.count( mangleName ) ) {
			// generate struct type to replace tuple type
			StructDecl * decl = new StructDecl( "_tuple_type_" + mangleName );
			decl->set_body( true );
			int cnt = 0;
			for ( Type * t : *newType ) {
				decl->get_members().push_back( new ObjectDecl( "field_"+std::to_string(++cnt), DeclarationNode::NoStorageClass, LinkageSpec::C, nullptr, t->clone(), nullptr ) );
			}
			typeMap[mangleName] = decl;
			addDeclaration( decl );
		}
		Type::Qualifiers qualifiers = newType->get_qualifiers();
		delete newType;
		return new StructInstType( qualifiers, typeMap[mangleName] );
	}

	Expression * TupleIndexExpander::mutate( TupleIndexExpr * tupleExpr ) {
		Expression * tuple = maybeMutate( tupleExpr->get_tuple(), *this );
		assert( tuple );
		tupleExpr->set_tuple( nullptr );
		unsigned int idx = tupleExpr->get_index();
		delete tupleExpr;

		StructInstType * type = safe_dynamic_cast< StructInstType * >( tuple->get_result() );
		StructDecl * structDecl = type->get_baseStruct();
		assert( structDecl->get_members().size() > idx );
		Declaration * member = *std::next(structDecl->get_members().begin(), idx);
		return new MemberExpr( safe_dynamic_cast< DeclarationWithType * >( member ), tuple );
	}

	Expression * TupleExprExpander::mutate( TupleExpr * tupleExpr ) {
		assert( tupleExpr->get_result() );
		std::list< Initializer * > inits;
		for ( Expression * expr : tupleExpr->get_exprs() ) {
			inits.push_back( new SingleInit( expr ) );
		}
		return new CompoundLiteralExpr( tupleExpr->get_result(), new ListInit( inits ) );
	}

	TupleType * makeTupleType( const std::list< Expression * > & exprs ) {
		TupleType *tupleType = new TupleType( Type::Qualifiers(true, true, true, true, true, false) );
		Type::Qualifiers &qualifiers = tupleType->get_qualifiers();
		for ( Expression * expr : exprs ) {
			assert( expr->get_result() );
			Type * type = expr->get_result()->clone();
			tupleType->get_types().push_back( type );
			qualifiers &= type->get_qualifiers();
		} // for
		return tupleType;
	}
} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //

