//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TupleAssignment.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 15:02:53 2015
// Update Count     : 2
//

#include "ResolvExpr/AlternativeFinder.h"
#include "ResolvExpr/Alternative.h"
#include "ResolvExpr/typeops.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "Tuples.h"
#include "Common/SemanticError.h"

#include <functional>
#include <algorithm>
#include <iterator>
#include <iostream>
#include <cassert>
#include <set>
#include <unordered_set>

namespace Tuples {
	class TupleAssignSpotter {
	  public:
		// dispatcher for Tuple (multiple and mass) assignment operations
		TupleAssignSpotter( ResolvExpr::AlternativeFinder & );
		void spot( UntypedExpr * expr, std::list<ResolvExpr::AltList> &possibilities );

	  private:
		void match();

		struct Matcher {
		  public:
			Matcher( TupleAssignSpotter &spotter, ResolvExpr::Alternative & lhs, ResolvExpr::Alternative & rhs );
			virtual ~Matcher() {}
			virtual void match( std::list< Expression * > &out ) = 0;
			ResolvExpr::AltList lhs, rhs;
			TupleAssignSpotter &spotter;
			std::list< ObjectDecl * > tmpDecls;
		};

		struct MassAssignMatcher : public Matcher {
		  public:
			MassAssignMatcher( TupleAssignSpotter &spotter, ResolvExpr::Alternative & lhs, ResolvExpr::Alternative & rhs );
			virtual void match( std::list< Expression * > &out );
		};

		struct MultipleAssignMatcher : public Matcher {
		  public:
			MultipleAssignMatcher( TupleAssignSpotter &spot, ResolvExpr::Alternative & lhs, ResolvExpr::Alternative & rhs );
			virtual void match( std::list< Expression * > &out );
		};

		ResolvExpr::AlternativeFinder &currentFinder;
		// Expression *rhs, *lhs;
		Matcher *matcher = nullptr;
	};

	/// true if expr is an expression of tuple type, i.e. a tuple expression, tuple variable, or MRV (multiple-return-value) function
	bool isTuple( Expression *expr ) {
		if ( ! expr ) return false;
		assert( expr->has_result() );
		return dynamic_cast<TupleExpr *>(expr) || expr->get_result()->size() > 1;
	}

	bool pointsToTuple( Expression *expr ) {
		// also check for function returning tuple of reference types
		if ( AddressExpr *addr = dynamic_cast< AddressExpr * >( expr) ) {
			return isTuple( addr->get_arg() );
		}
		return false;
	}

	void handleTupleAssignment( ResolvExpr::AlternativeFinder & currentFinder, UntypedExpr * expr, std::list<ResolvExpr::AltList> &possibilities ) {
		TupleAssignSpotter spotter( currentFinder );
		spotter.spot( expr, possibilities );
	}

	TupleAssignSpotter::TupleAssignSpotter( ResolvExpr::AlternativeFinder &f )
		: currentFinder(f) {}

	void TupleAssignSpotter::spot( UntypedExpr * expr, std::list<ResolvExpr::AltList> &possibilities ) {
		if (  NameExpr *assgnop = dynamic_cast< NameExpr * >(expr->get_function()) ) {
			if ( assgnop->get_name() == "?=?" ) {
				for ( std::list<ResolvExpr::AltList>::iterator ali = possibilities.begin(); ali != possibilities.end(); ++ali ) {
					if ( ali->size() != 2 ) continue; // what does it mean if an assignment takes >2 arguments? grab args 2-N and group into a TupleExpr, then proceed?
					ResolvExpr::Alternative & alt1 = ali->front(), & alt2 = ali->back();

					if ( pointsToTuple(alt1.expr) ) {
						MultipleAssignMatcher multiMatcher( *this, alt1, alt2 );
						MassAssignMatcher massMatcher( *this,  alt1, alt2 );
						if ( isTuple( alt2.expr ) ) {
							matcher = &multiMatcher;
						} else {
							// mass assignment
							matcher = &massMatcher;
						}
						match();
					} else if ( isTuple( alt2.expr ) ) {
						throw SemanticError("Cannot assign a tuple value into a non-tuple lvalue.", expr);
					}
				}
			}
		}
	}

	void TupleAssignSpotter::match() {
		assert ( matcher != 0 );

		std::list< Expression * > new_assigns;
		matcher->match( new_assigns );

		if ( new_assigns.empty() ) return;
		ResolvExpr::AltList current;
		// now resolve new assignments
		for ( std::list< Expression * >::iterator i = new_assigns.begin(); i != new_assigns.end(); ++i ) {
			ResolvExpr::AlternativeFinder finder( currentFinder.get_indexer(), currentFinder.get_environ() );
			finder.findWithAdjustment(*i);
			// prune expressions that don't coincide with
			ResolvExpr::AltList alts = finder.get_alternatives();
			assert( alts.size() == 1 );
			assert( alts.front().expr != 0 );
			current.push_back( alts.front() );
		}

		// extract expressions from the assignment alternatives to produce a list of assignments that
		// together form a single alternative
		std::list< Expression *> solved_assigns;
		for ( ResolvExpr::Alternative & alt : current ) {
			solved_assigns.push_back( alt.expr->clone() );
		}
		// xxx - need to do this??
		// TypeEnvironment compositeEnv;
		// simpleCombineEnvironments( i->begin(), i->end(), compositeEnv );
		currentFinder.get_alternatives().push_front( ResolvExpr::Alternative(new TupleAssignExpr(solved_assigns, matcher->tmpDecls), currentFinder.get_environ(), ResolvExpr::sumCost( current ) ) );
	}

	TupleAssignSpotter::Matcher::Matcher( TupleAssignSpotter &spotter, ResolvExpr::Alternative & lhs, ResolvExpr::Alternative & rhs ) : spotter(spotter) {
		if (AddressExpr *addr = dynamic_cast<AddressExpr *>(lhs.expr) ) {
			// xxx - not every assignment NEEDS to have the first argument as address-taken, e.g. a manual call to assignment. What to do in this case? skip it as a possibility for TupleAssignment, since the type will always be T*, where T can never be a tuple? Is this true?

			// explode the lhs so that each field of the tuple-valued-expr is assigned.
			ResolvExpr::Alternative lhsAlt( addr->get_arg()->clone(), lhs.env, lhs.cost, lhs.cvtCost );
			explode( lhsAlt, back_inserter(this->lhs) );
		}
	}

	TupleAssignSpotter::MassAssignMatcher::MassAssignMatcher( TupleAssignSpotter &spotter, ResolvExpr::Alternative & lhs, ResolvExpr::Alternative & rhs ) : Matcher( spotter, lhs, rhs ) {
		this->rhs.push_back( rhs );
	}

	TupleAssignSpotter::MultipleAssignMatcher::MultipleAssignMatcher( TupleAssignSpotter &spotter, ResolvExpr::Alternative & lhs, ResolvExpr::Alternative & rhs ) : Matcher( spotter, lhs, rhs ) {

		// explode the rhs so that each field of the tuple-valued-expr is assigned.
		explode( rhs, back_inserter(this->rhs) );
	}

	UntypedExpr * createAssgn( ObjectDecl *left, ObjectDecl *right ) {
		assert( left && right );
		std::list< Expression * > args;
		args.push_back( new AddressExpr( new UntypedExpr( new NameExpr("*?"), std::list< Expression * >{ new VariableExpr( left ) } ) ) );
		args.push_back( new VariableExpr( right ) );
		return new UntypedExpr( new NameExpr( "?=?" ), args );
	}

	ObjectDecl * newObject( UniqueName & namer, Expression * expr ) {
		assert( expr->has_result() && ! expr->get_result()->isVoid() );
		return new ObjectDecl( namer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, nullptr, expr->get_result()->clone(), new SingleInit( expr->clone() ) );
	}

	void TupleAssignSpotter::MassAssignMatcher::match( std::list< Expression * > &out ) {
		static UniqueName lhsNamer( "__massassign_L" );
		static UniqueName rhsNamer( "__massassign_R" );
		assert ( ! lhs.empty() && rhs.size() == 1);

		ObjectDecl * rtmp = newObject( rhsNamer, rhs.front().expr );
		for ( ResolvExpr::Alternative & lhsAlt : lhs ) {
			ObjectDecl * ltmp = newObject( lhsNamer, new AddressExpr( lhsAlt.expr ) );
			out.push_back( createAssgn( ltmp, rtmp ) );
			tmpDecls.push_back( ltmp );
		}
		tmpDecls.push_back( rtmp );
	}

	void TupleAssignSpotter::MultipleAssignMatcher::match( std::list< Expression * > &out ) {
		static UniqueName lhsNamer( "__multassign_L" );
		static UniqueName rhsNamer( "__multassign_R" );
		// xxx - need more complicated matching?
		if ( lhs.size() == rhs.size() ) {
			std::list< ObjectDecl * > ltmp;
			std::list< ObjectDecl * > rtmp;
			std::transform( lhs.begin(), lhs.end(), back_inserter( ltmp ), []( ResolvExpr::Alternative & alt ){
				return newObject( lhsNamer, new AddressExpr( alt.expr ) );
			});
			std::transform( rhs.begin(), rhs.end(), back_inserter( rtmp ), []( ResolvExpr::Alternative & alt ){
				return newObject( rhsNamer, alt.expr );
			});
			zipWith( ltmp.begin(), ltmp.end(), rtmp.begin(), rtmp.end(), back_inserter(out), createAssgn );
			tmpDecls.splice( tmpDecls.end(), ltmp );
			tmpDecls.splice( tmpDecls.end(), rtmp );
		}
	}
} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
