//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TupleExpr.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 10:59:19 2015
// Update Count     : 1
//

#include "Expression.h"
#include "Common/utility.h"
#include "Type.h"
#include "Declaration.h"
#include "Tuples/Tuples.h"

TupleExpr::TupleExpr( const std::list< Expression * > & exprs, Expression *_aname ) : Expression( _aname ), exprs( exprs ) {
	if ( ! exprs.empty() ) {
		if ( std::all_of( exprs.begin(), exprs.end(), [](Expression * expr) { return expr->get_result(); } ) ) {
			set_result( Tuples::makeTupleType( exprs ) );
		}
	}
}

TupleExpr::TupleExpr( const TupleExpr &other ) : Expression( other ) {
	cloneAll( other.exprs, exprs );
}

TupleExpr::~TupleExpr() {
	deleteAll( exprs );
}

void TupleExpr::print( std::ostream &os, int indent ) const {
	os << "Tuple:" << std::endl;
	printAll( exprs, os, indent+2 );
	Expression::print( os, indent );
}

TupleIndexExpr::TupleIndexExpr( Expression * tuple, unsigned int index ) : tuple( tuple ), index( index )  {
	TupleType * type = safe_dynamic_cast< TupleType * >( tuple->get_result() );
	assert( type->size() > index );
	set_result( (*std::next( type->get_types().begin(), index ))->clone() );
}

TupleIndexExpr::TupleIndexExpr( const TupleIndexExpr &other ) : Expression( other ), tuple( other.tuple->clone() ), index( other.index ) {
}

TupleIndexExpr::~TupleIndexExpr() {
	delete tuple;
}

void TupleIndexExpr::print( std::ostream &os, int indent ) const {
	os << "Tuple Index Expression, with tuple:" << std::endl;
	os << std::string( indent+2, ' ' );
	tuple->print( os, indent+2 );
	os << std::string( indent+2, ' ' ) << "with index: " << index << std::endl;
	Expression::print( os, indent );
}

MemberTupleExpr::MemberTupleExpr( Expression * member, Expression * aggregate, Expression * _aname ) : Expression( _aname ) {
	set_result( maybeClone( member->get_result() ) ); // xxx - ???
}

MemberTupleExpr::MemberTupleExpr( const MemberTupleExpr &other ) : Expression( other ), member( other.member->clone() ), aggregate( other.aggregate->clone() ) {
}

MemberTupleExpr::~MemberTupleExpr() {
	delete member;
	delete aggregate;
}

void MemberTupleExpr::print( std::ostream &os, int indent ) const {
	os << "Member Tuple Expression, with aggregate:" << std::endl;
	os << std::string( indent+2, ' ' );
	aggregate->print( os, indent+2 );
	os << std::string( indent+2, ' ' ) << "with member: " << std::endl;
	os << std::string( indent+2, ' ' );
	member->print( os, indent+2 );
	Expression::print( os, indent );
}


TupleAssignExpr::TupleAssignExpr( const std::list< Expression * > & assigns, const std::list< ObjectDecl * > & tempDecls, Expression * _aname ) : Expression( _aname ), assigns( assigns ), tempDecls( tempDecls ) {
	TupleType * type = new TupleType( Type::Qualifiers() );
	for ( Expression * expr : assigns ) {
		assert( expr->has_result() );
		type->get_types().push_back( expr->get_result()->clone() );
	}
	set_result( type );
}

TupleAssignExpr::TupleAssignExpr( const TupleAssignExpr &other ) : Expression( other ), tempDecls( other.tempDecls ) /* temporary */ {
	cloneAll( other.assigns, assigns );
	// xxx - clone needs to go into assigns and replace tempDecls
}

TupleAssignExpr::~TupleAssignExpr() {
	deleteAll( assigns );
	// deleteAll( tempDecls );
}

void TupleAssignExpr::print( std::ostream &os, int indent ) const {
	os << "Tuple Assignment Expression, with temporaries:" << std::endl;
	printAll( tempDecls, os, indent+4 );
	os << std::string( indent+2, ' ' ) << "with assignments: " << std::endl;
	printAll( assigns, os, indent+4 );
	Expression::print( os, indent );
}



// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
