//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Autogen.h --
//
// Author           : Rob Schluntz
// Created On       : Sun May 17 21:53:34 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue May 19 16:49:43 2015
// Update Count     : 1
//

#ifndef AUTOGEN_H
#define AUTOGEN_H

#include <string>
#include "SynTree/Statement.h"
#include "SynTree/Expression.h"
#include "SynTree/Declaration.h"
#include "SynTree/Initializer.h"
#include "InitTweak/InitTweak.h"

namespace SymTab {
	/// Generates assignment operators, constructors, and destructor for aggregate types as required
	void autogenerateRoutines( std::list< Declaration * > &translationUnit );

	/// returns true if obj's name is the empty string and it has a bitfield width
	bool isUnnamedBitfield( ObjectDecl * obj );

	/// size_t type - set when size_t typedef is seen. Useful in a few places,
	/// such as in determining array dimension type
	extern Type * SizeType;

	/// inserts into out a generated call expression to function fname with arguments dstParam and srcParam. Intended to be used with generated ?=?, ?{}, and ^?{} calls.
	template< typename OutputIterator >
	void genCall( InitTweak::InitExpander & srcParam, Expression * dstParam, const std::string & fname, OutputIterator out, Type * type, bool forward = true );

	/// inserts into out a generated call expression to function fname with arguments dstParam and srcParam. Should only be called with non-array types.
	template< typename OutputIterator >
	void genScalarCall( InitTweak::InitExpander & srcParam, Expression *dstParam, const std::string & fname, OutputIterator out ) {
		// want to be able to generate assignment, ctor, and dtor generically,
		// so fname is either ?=?, ?{}, or ^?{}
		UntypedExpr *fExpr = new UntypedExpr( new NameExpr( fname ) );

		// do something special for unnamed members
		fExpr->get_args().push_back( new AddressExpr( dstParam ) );

    Statement * listInit = srcParam.buildListInit( fExpr );
    if ( listInit ) {
      *out++ = listInit;
    }

    std::list< Expression * > args = *++srcParam;
    fExpr->get_args().splice( fExpr->get_args().end(), args );
/*		if ( srcParam ) {
			// xxx -
			// make srcParam more complicated
			// if srcParam contains
			fExpr->get_args().push_back( srcParam );
		}
*/
		*out++ = new ExprStmt( noLabels, fExpr );
	}

	/// Store in out a loop which calls fname on each element of the array with srcParam and dstParam as arguments.
	/// If forward is true, loop goes from 0 to N-1, else N-1 to 0
	template< typename OutputIterator >
	void genArrayCall( InitTweak::InitExpander & srcParam, Expression *dstParam, const std::string & fname, OutputIterator out, ArrayType *array, bool forward = true ) {
		static UniqueName indexName( "_index" );

		// for a flexible array member nothing is done -- user must define own assignment
		if ( ! array->get_dimension() ) return ;

		Expression * begin, * end, * update, * cmp;
		if ( forward ) {
			// generate: for ( int i = 0; i < 0; ++i )
			begin = new NameExpr( "0" );
			end = array->get_dimension()->clone();
			cmp = new NameExpr( "?<?" );
			update = new NameExpr( "++?" );
		} else {
			// generate: for ( int i = N-1; i >= 0; --i )
			begin = new UntypedExpr( new NameExpr( "?-?" ) );
			((UntypedExpr*)begin)->get_args().push_back( array->get_dimension()->clone() );
			((UntypedExpr*)begin)->get_args().push_back( new NameExpr( "1" ) );
			end = new NameExpr( "0" );
			cmp = new NameExpr( "?>=?" );
			update = new NameExpr( "--?" );
		}

		ObjectDecl *index = new ObjectDecl( indexName.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new BasicType( Type::Qualifiers(), BasicType::SignedInt ), NULL );

		UntypedExpr *init = new UntypedExpr( new NameExpr( "?=?" ) );
		init->get_args().push_back( new AddressExpr( new VariableExpr( index ) ) );
		init->get_args().push_back( begin );
		index->set_init( new SingleInit( init, std::list<Expression*>() ) );

		UntypedExpr *cond = new UntypedExpr( cmp );
		cond->get_args().push_back( new VariableExpr( index ) );
		cond->get_args().push_back( end );

		UntypedExpr *inc = new UntypedExpr( update );
		inc->get_args().push_back( new AddressExpr( new VariableExpr( index ) ) );

		UntypedExpr *dstIndex = new UntypedExpr( new NameExpr( "?[?]" ) );
		dstIndex->get_args().push_back( dstParam );
		dstIndex->get_args().push_back( new VariableExpr( index ) );
		dstParam = dstIndex;

		// srcParam must keep track of the array indices to build the
		// source parameter and/or array list initializer
		srcParam.addArrayIndex( new VariableExpr( index ), array->get_dimension()->clone() );

		// if ( srcParam ) {
		// 	UntypedExpr *srcIndex = new UntypedExpr( new NameExpr( "?[?]" ) );
		// 	srcIndex->get_args().push_back( srcParam );
		// 	srcIndex->get_args().push_back( new VariableExpr( index ) );
		// 	srcParam = srcIndex;
		// }

		// for stmt's body, eventually containing call
		CompoundStmt * body = new CompoundStmt( noLabels );
		genCall( srcParam, dstParam, fname, back_inserter( body->get_kids() ), array->get_base(), forward );

		// block containing for stmt and index variable
		std::list<Statement *> initList;
		CompoundStmt * block = new CompoundStmt( noLabels );
		block->get_kids().push_back( new DeclStmt( noLabels, index ) );
		block->get_kids().push_back( new ForStmt( noLabels, initList, cond, inc, body ) );

		*out++ = block;
	}

	template< typename OutputIterator >
	void genCall( InitTweak::InitExpander &  srcParam, Expression * dstParam, const std::string & fname, OutputIterator out, Type * type, bool forward ) {
		if ( ArrayType * at = dynamic_cast< ArrayType * >( type ) ) {
			genArrayCall( srcParam, dstParam, fname, out, at, forward );
		} else {
			genScalarCall( srcParam, dstParam, fname, out );
		}
	}

	/// inserts into out a generated call expression to function fname with arguments dstParam
	/// and srcParam. Intended to be used with generated ?=?, ?{}, and ^?{} calls. decl is the
	/// object being constructed. The function wraps constructor and destructor calls in an
	/// ImplicitCtorDtorStmt node.
	template< typename OutputIterator >
	void genImplicitCall( InitTweak::InitExpander &  srcParam, Expression * dstParam, const std::string & fname, OutputIterator out, DeclarationWithType * decl, bool forward = true ) {
		ObjectDecl *obj = dynamic_cast<ObjectDecl *>( decl );
		assert( obj );
		// unnamed bit fields are not copied as they cannot be accessed
		if ( isUnnamedBitfield( obj ) ) return;

		std::list< Statement * > stmts;
		genCall( srcParam, dstParam, fname, back_inserter( stmts ), obj->get_type(), forward );

		// currently genCall should produce at most one element, but if that changes then the next line needs to be updated to grab the statement which contains the call
		assert( stmts.size() <= 1 );
    if ( stmts.size() == 1 ) {
  		Statement * callStmt = stmts.front();
  		if ( (fname == "?{}" || fname == "^?{}") && ( !obj || ( obj && obj->get_bitfieldWidth() == NULL ) ) ) {
  			// implicitly generated ctor/dtor calls should be wrapped
  			// so that later passes are aware they were generated.
  			// xxx - don't mark as an implicit ctor/dtor if obj is a bitfield,
  			// because this causes the address to be taken at codegen, which is illegal in C.
  			callStmt = new ImplicitCtorDtorStmt( callStmt );
  		}
  		*out++ = callStmt;
    }
	}
} // namespace SymTab
#endif // AUTOGEN_H

