//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// FixInit.h --
//
// Author           : Rob Schluntz
// Created On       : Wed Jan 13 16:29:30 2016
// Last Modified By : Rob Schluntz
// Last Modified On : Thu Mar 31 13:54:58 2016
// Update Count     : 30
//

#include <stack>
#include <list>
#include "RemoveInit.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Initializer.h"
#include "SynTree/Mutator.h"
#include "GenPoly/PolyMutator.h"

namespace InitTweak {
	namespace {
		const std::list<Label> noLabels;
		const std::list<Expression*> noDesignators;
	}

	class FixInit : public GenPoly::PolyMutator {
	  public:
		static void fixInitializers( std::list< Declaration * > &translationUnit );

		virtual ObjectDecl * mutate( ObjectDecl *objDecl );

		virtual CompoundStmt * mutate( CompoundStmt * compoundStmt );
		virtual Statement * mutate( ReturnStmt * returnStmt );
		virtual Statement * mutate( BranchStmt * branchStmt );

	  private:
		// stack of list of statements - used to differentiate scopes
		std::list< std::list< Statement * > > dtorStmts;
	};

	void fix( std::list< Declaration * > & translationUnit ) {
		FixInit::fixInitializers( translationUnit );
	}

	void FixInit::fixInitializers( std::list< Declaration * > & translationUnit ) {
		FixInit fixer;
		mutateAll( translationUnit, fixer );
	}

	ObjectDecl *FixInit::mutate( ObjectDecl *objDecl ) {
		if ( ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() ) ) {
			// a decision should have been made by the resolver, so ctor and init are not both non-NULL
			assert( ! ctorInit->get_ctor() || ! ctorInit->get_init() );
			if ( Statement * ctor = ctorInit->get_ctor() ) {
				if ( objDecl->get_storageClass() == DeclarationNode::Static ) {
					// generate:
					// static bool __objName_uninitialized = true;
					// if (__objName_uninitialized) {
					//   __ctor(__objName);
					//   void dtor_atexit() {
					//     __dtor(__objName);
					//   }
					//   on_exit(dtorOnExit, &__objName);
					//   __objName_uninitialized = false;
					// }

					// generate first line
					BasicType * boolType = new BasicType( Type::Qualifiers(), BasicType::Bool );
					SingleInit * boolInitExpr = new SingleInit( new ConstantExpr( Constant( boolType->clone(), "1" ) ), noDesignators );
					ObjectDecl * isUninitializedVar = new ObjectDecl( objDecl->get_mangleName() + "_uninitialized", DeclarationNode::Static, LinkageSpec::Cforall, 0, boolType, boolInitExpr );
					isUninitializedVar->fixUniqueId();

					// void dtor_atexit(...) {...}
					FunctionDecl * dtorCaller = new FunctionDecl( objDecl->get_mangleName() + "_dtor_atexit", DeclarationNode::NoStorageClass, LinkageSpec::C, new FunctionType( Type::Qualifiers(), false ), new CompoundStmt( noLabels ), false, false );
					dtorCaller->fixUniqueId();
					dtorCaller->get_statements()->get_kids().push_back( ctorInit->get_dtor() );

					// on_exit(dtor_atexit);
					UntypedExpr * callAtexit = new UntypedExpr( new NameExpr( "atexit" ) );
					callAtexit->get_args().push_back( new VariableExpr( dtorCaller ) );

					// __objName_uninitialized = false;
					UntypedExpr * setTrue = new UntypedExpr( new NameExpr( "?=?" ) );
					setTrue->get_args().push_back( new VariableExpr( isUninitializedVar ) );
					setTrue->get_args().push_back( new ConstantExpr( Constant( boolType->clone(), "0" ) ) );

					// generate body of if
					CompoundStmt * initStmts = new CompoundStmt( noLabels );
					std::list< Statement * > & body = initStmts->get_kids();
					body.push_back( ctor );
					body.push_back( new DeclStmt( noLabels, dtorCaller ) );
					body.push_back( new ExprStmt( noLabels, callAtexit ) );
					body.push_back( new ExprStmt( noLabels, setTrue ) );

					// put it all together
					IfStmt * ifStmt = new IfStmt( noLabels, new VariableExpr( isUninitializedVar ), initStmts, 0 );
					stmtsToAddAfter.push_back( new DeclStmt( noLabels, isUninitializedVar ) );
					stmtsToAddAfter.push_back( ifStmt );
				} else {
					stmtsToAddAfter.push_back( ctor );
					dtorStmts.back().push_front( ctorInit->get_dtor() );
				}
				objDecl->set_init( NULL );
				ctorInit->set_ctor( NULL );
				ctorInit->set_dtor( NULL );  // xxx - only destruct when constructing? Probably not?
			} else if ( Initializer * init = ctorInit->get_init() ) {
				objDecl->set_init( init );
				ctorInit->set_init( NULL );
			} else {
				// no constructor and no initializer, which is okay
				objDecl->set_init( NULL );
			}
			delete ctorInit;
		}
		return objDecl;
	}

	template<typename Iterator, typename OutputIterator>
	void insertDtors( Iterator begin, Iterator end, OutputIterator out ) {
		for ( Iterator it = begin ; it != end ; ++it ) {
			// remove if instrinsic destructor statement
			// xxx - test user manually calling intrinsic functions - what happens?
			if ( ExprStmt * exprStmt = dynamic_cast< ExprStmt * >( *it ) ) {
				ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( exprStmt->get_expr() );
				assert( appExpr );
				VariableExpr * function = dynamic_cast< VariableExpr * >( appExpr->get_function() );
				assert( function );
				// check for Intrinsic only - don't want to remove all overridable dtors because autogenerated dtor
				// will call all member dtors, and some members may have a user defined dtor.
				if ( function->get_var()->get_linkage() == LinkageSpec::Intrinsic ) {
					// don't need to call intrinsic dtor, because it does nothing
				} else {
					// non-intrinsic dtors must be called
					*out++ = (*it)->clone();
				}
			} else {
				// could also be a compound statement with a loop, in the case of an array
				*out++ = (*it)->clone();
			}
		}
	}


	CompoundStmt * FixInit::mutate( CompoundStmt * compoundStmt ) {
		// mutate statements - this will also populate dtorStmts list.
		// don't want to dump all destructors when block is left,
		// just the destructors associated with variables defined in this block,
		// so push a new list to the top of the stack so that we can differentiate scopes
		dtorStmts.push_back( std::list<Statement *>() );

		compoundStmt = PolyMutator::mutate( compoundStmt );
		std::list< Statement * > & statements = compoundStmt->get_kids();

		insertDtors( dtorStmts.back().begin(), dtorStmts.back().end(), back_inserter( statements ) );

		deleteAll( dtorStmts.back() );
		dtorStmts.pop_back();
		return compoundStmt;
	}

	Statement * FixInit::mutate( ReturnStmt * returnStmt ) {
		for ( std::list< std::list< Statement * > >::reverse_iterator list = dtorStmts.rbegin(); list != dtorStmts.rend(); ++list ) {
			insertDtors( list->begin(), list->end(), back_inserter( stmtsToAdd ) );
		}
		return returnStmt;
	}

	Statement * FixInit::mutate( BranchStmt * branchStmt ) {
		// TODO: adding to the end of a block isn't sufficient, since
		// return/break/goto should trigger destructor when block is left.
		switch( branchStmt->get_type() ) {
			case BranchStmt::Continue:
			case BranchStmt::Break:
				insertDtors( dtorStmts.back().begin(), dtorStmts.back().end(), back_inserter( stmtsToAdd ) );
				break;
			case BranchStmt::Goto:
				// xxx
				// if goto leaves a block, generate dtors for every block it leaves
				// if goto is in same block but earlier statement, destruct every object that was defined after the statement
				break;
			default:
				assert( false );
		}
		return branchStmt;
	}


} // namespace InitTweak

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
