//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// GenPoly.h --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Nov 24 15:24:38 2015
// Update Count     : 6
//

#ifndef GENPOLY_H
#define GENPOLY_H

#include <string>
#include <iostream>
#include <utility>

#include "ErasableScopedMap.h"

#include "SymTab/Mangler.h"

#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/TypeSubstitution.h"

namespace GenPoly {
	typedef ErasableScopedMap< std::string, TypeDecl::Kind > TyVarMap;

	/// Replaces a TypeInstType by its referrent in the environment, if applicable
	Type* replaceTypeInst( Type* type, const TypeSubstitution* env );

	/// returns polymorphic type if is polymorphic type, NULL otherwise; will look up substitution in env if provided
	Type *isPolyType( Type *type, const TypeSubstitution *env = 0 );

	/// returns polymorphic type if is polymorphic type in tyVars, NULL otherwise; will look up substitution in env if provided
	Type *isPolyType( Type *type, const TyVarMap &tyVars, const TypeSubstitution *env = 0 );

	/// returns dynamic-layout type if is dynamic-layout type in tyVars, NULL otherwise; will look up substitution in env if provided
	ReferenceToType *isDynType( Type *type, const TyVarMap &tyVars, const TypeSubstitution *env = 0 );

	/// true iff function has dynamic-layout return type under the given type variable map
	ReferenceToType *isDynRet( FunctionType *function, const TyVarMap &tyVars );

	/// true iff function has dynamic-layout return type under the type variable map generated from its forall-parameters
	ReferenceToType *isDynRet( FunctionType *function );

	/// A function needs an adapter if it returns a dynamic-layout value or if any of its parameters have dynamic-layout type
	bool needsAdapter( FunctionType *adaptee, const TyVarMap &tyVarr );

	/// returns polymorphic type if is pointer to polymorphic type, NULL otherwise; will look up substitution in env if provided
	Type *isPolyPtr( Type *type, const TypeSubstitution *env = 0 );

	/// returns polymorphic type if is pointer to polymorphic type in tyVars, NULL otherwise; will look up substitution in env if provided
	Type *isPolyPtr( Type *type, const TyVarMap &tyVars, const TypeSubstitution *env = 0 );

	/// if the base type (after dereferencing N >= 0 pointers) is a polymorphic type, returns the base type, NULL otherwise;
	/// N will be stored in levels, if provided, will look up substitution in env if provided
	Type *hasPolyBase( Type *type, int *levels = 0, const TypeSubstitution *env = 0 );

	/// if the base type (after dereferencing N >= 0 pointers) is a polymorphic type in tyVars, returns the base type, NULL otherwise;
	/// N will be stored in levels, if provided, will look up substitution in env if provided
	Type *hasPolyBase( Type *type, const TyVarMap &tyVars, int *levels = 0, const TypeSubstitution *env = 0 );

	/// Returns a pointer to the base FunctionType if ty is the type of a function (or pointer to one), NULL otherwise
	FunctionType *getFunctionType( Type *ty );

	/// If expr (after dereferencing N >= 0 pointers) is a variable expression, returns the variable expression, NULL otherwise;
	/// N will be stored in levels, if provided
	VariableExpr *getBaseVar( Expression *expr, int *levels = 0 );

	/// Adds the declarations in the forall list of type (and its pointed-to type if it's a pointer type) to `tyVarMap`
	void makeTyVarMap( Type *type, TyVarMap &tyVarMap );

	/// Prints type variable map
	void printTyVarMap( std::ostream &os, const TyVarMap &tyVarMap );

	/// Gets the mangled name of this type; alias for SymTab::Mangler::mangleType().
	inline std::string mangleType( Type *ty ) { return SymTab::Mangler::mangleType( ty ); }

	/// Gets the name of the sizeof parameter for the type, given its mangled name
	inline std::string sizeofName( const std::string &name ) { return std::string( "_sizeof_" ) + name; }

	/// Gets the name of the alignof parameter for the type, given its mangled name
	inline std::string alignofName( const std::string &name ) { return std::string( "_alignof_" ) + name; }

	/// Gets the name of the offsetof parameter for the type, given its mangled name
	inline std::string offsetofName( const std::string &name ) { return std::string( "_offsetof_" ) + name; }

	/// Gets the name of the layout function for a given aggregate type, given its declaration
	inline std::string layoutofName( AggregateDecl *decl ) { return std::string( "_layoutof_" ) + decl->get_name(); }

} // namespace GenPoly

#endif // GENPOLY_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
