//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Decl.cpp --
//
// Author           : Aaron B. Moss
// Created On       : Thu May 9 10:00:00 2019
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Dec 11 16:41:39 2019
// Update Count     : 18
//

#include "Decl.hpp"

#include <cassert>             // for assert, strict_dynamic_cast
#include <iostream>
#include <unordered_map>

#include "Common/utility.h"

#include "Fwd.hpp"             // for UniqueId
#include "Init.hpp"
#include "Node.hpp"            // for readonly
#include "Type.hpp"            // for readonly
#include "Parser/ParseNode.h"  // for DeclarationNode

namespace ast {

// To canonicalize declarations
static UniqueId lastUniqueId = 0;

using IdMapType = std::unordered_map< UniqueId, readonly<Decl> >;
static IdMapType idMap;

void Decl::fixUniqueId() {
	if ( uniqueId ) return;  // ensure only set once
	uniqueId = ++lastUniqueId;
	idMap[ uniqueId ] = this;
}

readonly<Decl> Decl::fromId( UniqueId id ) {
	IdMapType::const_iterator i = idMap.find( id );
	if ( i != idMap.end() ) return i->second;
	return {};
}

// --- FunctionDecl

const Type * FunctionDecl::get_type() const { return type.get(); }
void FunctionDecl::set_type(Type * t) { type = strict_dynamic_cast< FunctionType* >( t ); }

// --- TypeDecl

const char * TypeDecl::typeString() const {
	static const char * kindNames[] = { "sized object type", "sized function type", "sized tuple type" };
	assertf( sizeof(kindNames)/sizeof(kindNames[0]) == DeclarationNode::NoTypeClass-1,
		"typeString: kindNames is out of sync." );
	assertf( kind < sizeof(kindNames)/sizeof(kindNames[0]), "TypeDecl's kind is out of bounds." );
	return sized ? kindNames[ kind ] : &kindNames[ kind ][ sizeof("sized") ]; // sizeof includes '\0'
}

const char * TypeDecl::genTypeString() const {
	static const char * kindNames[] = { "dtype", "ftype", "ttype" };
	assertf( sizeof(kindNames)/sizeof(kindNames[0]) == DeclarationNode::NoTypeClass-1, "genTypeString: kindNames is out of sync." );
	assertf( kind < sizeof(kindNames)/sizeof(kindNames[0]), "TypeDecl's kind is out of bounds." );
	return kindNames[ kind ];
}

std::ostream & operator<< ( std::ostream & out, const TypeDecl::Data & data ) {
	return out << data.kind << ", " << data.isComplete;
}

// --- AggregateDecl

// These must harmonize with the corresponding AggregateDecl::Aggregate enumerations.
static const char * aggregateNames[] = { "struct", "union", "enum", "exception", "trait", "generator", "coroutine", "monitor", "thread", "NoAggregateName" };

const char * AggregateDecl::aggrString( AggregateDecl::Aggregate aggr ) {
	return aggregateNames[aggr];
}

// --- EnumDecl

bool EnumDecl::valueOf( const Decl * enumerator, long long& value ) const {
	if ( enumValues.empty() ) {
		long long crntVal = 0;
		for ( const Decl * member : members ) {
			const ObjectDecl* field = strict_dynamic_cast< const ObjectDecl* >( member );
			if ( field->init ) {
				const SingleInit * init = strict_dynamic_cast< const SingleInit* >( field->init.get() );
				auto result = eval( init->value );
				if ( ! result.second ) {
					SemanticError( init->location, ::toString( "Non-constexpr in initialization of "
						"enumerator: ", field ) );
				}
				crntVal = result.first;
			}
			if ( enumValues.count( field->name ) != 0 ) {
				SemanticError( location, ::toString( "Enum ", name, " has multiple members with the " 	"name ", field->name ) );
			}
			enumValues[ field->name ] = crntVal;
			++crntVal;
		}
	}

	auto it = enumValues.find( enumerator->name );
	if ( it != enumValues.end() ) {
		value = it->second;
		return true;
	}
	return false;
}

}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
