//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Indexer.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:37:33 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Aug 17 16:08:40 2017
// Update Count     : 20
//

#include "Indexer.h"

#include <cassert>                 // for assert, strict_dynamic_cast
#include <iostream>                // for operator<<, basic_ostream, ostream
#include <string>                  // for string, operator<<, operator!=
#include <unordered_map>           // for operator!=, unordered_map<>::const...
#include <unordered_set>           // for unordered_set
#include <utility>                 // for pair, make_pair, move

#include "CodeGen/OperatorTable.h" // for isCtorDtor, isCtorDtorAssign
#include "Common/SemanticError.h"  // for SemanticError
#include "Common/utility.h"        // for cloneAll
#include "GenPoly/GenPoly.h"
#include "InitTweak/InitTweak.h"   // for isConstructor, isCopyFunction, isC...
#include "Mangler.h"               // for Mangler
#include "Parser/LinkageSpec.h"    // for isMangled, isOverridable, Spec
#include "ResolvExpr/typeops.h"    // for typesCompatible
#include "SynTree/Constant.h"      // for Constant
#include "SynTree/Declaration.h"   // for DeclarationWithType, FunctionDecl
#include "SynTree/Expression.h"    // for Expression, ImplicitCopyCtorExpr
#include "SynTree/Initializer.h"   // for Initializer
#include "SynTree/Statement.h"     // for CompoundStmt, Statement, ForStmt (...
#include "SynTree/Type.h"          // for Type, StructInstType, UnionInstType

#define debugPrint(x) if ( doDebug ) { std::cerr << x; }

namespace SymTab {
	std::ostream & operator<<( std::ostream & out, const Indexer::IdData & data ) {
		return out << "(" << data.id << "," << data.baseExpr << ")";
	}

	typedef std::unordered_map< std::string, Indexer::IdData > MangleTable;
	typedef std::unordered_map< std::string, MangleTable > IdTable;
	typedef std::unordered_map< std::string, NamedTypeDecl* > TypeTable;
	typedef std::unordered_map< std::string, StructDecl* > StructTable;
	typedef std::unordered_map< std::string, EnumDecl* > EnumTable;
	typedef std::unordered_map< std::string, UnionDecl* > UnionTable;
	typedef std::unordered_map< std::string, TraitDecl* > TraitTable;

	void dump( const IdTable &table, std::ostream &os ) {
		for ( IdTable::const_iterator id = table.begin(); id != table.end(); ++id ) {
			for ( MangleTable::const_iterator mangle = id->second.begin(); mangle != id->second.end(); ++mangle ) {
				os << mangle->second << std::endl;
			}
		}
	}

	template< typename Decl >
	void dump( const std::unordered_map< std::string, Decl* > &table, std::ostream &os ) {
		for ( typename std::unordered_map< std::string, Decl* >::const_iterator it = table.begin(); it != table.end(); ++it ) {
			os << it->second << std::endl;
		} // for
	}

	struct Indexer::Impl {
		Impl( unsigned long _scope ) : refCount(1), scope( _scope ), size( 0 ), base(),
				idTable(), typeTable(), structTable(), enumTable(), unionTable(), traitTable() {}
		Impl( unsigned long _scope, Indexer &&_base ) : refCount(1), scope( _scope ), size( 0 ), base( _base ),
				idTable(), typeTable(), structTable(), enumTable(), unionTable(), traitTable() {}
		unsigned long refCount;   ///< Number of references to these tables
		unsigned long scope;      ///< Scope these tables are associated with
		unsigned long size;       ///< Number of elements stored in this table
		const Indexer base;       ///< Base indexer this extends

		IdTable idTable;          ///< Identifier namespace
		TypeTable typeTable;      ///< Type namespace
		StructTable structTable;  ///< Struct namespace
		EnumTable enumTable;      ///< Enum namespace
		UnionTable unionTable;    ///< Union namespace
		TraitTable traitTable;    ///< Trait namespace
	};

	Indexer::Impl *Indexer::newRef( Indexer::Impl *toClone ) {
		if ( ! toClone ) return 0;

		// shorten the search chain by skipping empty links
		Indexer::Impl *ret = toClone->size == 0 ? toClone->base.tables : toClone;
		if ( ret ) { ++ret->refCount; }

		return ret;
	}

	void Indexer::deleteRef( Indexer::Impl *toFree ) {
		if ( ! toFree ) return;

		if ( --toFree->refCount == 0 ) delete toFree;
	}

	void Indexer::removeSpecialOverrides( const std::string &id, std::vector< IdData > & out ) const {
		// only need to perform this step for constructors, destructors, and assignment functions
		if ( ! CodeGen::isCtorDtorAssign( id ) ) return;

		// helpful data structure to organize properties for a type
		struct ValueType {
			struct DeclBall { // properties for this particular decl
				IdData decl;
				bool isUserDefinedFunc;
				bool isCopyFunc;
			};
			// properties for this type
			bool existsUserDefinedCopyFunc = false;    // user-defined copy ctor found
			BaseSyntaxNode * deleteStmt = nullptr;     // non-null if a user-defined function is found
			std::vector< DeclBall > decls;

			// another FunctionDecl for the current type was found - determine
			// if it has special properties and update data structure accordingly
			ValueType & operator+=( IdData data ) {
				DeclarationWithType * function = data.id;
				bool isUserDefinedFunc = ! LinkageSpec::isOverridable( function->linkage );
				bool isCopyFunc = InitTweak::isCopyFunction( function, function->name );
				decls.push_back( DeclBall{ data, isUserDefinedFunc, isCopyFunc } );
				existsUserDefinedCopyFunc = existsUserDefinedCopyFunc || (isUserDefinedFunc && isCopyFunc);
				if ( isUserDefinedFunc && ! deleteStmt ) {
					// any user-defined function can act as an implicit delete statement for generated constructors.
					// a delete stmt should not act as an implicit delete statement.
					deleteStmt = data.id;
				}
				return *this;
			}
		}; // ValueType

		std::vector< IdData > copy;
		swap( copy, out );

		// organize discovered declarations by type
		std::unordered_map< std::string, ValueType > funcMap;
		for ( auto decl : copy ) {
			if ( FunctionDecl * function = dynamic_cast< FunctionDecl * >( decl.id ) ) {
				auto & params = function->type->parameters;
				assert( ! params.empty() );
				// use base type of pointer, so that qualifiers on the pointer type aren't considered.
				Type * base = InitTweak::getPointerBase( params.front()->get_type() );
				assert( base );
				funcMap[ Mangler::mangle( base ) ] += decl;
			} else {
				out.push_back( decl );
			}
		}

		// if a type contains user defined ctor/dtor/assign, then special rules trigger, which determine
		// the set of ctor/dtor/assign that can be used  by the requester. In particular, if the user defines
		// a default ctor, then the generated default ctor is unavailable, likewise for copy ctor
		// and dtor. If the user defines any ctor/dtor, then no generated field ctors are available.
		// If the user defines any ctor then the generated default ctor is unavailable (intrinsic default
		// ctor must be overridden exactly). If the user defines anything that looks like a copy constructor,
		// then the generated copy constructor is unavailable, and likewise for the assignment operator.
		for ( std::pair< const std::string, ValueType > & pair : funcMap ) {
			ValueType & val = pair.second;
			for ( ValueType::DeclBall ball : val.decls ) {
				bool isNotUserDefinedFunc = ! ball.isUserDefinedFunc && ball.decl.id->linkage != LinkageSpec::Intrinsic;
				bool isCopyFunc = ball.isCopyFunc;
				bool existsUserDefinedCopyFunc = val.existsUserDefinedCopyFunc;

				// only implicitly delete non-user defined functions that are not intrinsic, and are
				// not copy functions (assignment or copy constructor). If a  user-defined copy function exists,
				// do not pass along the non-user-defined copy functions since signatures do not have to match,
				// and the generated functions will often be cheaper.
				if ( isNotUserDefinedFunc ) {
					if ( isCopyFunc ) {
						// Skip over non-user-defined copy functions when there is a user-defined copy function.
						// Since their signatures do not have to be exact, deleting them is the wrong choice.
						if ( existsUserDefinedCopyFunc ) continue;
					} else {
						// delete non-user-defined non-copy functions if applicable.
						// deleteStmt will be non-null only if a user-defined function is found.
						ball.decl.deleteStmt = val.deleteStmt;
					}
				}
				out.push_back( ball.decl );
			}
		}
	}

	void Indexer::makeWritable() {
		if ( ! tables ) {
			// create indexer if not yet set
			tables = new Indexer::Impl( scope );
		} else if ( tables->refCount > 1 || tables->scope != scope ) {
			// make this indexer the base of a fresh indexer at the current scope
			tables = new Indexer::Impl( scope, std::move( *this ) );
		}
	}

	Indexer::Indexer() : tables( 0 ), scope( 0 ) {}

	Indexer::Indexer( const Indexer &that ) : doDebug( that.doDebug ), tables( newRef( that.tables ) ), scope( that.scope ) {}

	Indexer::Indexer( Indexer &&that ) : doDebug( that.doDebug ), tables( that.tables ), scope( that.scope ) {
		that.tables = 0;
	}

	Indexer::~Indexer() {
		deleteRef( tables );
	}

	Indexer& Indexer::operator= ( const Indexer &that ) {
		deleteRef( tables );

		tables = newRef( that.tables );
		scope = that.scope;
		doDebug = that.doDebug;

		return *this;
	}

	Indexer& Indexer::operator= ( Indexer &&that ) {
		deleteRef( tables );

		tables = that.tables;
		scope = that.scope;
		doDebug = that.doDebug;

		that.tables = 0;

		return *this;
	}

	void Indexer::lookupId( const std::string &id, std::vector< IdData > &out ) const {
		std::unordered_set< std::string > foundMangleNames;

		Indexer::Impl *searchTables = tables;
		while ( searchTables ) {

			IdTable::const_iterator decls = searchTables->idTable.find( id );
			if ( decls != searchTables->idTable.end() ) {
				const MangleTable &mangleTable = decls->second;
				for ( MangleTable::const_iterator decl = mangleTable.begin(); decl != mangleTable.end(); ++decl ) {
					// mark the mangled name as found, skipping this insertion if a declaration for that name has already been found
					if ( foundMangleNames.insert( decl->first ).second == false ) continue;

					out.push_back( decl->second );
				}
			}

			// get declarations from base indexers
			searchTables = searchTables->base.tables;
		}

		// some special functions, e.g. constructors and destructors
		// remove autogenerated functions when they are defined so that
		// they can never be matched
		removeSpecialOverrides( id, out );
	}

	NamedTypeDecl *Indexer::lookupType( const std::string &id ) const {
		if ( ! tables ) return 0;

		TypeTable::const_iterator ret = tables->typeTable.find( id );
		return ret != tables->typeTable.end() ? ret->second : tables->base.lookupType( id );
	}

	StructDecl *Indexer::lookupStruct( const std::string &id ) const {
		if ( ! tables ) return 0;

		StructTable::const_iterator ret = tables->structTable.find( id );
		return ret != tables->structTable.end() ? ret->second : tables->base.lookupStruct( id );
	}

	NamedTypeDecl *Indexer::globalLookupType( const std::string &id ) const {
		return lookupTypeAtScope( id, 0 );
	}

	StructDecl *Indexer::globalLookupStruct( const std::string &id ) const {
		return lookupStructAtScope( id, 0 );
	}

	UnionDecl *Indexer::globalLookupUnion( const std::string &id ) const {
		return lookupUnionAtScope( id, 0 );
	}

	EnumDecl *Indexer::globalLookupEnum( const std::string &id ) const {
		return lookupEnumAtScope( id, 0 );
	}

	EnumDecl *Indexer::lookupEnum( const std::string &id ) const {
		if ( ! tables ) return 0;

		EnumTable::const_iterator ret = tables->enumTable.find( id );
		return ret != tables->enumTable.end() ? ret->second : tables->base.lookupEnum( id );
	}

	UnionDecl *Indexer::lookupUnion( const std::string &id ) const {
		if ( ! tables ) return 0;

		UnionTable::const_iterator ret = tables->unionTable.find( id );
		return ret != tables->unionTable.end() ? ret->second : tables->base.lookupUnion( id );
	}

	TraitDecl *Indexer::lookupTrait( const std::string &id ) const {
		if ( ! tables ) return 0;

		TraitTable::const_iterator ret = tables->traitTable.find( id );
		return ret != tables->traitTable.end() ? ret->second : tables->base.lookupTrait( id );
	}

	const Indexer::IdData * Indexer::lookupIdAtScope( const std::string &id, const std::string &mangleName, unsigned long scope ) const {
		if ( ! tables ) return nullptr;
		if ( tables->scope < scope ) return nullptr;

		IdTable::const_iterator decls = tables->idTable.find( id );
		if ( decls != tables->idTable.end() ) {
			const MangleTable &mangleTable = decls->second;
			MangleTable::const_iterator decl = mangleTable.find( mangleName );
			if ( decl != mangleTable.end() ) return &decl->second;
		}

		return tables->base.lookupIdAtScope( id, mangleName, scope );
	}

	Indexer::IdData * Indexer::lookupIdAtScope( const std::string &id, const std::string &mangleName, unsigned long scope ) {
		return const_cast<IdData *>(const_cast<const Indexer *>(this)->lookupIdAtScope( id, mangleName, scope ));
	}

	bool Indexer::hasIncompatibleCDecl( const std::string &id, const std::string &mangleName, unsigned long scope ) const {
		if ( ! tables ) return false;
		if ( tables->scope < scope ) return false;

		IdTable::const_iterator decls = tables->idTable.find( id );
		if ( decls != tables->idTable.end() ) {
			const MangleTable &mangleTable = decls->second;
			for ( MangleTable::const_iterator decl = mangleTable.begin(); decl != mangleTable.end(); ++decl ) {
				// check for C decls with the same name, skipping those with a compatible type (by mangleName)
				if ( ! LinkageSpec::isMangled( decl->second.id->get_linkage() ) && decl->first != mangleName ) return true;
			}
		}

		return tables->base.hasIncompatibleCDecl( id, mangleName, scope );
	}

	bool Indexer::hasCompatibleCDecl( const std::string &id, const std::string &mangleName, unsigned long scope ) const {
		if ( ! tables ) return false;
		if ( tables->scope < scope ) return false;

		IdTable::const_iterator decls = tables->idTable.find( id );
		if ( decls != tables->idTable.end() ) {
			const MangleTable &mangleTable = decls->second;
			for ( MangleTable::const_iterator decl = mangleTable.begin(); decl != mangleTable.end(); ++decl ) {
				// check for C decls with the same name, skipping
				// those with an incompatible type (by mangleName)
				if ( ! LinkageSpec::isMangled( decl->second.id->get_linkage() ) && decl->first == mangleName ) return true;
			}
		}

		return tables->base.hasCompatibleCDecl( id, mangleName, scope );
	}

	NamedTypeDecl *Indexer::lookupTypeAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;
		if ( tables->scope > scope ) return tables->base.lookupTypeAtScope( id, scope );

		TypeTable::const_iterator ret = tables->typeTable.find( id );
		return ret != tables->typeTable.end() ? ret->second : tables->base.lookupTypeAtScope( id, scope );
	}

	StructDecl *Indexer::lookupStructAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;
		if ( tables->scope > scope ) return tables->base.lookupStructAtScope( id, scope );

		StructTable::const_iterator ret = tables->structTable.find( id );
		return ret != tables->structTable.end() ? ret->second : tables->base.lookupStructAtScope( id, scope );
	}

	EnumDecl *Indexer::lookupEnumAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;
		if ( tables->scope > scope ) return tables->base.lookupEnumAtScope( id, scope );

		EnumTable::const_iterator ret = tables->enumTable.find( id );
		return ret != tables->enumTable.end() ? ret->second : tables->base.lookupEnumAtScope( id, scope );
	}

	UnionDecl *Indexer::lookupUnionAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;
		if ( tables->scope > scope ) return tables->base.lookupUnionAtScope( id, scope );

		UnionTable::const_iterator ret = tables->unionTable.find( id );
		return ret != tables->unionTable.end() ? ret->second : tables->base.lookupUnionAtScope( id, scope );
	}

	TraitDecl *Indexer::lookupTraitAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;
		if ( tables->scope > scope ) return tables->base.lookupTraitAtScope( id, scope );

		TraitTable::const_iterator ret = tables->traitTable.find( id );
		return ret != tables->traitTable.end() ? ret->second : tables->base.lookupTraitAtScope( id, scope );
	}

	bool isFunction( DeclarationWithType * decl ) {
		return GenPoly::getFunctionType( decl->get_type() );
	}

	bool isObject( DeclarationWithType * decl ) {
		return ! isFunction( decl );
	}

	bool isDefinition( DeclarationWithType * decl ) {
		if ( FunctionDecl * func = dynamic_cast< FunctionDecl * >( decl ) ) {
			// a function is a definition if it has a body
			return func->statements;
		} else {
			// an object is a definition if it is not marked extern.
			// both objects must be marked extern
			return ! decl->get_storageClasses().is_extern;
		}
	}

	bool addedIdConflicts( Indexer::IdData & existing, DeclarationWithType *added, BaseSyntaxNode * deleteStmt, Indexer::ConflictFunction handleConflicts ) {
		// if we're giving the same name mangling to things of different types then there is something wrong
		assert( (isObject( added ) && isObject( existing.id ) )
			|| ( isFunction( added ) && isFunction( existing.id ) ) );

		if ( LinkageSpec::isOverridable( existing.id->get_linkage() ) ) {
			// new definition shadows the autogenerated one, even at the same scope
			return false;
		} else if ( LinkageSpec::isMangled( added->get_linkage() ) || ResolvExpr::typesCompatible( added->get_type(), existing.id->get_type(), Indexer() ) ) {

			// it is a conflict if one declaration is deleted and the other is not
			if ( deleteStmt && ! existing.deleteStmt ) {
				return handleConflicts( existing, "deletion of defined identifier " );
			} else if ( ! deleteStmt && existing.deleteStmt ) {
				return handleConflicts( existing, "definition of deleted identifier " );
			}

			if ( isDefinition( added ) && isDefinition( existing.id ) ) {
				if ( isFunction( added ) ) {
					return handleConflicts( existing, "duplicate function definition for " );
				} else {
					return handleConflicts( existing, "duplicate object definition for " );
				} // if
			} // if
		} else {
			return handleConflicts( existing, "duplicate definition for " );
		} // if

		return true;
	}

	void Indexer::addId( DeclarationWithType *decl, ConflictFunction handleConflicts, Expression * baseExpr, BaseSyntaxNode * deleteStmt ) {
		if ( decl->name == "" ) return;
		debugPrint( "Adding Id " << decl->name << std::endl );
		makeWritable();

		const std::string &name = decl->name;
		std::string mangleName;
		if ( LinkageSpec::isOverridable( decl->linkage ) ) {
			// mangle the name without including the appropriate suffix, so overridable routines are placed into the
			// same "bucket" as their user defined versions.
			mangleName = Mangler::mangle( decl, false );
		} else {
			mangleName = Mangler::mangle( decl );
		} // if

		// this ensures that no two declarations with the same unmangled name at the same scope both have C linkage
		if ( ! LinkageSpec::isMangled( decl->linkage ) ) {
			// NOTE this is broken in Richard's original code in such a way that it never triggers (it
			// doesn't check decls that have the same manglename, and all C-linkage decls are defined to
			// have their name as their manglename, hence the error can never trigger).
			// The code here is closer to correct, but name mangling would have to be completely
			// isomorphic to C type-compatibility, which it may not be.
			if ( hasIncompatibleCDecl( name, mangleName, scope ) ) {
				SemanticError( decl, "conflicting overload of C function " );
			}
		} else {
			// Check that a Cforall declaration doesn't override any C declaration
			if ( hasCompatibleCDecl( name, mangleName, scope ) ) {
				SemanticError( decl, "Cforall declaration hides C function " );
			}
		}

		// Skip repeat declarations of the same identifier
		IdData * existing = lookupIdAtScope( name, mangleName, scope );
		if ( existing && existing->id && addedIdConflicts( *existing, decl, deleteStmt, handleConflicts ) ) return;

		// add to indexer
		tables->idTable[ name ][ mangleName ] = IdData{ decl, baseExpr, deleteStmt };
		++tables->size;
	}

	void Indexer::addId( DeclarationWithType * decl, Expression * baseExpr ) {
		// default handling of conflicts is to raise an error
		addId( decl, [decl](IdData &, const std::string & msg) { SemanticError( decl, msg ); return true; }, baseExpr, decl->isDeleted ? decl : nullptr );
	}

	void Indexer::addDeletedId( DeclarationWithType * decl, BaseSyntaxNode * deleteStmt ) {
		// default handling of conflicts is to raise an error
		addId( decl, [decl](IdData &, const std::string & msg) { SemanticError( decl, msg ); return true; }, nullptr, deleteStmt );
	}

	bool addedTypeConflicts( NamedTypeDecl *existing, NamedTypeDecl *added ) {
		if ( existing->base == nullptr ) {
			return false;
		} else if ( added->base == nullptr ) {
			return true;
		} else {
			assert( existing->base && added->base );
			// typedef redeclarations are errors only if types are different
			if ( ! ResolvExpr::typesCompatible( existing->base, added->base, Indexer() ) ) {
				SemanticError( added->location, "redeclaration of " + added->name );
			}
		}
		// does not need to be added to the table if both existing and added have a base that are the same
		return true;
	}

	void Indexer::addType( NamedTypeDecl *decl ) {
		debugPrint( "Adding type " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->name;
		TypeTable::iterator existing = tables->typeTable.find( id );
		if ( existing == tables->typeTable.end() ) {
			NamedTypeDecl *parent = tables->base.lookupTypeAtScope( id, scope );
			if ( ! parent || ! addedTypeConflicts( parent, decl ) ) {
				tables->typeTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedTypeConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	bool addedDeclConflicts( AggregateDecl *existing, AggregateDecl *added ) {
		if ( ! existing->body ) {
			return false;
		} else if ( added->body ) {
			SemanticError( added, "redeclaration of " );
		} // if
		return true;
	}

	void Indexer::addStruct( const std::string &id ) {
		debugPrint( "Adding fwd decl for struct " << id << std::endl );
		addStruct( new StructDecl( id ) );
	}

	void Indexer::addStruct( StructDecl *decl ) {
		debugPrint( "Adding struct " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->name;
		StructTable::iterator existing = tables->structTable.find( id );
		if ( existing == tables->structTable.end() ) {
			StructDecl *parent = tables->base.lookupStructAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->structTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addEnum( EnumDecl *decl ) {
		debugPrint( "Adding enum " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->name;
		EnumTable::iterator existing = tables->enumTable.find( id );
		if ( existing == tables->enumTable.end() ) {
			EnumDecl *parent = tables->base.lookupEnumAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->enumTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addUnion( const std::string &id ) {
		debugPrint( "Adding fwd decl for union " << id << std::endl );
		addUnion( new UnionDecl( id ) );
	}

	void Indexer::addUnion( UnionDecl *decl ) {
		debugPrint( "Adding union " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->name;
		UnionTable::iterator existing = tables->unionTable.find( id );
		if ( existing == tables->unionTable.end() ) {
			UnionDecl *parent = tables->base.lookupUnionAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->unionTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addTrait( TraitDecl *decl ) {
		debugPrint( "Adding trait " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->name;
		TraitTable::iterator existing = tables->traitTable.find( id );
		if ( existing == tables->traitTable.end() ) {
			TraitDecl *parent = tables->base.lookupTraitAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->traitTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addMembers( AggregateDecl * aggr, Expression * expr, ConflictFunction handleConflicts ) {
		for ( Declaration * decl : aggr->members ) {
			if ( DeclarationWithType * dwt = dynamic_cast< DeclarationWithType * >( decl ) ) {
				addId( dwt, handleConflicts, expr );
				if ( dwt->name == "" ) {
					Type * t = dwt->get_type()->stripReferences();
					if ( dynamic_cast< StructInstType * >( t ) || dynamic_cast< UnionInstType * >( t ) ) {
						Expression * base = expr->clone();
						ResolvExpr::Cost cost = ResolvExpr::Cost::zero; // xxx - carry this cost into the indexer as a base cost?
						ResolvExpr::referenceToRvalueConversion( base, cost );
						addMembers( t->getAggr(), new MemberExpr( dwt, base ), handleConflicts );
					}
				}
			}
		}
	}

	void Indexer::addWith( std::list< Expression * > & withExprs, BaseSyntaxNode * withStmt ) {
		for ( Expression * expr : withExprs ) {
			if ( expr->result ) {
				AggregateDecl * aggr = expr->result->stripReferences()->getAggr();
				assertf( aggr, "WithStmt expr has non-aggregate type: %s", toString( expr->result ).c_str() );

				addMembers( aggr, expr, [withStmt](IdData & existing, const std::string &) {
					// on conflict, delete the identifier
					existing.deleteStmt = withStmt;
					return true;
				});
			}
		}
	}

	void Indexer::addIds( const std::vector< DeclarationWithType * > & decls ) {
		for ( auto d : decls ) {
			addId( d );
		}
	}

	void Indexer::addTypes( const Type::ForallList & tds ) {
		for ( auto td : tds ) {
			addType( td );
			addIds( td->assertions );
		}
	}

	void Indexer::addFunctionType( FunctionType * ftype ) {
		addTypes( ftype->forall );
		addIds( ftype->returnVals );
		addIds( ftype->parameters );
	}

	void Indexer::enterScope() {
		++scope;

		if ( doDebug ) {
			std::cerr << "--- Entering scope " << scope << std::endl;
		}
	}

	void Indexer::leaveScope() {
		using std::cerr;

		assert( scope > 0 && "cannot leave initial scope" );
		if ( doDebug ) {
			cerr << "--- Leaving scope " << scope << " containing" << std::endl;
		}
		--scope;

		while ( tables && tables->scope > scope ) {
			if ( doDebug ) {
				dump( tables->idTable, cerr );
				dump( tables->typeTable, cerr );
				dump( tables->structTable, cerr );
				dump( tables->enumTable, cerr );
				dump( tables->unionTable, cerr );
				dump( tables->traitTable, cerr );
			}

			// swap tables for base table until we find one at an appropriate scope
			Indexer::Impl *base = newRef( tables->base.tables );
			deleteRef( tables );
			tables = base;
		}
	}

	void Indexer::print( std::ostream &os, int indent ) const {
		using std::cerr;

		if ( tables ) {
			os << "--- scope " << tables->scope << " ---" << std::endl;

			os << "===idTable===" << std::endl;
			dump( tables->idTable, os );
			os << "===typeTable===" << std::endl;
			dump( tables->typeTable, os );
			os << "===structTable===" << std::endl;
			dump( tables->structTable, os );
			os << "===enumTable===" << std::endl;
			dump( tables->enumTable, os );
			os << "===unionTable===" << std::endl;
			dump( tables->unionTable, os );
			os << "===contextTable===" << std::endl;
			dump( tables->traitTable, os );

			tables->base.print( os, indent );
		} else {
			os << "--- end ---" << std::endl;
		}

	}

	Expression * Indexer::IdData::combine( ResolvExpr::Cost & cost ) const {
		Expression * ret = nullptr;
		if ( baseExpr ) {
			Expression * base = baseExpr->clone();
			ResolvExpr::referenceToRvalueConversion( base, cost );
			ret = new MemberExpr( id, base );
			// xxx - this introduces hidden environments, for now remove them.
			// std::swap( base->env, ret->env );
			delete base->env;
			base->env = nullptr;
		} else {
			ret = new VariableExpr( id );
		}
		if ( deleteStmt ) ret = new DeletedExpr( ret, deleteStmt );
		return ret;
	}
} // namespace SymTab

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
