//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// typeops.h --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 07:28:22 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Jul 22 09:36:18 2017
// Update Count     : 3
//

#pragma once

#include <vector>

#include "SynTree/SynTree.h"
#include "SynTree/Type.h"
#include "SymTab/Indexer.h"
#include "Cost.h"
#include "TypeEnvironment.h"

namespace ResolvExpr {
	// combos: takes a list of sets and returns a set of lists representing every possible way of forming a list by
	// picking one element out of each set
	template< typename InputIterator, typename OutputIterator >
	void combos( InputIterator begin, InputIterator end, OutputIterator out ) {
		typedef typename InputIterator::value_type SetType;
		typedef typename std::vector< typename SetType::value_type > ListType;

		if ( begin == end )	{
			*out++ = ListType();
			return;
		} // if

		InputIterator current = begin;
		begin++;

		std::vector< ListType > recursiveResult;
		combos( begin, end, back_inserter( recursiveResult ) );

		for ( const auto& i : recursiveResult ) for ( const auto& j : *current ) {
			ListType result;
			std::back_insert_iterator< ListType > inserter = back_inserter( result );
			*inserter++ = j;
			std::copy( i.begin(), i.end(), inserter );
			*out++ = result;
		}
	}

	// in AdjustExprType.cc
	/// Replaces array types with the equivalent pointer, and function types with a pointer-to-function
	void adjustExprType( Type *&type, const TypeEnvironment &env, const SymTab::Indexer &indexer );

	/// Replaces array types with the equivalent pointer, and function types with a pointer-to-function using empty TypeEnvironment and Indexer
	void adjustExprType( Type *& type );

	template< typename ForwardIterator >
	void adjustExprTypeList( ForwardIterator begin, ForwardIterator end, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
		while ( begin != end ) {
			adjustExprType( *begin++, env, indexer );
		} // while
	}

	// in CastCost.cc
	Cost castCost( Type *src, Type *dest, const SymTab::Indexer &indexer, const TypeEnvironment &env );

	// in ConversionCost.cc
	Cost conversionCost( Type *src, Type *dest, const SymTab::Indexer &indexer, const TypeEnvironment &env );

	// in AlternativeFinder.cc
	Cost computeConversionCost( Type *actualType, Type *formalType,
		const SymTab::Indexer &indexer, const TypeEnvironment &env );

	// in PtrsAssignable.cc
	int ptrsAssignable( Type *src, Type *dest, const TypeEnvironment &env );

	// in PtrsCastable.cc
	int ptrsCastable( Type *src, Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer );

	// in Unify.cc
	bool isFtype( Type *type );
	bool typesCompatible( Type *, Type *, const SymTab::Indexer &indexer, const TypeEnvironment &env );
	bool typesCompatibleIgnoreQualifiers( Type *, Type *, const SymTab::Indexer &indexer, const TypeEnvironment &env );

	inline bool typesCompatible( Type *t1, Type *t2, const SymTab::Indexer &indexer ) {
		TypeEnvironment env;
		return typesCompatible( t1, t2, indexer, env );
	}

	inline bool typesCompatibleIgnoreQualifiers( Type *t1, Type *t2, const SymTab::Indexer &indexer ) {
		TypeEnvironment env;
		return typesCompatibleIgnoreQualifiers( t1, t2, indexer, env );
	}

	/// creates the type represented by the list of returnVals in a FunctionType. The caller owns the return value.
	Type * extractResultType( FunctionType * functionType );

	// in CommonType.cc
	Type *commonType( Type *type1, Type *type2, bool widenFirst, bool widenSecond, const SymTab::Indexer &indexer, TypeEnvironment &env, const OpenVarSet &openVars );

	// in PolyCost.cc
	int polyCost( Type *type, const TypeEnvironment &env, const SymTab::Indexer &indexer );

	// in SpecCost.cc
	int specCost( Type *type );

	// in Occurs.cc
	bool occurs( Type *type, std::string varName, const TypeEnvironment &env );

	template<typename Iter>
	bool occursIn( Type* ty, Iter begin, Iter end, const TypeEnvironment &env ) {
		while ( begin != end ) {
			if ( occurs( ty, *begin, env ) ) return true;
			++begin;
		}
		return false;
	}

	// in AlternativeFinder.cc
	void referenceToRvalueConversion( Expression *& expr, Cost & cost );

	// flatten tuple type into list of types
	template< typename OutputIterator >
	void flatten( Type * type, OutputIterator out ) {
		if ( TupleType * tupleType = dynamic_cast< TupleType * >( type ) ) {
			for ( Type * t : tupleType->types ) {
				flatten( t, out );
			}
		} else {
			*out++ = type->clone();
		}
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
