//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// FindOpenVars.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 09:42:48 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun May 17 09:45:25 2015
// Update Count     : 3
//

#include "FindOpenVars.h"

#include <list>                   // for _List_const_iterator, list<>::const...
#include <map>                    // for map<>::mapped_type

#include "Common/PassVisitor.h"
#include "SynTree/Declaration.h"  // for TypeDecl, DeclarationWithType (ptr ...
#include "SynTree/Type.h"         // for Type, Type::ForallList, ArrayType

namespace ResolvExpr {
	struct FindOpenVars : public WithGuards {
		FindOpenVars( OpenVarSet &openVars, OpenVarSet &closedVars, AssertionSet &needAssertions, AssertionSet &haveAssertions, bool firstIsOpen );

		void previsit( PointerType * pointerType );
		void previsit( ArrayType * arrayType );
		void previsit( FunctionType * functionType );
		void previsit( TupleType * tupleType );

		void common_action( Type *type );

		OpenVarSet &openVars, &closedVars;
		AssertionSet &needAssertions, &haveAssertions;
		bool nextIsOpen;
	};

	void findOpenVars( Type *type, OpenVarSet &openVars, OpenVarSet &closedVars, AssertionSet &needAssertions, AssertionSet &haveAssertions, bool firstIsOpen ) {
		PassVisitor<FindOpenVars> finder( openVars, closedVars, needAssertions, haveAssertions, firstIsOpen );
		type->accept( finder );
	}

	FindOpenVars::FindOpenVars( OpenVarSet &openVars, OpenVarSet &closedVars, AssertionSet &needAssertions, AssertionSet &haveAssertions, bool firstIsOpen )
		: openVars( openVars ), closedVars( closedVars ), needAssertions( needAssertions ), haveAssertions( haveAssertions ), nextIsOpen( firstIsOpen )	{
	}

	void FindOpenVars::common_action( Type *type ) {
		if ( nextIsOpen ) {
			for ( const auto i : type->get_forall() ) {
				openVars[ i->get_name() ] = TypeDecl::Data{ i };
				for ( const auto assert : i->assertions ) {
					needAssertions[ assert ].isUsed = false;
				}
			}
		} else {
			for ( const auto i : type->get_forall() ) {
				closedVars[ i->get_name() ] = TypeDecl::Data{ i };
				for ( const auto assert : i->assertions ) {
					haveAssertions[ assert ].isUsed = false;
				}
			} // for
		} // if
	}

	void FindOpenVars::previsit(PointerType *pointerType) {
		common_action( pointerType );
	}

	void FindOpenVars::previsit(ArrayType *arrayType) {
		common_action( arrayType );
	}

	void FindOpenVars::previsit(FunctionType *functionType) {
		common_action( functionType );
		nextIsOpen = ! nextIsOpen;
		GuardAction( [this](){ nextIsOpen = ! nextIsOpen; } );
	}

	void FindOpenVars::previsit(TupleType *tupleType) {
		common_action( tupleType );
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
