//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// FunctionDecl.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Oct  1 23:06:32 2016
// Update Count     : 21
//

#include <cassert>

#include "Declaration.h"
#include "Statement.h"
#include "Type.h"
#include "Attribute.h"
#include "Common/utility.h"
#include "InitTweak/InitTweak.h"

FunctionDecl::FunctionDecl( const std::string &name, DeclarationNode::StorageClass sc, LinkageSpec::Spec linkage, FunctionType *type, CompoundStmt *statements, bool isInline, bool isNoreturn, std::list< Attribute * > attributes )
		: Parent( name, sc, linkage, attributes ), type( type ), statements( statements ) {
	set_isInline( isInline );
	set_isNoreturn( isNoreturn );
	// this is a brazen hack to force the function "main" to have C linkage
	if ( name == "main" ) {
		set_linkage( LinkageSpec::C );
	} // if
}

FunctionDecl::FunctionDecl( const FunctionDecl &other )
	: Parent( other ), type( maybeClone( other.type ) ), statements( maybeClone( other.statements ) ) {
}

FunctionDecl::~FunctionDecl() {
	delete type;
	delete statements;
	deleteAll( oldDecls );
}

Type * FunctionDecl::get_type() const {
	return type;
}

void FunctionDecl::set_type( Type *t ) {
	type = dynamic_cast< FunctionType* >( t );
	assert( type );
}

void FunctionDecl::print( std::ostream &os, int indent ) const {
	using std::endl;
	using std::string;

	if ( get_name() != "" ) {
		os << get_name() << ": ";
	} // if
	if ( get_linkage() != LinkageSpec::Cforall ) {
		os << LinkageSpec::linkageName( get_linkage() ) << " ";
	} // if
	if ( get_isInline() ) {
		os << "inline ";
	} // if
	if ( get_isNoreturn() ) {
		os << "_Noreturn ";
	} // if

	printAll( get_attributes(), os, indent );

	if ( get_storageClass() != DeclarationNode::NoStorageClass ) {
		os << DeclarationNode::storageName[ get_storageClass() ] << ' ';
	} // if
	if ( get_type() ) {
		get_type()->print( os, indent );
	} else {
		os << "untyped entity ";
	} // if

	if ( ! oldIdents.empty() ) {
		os << string( indent + 2, ' ' ) << "with parameter names" << endl;
		for ( std::list< std::string >::const_iterator i = oldIdents.begin(); i != oldIdents.end(); ++i ) {
			os << string( indent + 4, ' ' ) << *i << endl;
		} // for
	} // if

	if ( ! oldDecls.empty() ) {
		os << string( indent + 2, ' ' ) << "with parameter declarations" << endl;
		printAll( oldDecls, os, indent + 4 );
	} // if
	if ( statements ) {
		os << string( indent + 2, ' ' ) << "with body " << endl;
		os << string( indent + 4, ' ' );
		statements->print( os, indent + 4 );
	} // if
}

void FunctionDecl::printShort( std::ostream &os, int indent ) const {
	using std::endl;
	using std::string;

	if ( get_name() != "" ) {
		os << get_name() << ": ";
	} // if
	if ( get_isInline() ) {
		os << "inline ";
	} // if
	if ( get_isNoreturn() ) {
		os << "_Noreturn ";
	} // if

	// xxx - should printShort print attributes?

	if ( get_storageClass() != DeclarationNode::NoStorageClass ) {
		os << DeclarationNode::storageName[ get_storageClass() ] << ' ';
	} // if
	if ( get_type() ) {
		get_type()->print( os, indent );
	} else {
		os << "untyped entity ";
	} // if
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
