//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Stmt.hpp --
//
// Author           : Aaron B. Moss
// Created On       : Wed May 8 13:00:00 2019
// Last Modified By : Aaron B. Moss
// Last Modified On : Wed May 8 13:00:00 2019
// Update Count     : 1
//

#pragma once

#include <list>
#include <utility>                // for move
#include <vector>

#include "Label.hpp"
#include "Node.hpp"               // for node, ptr
#include "ParseNode.hpp"
#include "Visitor.hpp"
#include "Common/CodeLocation.h"

namespace ast {

class Expr;

/// Base statement node
class Stmt : public ParseNode {
public:
	std::vector<Label> labels;

	Stmt( const CodeLocation& loc, std::vector<Label>&& labels = {} )
	: ParseNode(loc), labels(std::move(labels)) {}

	Stmt(const Stmt& o) : ParseNode(o), labels(o.labels) {}

	virtual Stmt* accept( Visitor& v ) override = 0;
private:
	virtual Stmt* clone() const override = 0;
};

/// Compound statement `{ ... }`
class CompoundStmt final : public Stmt {
public:
	std::list<ptr<Stmt>> kids;

	CompoundStmt(const CodeLocation& loc, std::list<ptr<Stmt>>&& ks = {} )
	: Stmt(loc), kids(std::move(ks)) {}

	CompoundStmt( const CompoundStmt& o );
	CompoundStmt( CompoundStmt&& o ) = default;

	void push_back( Stmt* s ) { kids.emplace_back( s ); }
	void push_front( Stmt* s ) { kids.emplace_front( s ); }

	CompoundStmt* accept( Visitor& v ) override { return v.visit( this ); }
private:
	CompoundStmt* clone() const override { return new CompoundStmt{ *this }; }
};

/// Empty statment `;`
class NullStmt final : public Stmt {
public:
	NullStmt( const CodeLocation& loc, std::vector<Label>&& labels = {} )
	: Stmt(loc, std::move(labels)) {}

	NullStmt* accept( Visitor& v ) override { return v.visit( this ); }
private:
	NullStmt* clone() const override { return new NullStmt{ *this }; }
};

/// Expression wrapped by statement
class ExprStmt final : public Stmt {
public:
	ptr<Expr> expr;

	ExprStmt( const CodeLocation& loc, Expr* e ) : Stmt(loc), expr(e) {}

	Stmt* accept( Visitor& v ) override { return v.visit( this ); }
private:
	ExprStmt* clone() const override { return new ExprStmt{ *this }; }
};



}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //