//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Visitor.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Andrew Beach
// Last Modified On : Thu Jun 22 13:41:00 2017
// Update Count     : 26
//

#include <cassert>
#include "Visitor.h"
#include "Initializer.h"
#include "Statement.h"
#include "Type.h"
#include "Declaration.h"
#include "Expression.h"
#include "Constant.h"

Visitor::Visitor() {}

Visitor::~Visitor() {}

void Visitor::visit( ObjectDecl *objectDecl ) {
	maybeAccept( objectDecl->get_type(), *this );
	maybeAccept( objectDecl->get_init(), *this );
	maybeAccept( objectDecl->get_bitfieldWidth(), *this );
}

void Visitor::visit( FunctionDecl *functionDecl ) {
	maybeAccept( functionDecl->get_functionType(), *this );
	maybeAccept( functionDecl->get_statements(), *this );
}

void Visitor::handleAggregateDecl( AggregateDecl *aggregateDecl ) {
	acceptAll( aggregateDecl->get_parameters(), *this );
	acceptAll( aggregateDecl->get_members(), *this );
}

void Visitor::visit( StructDecl *aggregateDecl ) {
	handleAggregateDecl( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( UnionDecl *aggregateDecl ) {
	handleAggregateDecl( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( EnumDecl *aggregateDecl ) {
	handleAggregateDecl( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::visit( TraitDecl *aggregateDecl ) {
	handleAggregateDecl( static_cast< AggregateDecl* >( aggregateDecl ) );
}

void Visitor::handleNamedTypeDecl( NamedTypeDecl *typeDecl ) {
	acceptAll( typeDecl->get_parameters(), *this );
	acceptAll( typeDecl->get_assertions(), *this );
	maybeAccept( typeDecl->get_base(), *this );
}

void Visitor::visit( TypeDecl *typeDecl ) {
	handleNamedTypeDecl( static_cast< NamedTypeDecl* >( typeDecl ) );
	maybeAccept( typeDecl->get_init(), *this );
}

void Visitor::visit( TypedefDecl *typeDecl ) {
	handleNamedTypeDecl( static_cast< NamedTypeDecl* >( typeDecl ) );
}

void Visitor::visit( AsmDecl *asmDecl ) {
	maybeAccept( asmDecl->get_stmt(), *this );
}


void Visitor::visit( CompoundStmt *compoundStmt ) {
	acceptAll( compoundStmt->get_kids(), *this );
}

void Visitor::visit( ExprStmt *exprStmt ) {
	maybeAccept( exprStmt->get_expr(), *this );
}

void Visitor::visit( AsmStmt *asmStmt ) {
	maybeAccept( asmStmt->get_instruction(), *this );
	acceptAll( asmStmt->get_output(), *this );
	acceptAll( asmStmt->get_input(), *this );
	acceptAll( asmStmt->get_clobber(), *this );
}

void Visitor::visit( IfStmt *ifStmt ) {
	maybeAccept( ifStmt->get_condition(), *this );
	maybeAccept( ifStmt->get_thenPart(), *this );
	maybeAccept( ifStmt->get_elsePart(), *this );
}

void Visitor::visit( WhileStmt *whileStmt ) {
	maybeAccept( whileStmt->get_condition(), *this );
	maybeAccept( whileStmt->get_body(), *this );
}

void Visitor::visit( ForStmt *forStmt ) {
	acceptAll( forStmt->get_initialization(), *this );
	maybeAccept( forStmt->get_condition(), *this );
	maybeAccept( forStmt->get_increment(), *this );
	maybeAccept( forStmt->get_body(), *this );
}

void Visitor::visit( SwitchStmt *switchStmt ) {
	maybeAccept( switchStmt->get_condition(), *this );
	acceptAll( switchStmt->get_statements(), *this );
}

void Visitor::visit( CaseStmt *caseStmt ) {
	maybeAccept( caseStmt->get_condition(), *this );
	acceptAll( caseStmt->get_statements(), *this );
}

void Visitor::visit( __attribute__((unused)) BranchStmt *branchStmt ) {
}

void Visitor::visit( ReturnStmt *returnStmt ) {
	maybeAccept( returnStmt->get_expr(), *this );
}

void Visitor::visit( ThrowStmt * throwStmt ) {
	maybeAccept( throwStmt->get_expr(), *this );
	maybeAccept( throwStmt->get_target(), *this );
}

void Visitor::visit( TryStmt *tryStmt ) {
	maybeAccept( tryStmt->get_block(), *this );
	acceptAll( tryStmt->get_catchers(), *this );
	maybeAccept( tryStmt->get_finally(), *this );
}

void Visitor::visit( CatchStmt *catchStmt ) {
	maybeAccept( catchStmt->get_decl(), *this );
	maybeAccept( catchStmt->get_cond(), *this );
	maybeAccept( catchStmt->get_body(), *this );
}

void Visitor::visit( FinallyStmt *finalStmt ) {
	maybeAccept( finalStmt->get_block(), *this );
}

void Visitor::visit( __attribute__((unused)) NullStmt *nullStmt ) {
}

void Visitor::visit( DeclStmt *declStmt ) {
	maybeAccept( declStmt->get_decl(), *this );
}

void Visitor::visit( ImplicitCtorDtorStmt *impCtorDtorStmt ) {
	maybeAccept( impCtorDtorStmt->get_callStmt(), *this );
}


void Visitor::visit( ApplicationExpr *applicationExpr ) {
	maybeAccept( applicationExpr->get_result(), *this );
	maybeAccept( applicationExpr->get_function(), *this );
	acceptAll( applicationExpr->get_args(), *this );
}

void Visitor::visit( UntypedExpr *untypedExpr ) {
	maybeAccept( untypedExpr->get_result(), *this );
	acceptAll( untypedExpr->get_args(), *this );
}

void Visitor::visit( NameExpr *nameExpr ) {
	maybeAccept( nameExpr->get_result(), *this );
}

void Visitor::visit( AddressExpr *addressExpr ) {
	maybeAccept( addressExpr->get_result(), *this );
	maybeAccept( addressExpr->get_arg(), *this );
}

void Visitor::visit( LabelAddressExpr *labAddressExpr ) {
	maybeAccept( labAddressExpr->get_result(), *this );
	maybeAccept( labAddressExpr->get_arg(), *this );
}

void Visitor::visit( CastExpr *castExpr ) {
	maybeAccept( castExpr->get_result(), *this );
	maybeAccept( castExpr->get_arg(), *this );
}

void Visitor::visit( UntypedMemberExpr *memberExpr ) {
	maybeAccept( memberExpr->get_result(), *this );
	maybeAccept( memberExpr->get_aggregate(), *this );
	maybeAccept( memberExpr->get_member(), *this );
}

void Visitor::visit( MemberExpr *memberExpr ) {
	maybeAccept( memberExpr->get_result(), *this );
	maybeAccept( memberExpr->get_aggregate(), *this );
}

void Visitor::visit( VariableExpr *variableExpr ) {
	maybeAccept( variableExpr->get_result(), *this );
}

void Visitor::visit( ConstantExpr *constantExpr ) {
	maybeAccept( constantExpr->get_result(), *this );
	maybeAccept( constantExpr->get_constant(), *this );
}

void Visitor::visit( SizeofExpr *sizeofExpr ) {
	maybeAccept( sizeofExpr->get_result(), *this );
	if ( sizeofExpr->get_isType() ) {
		maybeAccept( sizeofExpr->get_type(), *this );
	} else {
		maybeAccept( sizeofExpr->get_expr(), *this );
	}
}

void Visitor::visit( AlignofExpr *alignofExpr ) {
	maybeAccept( alignofExpr->get_result(), *this );
	if ( alignofExpr->get_isType() ) {
		maybeAccept( alignofExpr->get_type(), *this );
	} else {
		maybeAccept( alignofExpr->get_expr(), *this );
	}
}

void Visitor::visit( UntypedOffsetofExpr *offsetofExpr ) {
	maybeAccept( offsetofExpr->get_result(), *this );
	maybeAccept( offsetofExpr->get_type(), *this );
}

void Visitor::visit( OffsetofExpr *offsetofExpr ) {
	maybeAccept( offsetofExpr->get_result(), *this );
	maybeAccept( offsetofExpr->get_type(), *this );
	maybeAccept( offsetofExpr->get_member(), *this );
}

void Visitor::visit( OffsetPackExpr *offsetPackExpr ) {
	maybeAccept( offsetPackExpr->get_result(), *this );
	maybeAccept( offsetPackExpr->get_type(), *this );
}

void Visitor::visit( AttrExpr *attrExpr ) {
	maybeAccept( attrExpr->get_result(), *this );
	if ( attrExpr->get_isType() ) {
		maybeAccept( attrExpr->get_type(), *this );
	} else {
		maybeAccept( attrExpr->get_expr(), *this );
	}
}

void Visitor::visit( LogicalExpr *logicalExpr ) {
	maybeAccept( logicalExpr->get_result(), *this );
	maybeAccept( logicalExpr->get_arg1(), *this );
	maybeAccept( logicalExpr->get_arg2(), *this );
}

void Visitor::visit( ConditionalExpr *conditionalExpr ) {
	maybeAccept( conditionalExpr->get_result(), *this );
	maybeAccept( conditionalExpr->get_arg1(), *this );
	maybeAccept( conditionalExpr->get_arg2(), *this );
	maybeAccept( conditionalExpr->get_arg3(), *this );
}

void Visitor::visit( CommaExpr *commaExpr ) {
	maybeAccept( commaExpr->get_result(), *this );
	maybeAccept( commaExpr->get_arg1(), *this );
	maybeAccept( commaExpr->get_arg2(), *this );
}

void Visitor::visit( TypeExpr *typeExpr ) {
	maybeAccept( typeExpr->get_result(), *this );
	maybeAccept( typeExpr->get_type(), *this );
}

void Visitor::visit( AsmExpr *asmExpr ) {
	maybeAccept( asmExpr->get_inout(), *this );
	maybeAccept( asmExpr->get_constraint(), *this );
	maybeAccept( asmExpr->get_operand(), *this );
}

void Visitor::visit( ImplicitCopyCtorExpr *impCpCtorExpr ) {
	maybeAccept( impCpCtorExpr->get_result(), *this );
	maybeAccept( impCpCtorExpr->get_callExpr(), *this );
	acceptAll( impCpCtorExpr->get_tempDecls(), *this );
	acceptAll( impCpCtorExpr->get_returnDecls(), *this );
	acceptAll( impCpCtorExpr->get_dtors(), *this );
}

void Visitor::visit( ConstructorExpr * ctorExpr ) {
	maybeAccept( ctorExpr->get_result(), *this );
	maybeAccept( ctorExpr->get_callExpr(), *this );
}

void Visitor::visit( CompoundLiteralExpr *compLitExpr ) {
	maybeAccept( compLitExpr->get_result(), *this );
	maybeAccept( compLitExpr->get_initializer(), *this );
}

void Visitor::visit( RangeExpr *rangeExpr ) {
	maybeAccept( rangeExpr->get_low(), *this );
	maybeAccept( rangeExpr->get_high(), *this );
}

void Visitor::visit( UntypedTupleExpr *tupleExpr ) {
	maybeAccept( tupleExpr->get_result(), *this );
	acceptAll( tupleExpr->get_exprs(), *this );
}

void Visitor::visit( TupleExpr *tupleExpr ) {
	maybeAccept( tupleExpr->get_result(), *this );
	acceptAll( tupleExpr->get_exprs(), *this );
}

void Visitor::visit( TupleIndexExpr *tupleExpr ) {
	maybeAccept( tupleExpr->get_result(), *this );
	maybeAccept( tupleExpr->get_tuple(), *this );
}

void Visitor::visit( TupleAssignExpr *assignExpr ) {
	maybeAccept( assignExpr->get_result(), *this );
	maybeAccept( assignExpr->get_stmtExpr(), *this );
}

void Visitor::visit( StmtExpr *stmtExpr ) {
	maybeAccept( stmtExpr->get_result(), *this );
	maybeAccept( stmtExpr->get_statements(), *this );
	acceptAll( stmtExpr->get_returnDecls(), *this );
	acceptAll( stmtExpr->get_dtors(), *this );
}

void Visitor::visit( UniqueExpr *uniqueExpr ) {
	maybeAccept( uniqueExpr->get_result(), *this );
	maybeAccept( uniqueExpr->get_expr(), *this );
}


void Visitor::visit( VoidType *voidType ) {
	acceptAll( voidType->get_forall(), *this );
}

void Visitor::visit( BasicType *basicType ) {
	acceptAll( basicType->get_forall(), *this );
}

void Visitor::visit( PointerType *pointerType ) {
	acceptAll( pointerType->get_forall(), *this );
	maybeAccept( pointerType->get_base(), *this );
}

void Visitor::visit( ArrayType *arrayType ) {
	acceptAll( arrayType->get_forall(), *this );
	maybeAccept( arrayType->get_dimension(), *this );
	maybeAccept( arrayType->get_base(), *this );
}

void Visitor::visit( FunctionType *functionType ) {
	acceptAll( functionType->get_forall(), *this );
	acceptAll( functionType->get_returnVals(), *this );
	acceptAll( functionType->get_parameters(), *this );
}

void Visitor::handleReferenceToType( ReferenceToType *aggregateUseType ) {
	acceptAll( aggregateUseType->get_forall(), *this );
	acceptAll( aggregateUseType->get_parameters(), *this );
}

void Visitor::visit( StructInstType *aggregateUseType ) {
	handleReferenceToType( static_cast< ReferenceToType * >( aggregateUseType ) );
}

void Visitor::visit( UnionInstType *aggregateUseType ) {
	handleReferenceToType( static_cast< ReferenceToType * >( aggregateUseType ) );
}

void Visitor::visit( EnumInstType *aggregateUseType ) {
	handleReferenceToType( static_cast< ReferenceToType * >( aggregateUseType ) );
}

void Visitor::visit( TraitInstType *aggregateUseType ) {
	handleReferenceToType( static_cast< ReferenceToType * >( aggregateUseType ) );
	acceptAll( aggregateUseType->get_members(), *this );
}

void Visitor::visit( TypeInstType *aggregateUseType ) {
	handleReferenceToType( static_cast< ReferenceToType * >( aggregateUseType ) );
}

void Visitor::visit( TupleType *tupleType ) {
	acceptAll( tupleType->get_forall(), *this );
	acceptAll( tupleType->get_types(), *this );
}

void Visitor::visit( TypeofType *typeofType ) {
	assert( typeofType->get_expr() );
	typeofType->get_expr()->accept( *this );
}

void Visitor::visit( AttrType *attrType ) {
	if ( attrType->get_isType() ) {
		assert( attrType->get_type() );
		attrType->get_type()->accept( *this );
	} else {
		assert( attrType->get_expr() );
		attrType->get_expr()->accept( *this );
	} // if
}

void Visitor::visit( VarArgsType *varArgsType ) {
	acceptAll( varArgsType->get_forall(), *this );
}

void Visitor::visit( ZeroType *zeroType ) {
	acceptAll( zeroType->get_forall(), *this );
}

void Visitor::visit( OneType *oneType ) {
	acceptAll( oneType->get_forall(), *this );
}


void Visitor::visit( SingleInit *singleInit ) {
	singleInit->get_value()->accept( *this );
}

void Visitor::visit( ListInit *listInit ) {
	acceptAll( listInit->get_designators(), *this );
	acceptAll( listInit->get_initializers(), *this );
}

void Visitor::visit( ConstructorInit *ctorInit ) {
	maybeAccept( ctorInit->get_ctor(), *this );
	maybeAccept( ctorInit->get_dtor(), *this );
	maybeAccept( ctorInit->get_init(), *this );
}


void Visitor::visit( __attribute__((unused)) Subrange *subrange ) {}


void Visitor::visit( __attribute__((unused)) Constant *constant ) {}
// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
