//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CodeGenerator.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Mar 30 16:38:01 2017
// Update Count     : 482
//

#include <algorithm>
#include <iostream>
#include <cassert>
#include <list>

#include "Parser/ParseNode.h"

#include "SynTree/Declaration.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Statement.h"
#include "SynTree/Type.h"
#include "SynTree/Attribute.h"

#include "Common/utility.h"
#include "Common/UnimplementedError.h"

#include "CodeGenerator.h"
#include "OperatorTable.h"
#include "GenType.h"

#include "InitTweak/InitTweak.h"

using namespace std;

namespace CodeGen {
	int CodeGenerator::tabsize = 4;

	// the kinds of statements that would ideally be followed by whitespace
	bool wantSpacing( Statement * stmt) {
		return dynamic_cast< IfStmt * >( stmt ) || dynamic_cast< CompoundStmt * >( stmt ) ||
			dynamic_cast< WhileStmt * >( stmt ) || dynamic_cast< ForStmt * >( stmt ) || dynamic_cast< SwitchStmt *>( stmt );
	}

	void CodeGenerator::extension( Expression * expr ) {
		if ( expr->get_extension() ) {
			output << "__extension__ ";
		} // if
	} // extension

	void CodeGenerator::extension( Declaration * decl ) {
		if ( decl->get_extension() ) {
			output << "__extension__ ";
		} // if
	} // extension

	void CodeGenerator::asmName( DeclarationWithType * decl ) {
		if ( ConstantExpr * asmName = decl->get_asmName() ) {
			output << " asm ( " << asmName->get_constant()->get_value() << " )";
		} // if
	} // extension

	ostream & CodeGenerator::Indenter::operator()( ostream & output ) const {
	  return output << string( cg.cur_indent, ' ' );
	}

	ostream & operator<<( ostream & output, const CodeGenerator::Indenter &indent ) {
		return indent( output );
	}

	CodeGenerator::LabelPrinter & CodeGenerator::LabelPrinter::operator()( std::list< Label > & l ) {
		labels = &l;
		return *this;
	}

	ostream & operator<<( ostream & output, CodeGenerator::LabelPrinter & printLabels ) {
		std::list< Label > & labs = *printLabels.labels;
		// l.unique(); // assumes a sorted list. Why not use set? Does order matter?
		for ( Label & l : labs ) {
			output << l.get_name() + ": ";
			printLabels.cg.genAttributes( l.get_attributes() );
		} // for
		return output;
	}

	CodeGenerator::CodeGenerator( std::ostream & os, bool pretty ) : indent( *this), cur_indent( 0 ), insideFunction( false ), output( os ), printLabels( *this ), pretty( pretty ) {}

	CodeGenerator::CodeGenerator( std::ostream & os, std::string init, int indentation, bool infunp )
			: indent( *this), cur_indent( indentation ), insideFunction( infunp ), output( os ), printLabels( *this ) {
		//output << std::string( init );
	}

	CodeGenerator::CodeGenerator( std::ostream & os, char * init, int indentation, bool infunp )
			: indent( *this ), cur_indent( indentation ), insideFunction( infunp ), output( os ), printLabels( *this ) {
		//output << std::string( init );
	}

	string CodeGenerator::mangleName( DeclarationWithType * decl ) {
		if ( pretty ) return decl->get_name();
		if ( decl->get_mangleName() != "" ) {
			// need to incorporate scope level in order to differentiate names for destructors
			return decl->get_scopedMangleName();
		} else {
			return decl->get_name();
		} // if
	}

	void CodeGenerator::genAttributes( list< Attribute * > & attributes ) {
	  if ( attributes.empty() ) return;
		output << "__attribute__ ((";
		for ( list< Attribute * >::iterator attr( attributes.begin() );; ) {
			output << (*attr)->get_name();
			if ( ! (*attr)->get_parameters().empty() ) {
				output << "(";
				genCommaList( (*attr)->get_parameters().begin(), (*attr)->get_parameters().end() );
				output << ")";
			} // if
		  if ( ++attr == attributes.end() ) break;
			output << ",";								// separator
		} // for
		output << ")) ";
	} // CodeGenerator::genAttributes


	//*** Declarations
	void CodeGenerator::visit( FunctionDecl * functionDecl ) {
		extension( functionDecl );
		genAttributes( functionDecl->get_attributes() );

		handleStorageClass( functionDecl );
		functionDecl->get_funcSpec().print( output );

		output << genType( functionDecl->get_functionType(), mangleName( functionDecl ), pretty );

		asmName( functionDecl );

		// acceptAll( functionDecl->get_oldDecls(), *this );
		if ( functionDecl->get_statements() ) {
			functionDecl->get_statements()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( ObjectDecl * objectDecl ) {
		if (objectDecl->get_name().empty()) {
			static UniqueName name = { "__anonymous_object" };
			objectDecl->set_name( name.newName() );
		}

		extension( objectDecl );
		genAttributes( objectDecl->get_attributes() );

		handleStorageClass( objectDecl );
		output << genType( objectDecl->get_type(), mangleName( objectDecl ), pretty );

		asmName( objectDecl );

		if ( objectDecl->get_init() ) {
			output << " = ";
			objectDecl->get_init()->accept( *this );
		} // if

		if ( objectDecl->get_bitfieldWidth() ) {
			output << ":";
			objectDecl->get_bitfieldWidth()->accept( *this );
		} // if
	}

	void CodeGenerator::handleAggregate( AggregateDecl * aggDecl ) {
		genAttributes( aggDecl->get_attributes() );

		if ( aggDecl->get_name() != "" )
			output << aggDecl->get_name();

		// std::list< Declaration * > & memb = aggDecl->get_members();
		// if ( ! memb.empty() ) {
		if ( aggDecl->has_body() ) {
			std::list< Declaration * > & memb = aggDecl->get_members();
			output << " {" << endl;

			cur_indent += CodeGenerator::tabsize;
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end(); i++ ) {
				output << indent;
				(*i)->accept( *this );
				output << ";" << endl;
			} // for

			cur_indent -= CodeGenerator::tabsize;

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( StructDecl * structDecl ) {
		extension( structDecl );
		output << "struct ";
		handleAggregate( structDecl );
	}

	void CodeGenerator::visit( UnionDecl * unionDecl ) {
		extension( unionDecl );
		output << "union ";
		handleAggregate( unionDecl );
	}

	void CodeGenerator::visit( EnumDecl * enumDecl ) {
		extension( enumDecl );
		output << "enum ";
		genAttributes( enumDecl->get_attributes() );

		if ( enumDecl->get_name() != "" )
			output << enumDecl->get_name();

		std::list< Declaration* > &memb = enumDecl->get_members();

		if ( ! memb.empty() ) {
			output << " {" << endl;

			cur_indent += CodeGenerator::tabsize;
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end();  i++) {
				ObjectDecl * obj = dynamic_cast< ObjectDecl* >( *i );
				assert( obj );
				output << indent << mangleName( obj );
				if ( obj->get_init() ) {
					output << " = ";
					obj->get_init()->accept( *this );
				} // if
				output << "," << endl;
			} // for

			cur_indent -= CodeGenerator::tabsize;

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( TraitDecl * traitDecl ) {}

	void CodeGenerator::visit( TypedefDecl * typeDecl ) {
		assert( false && "Typedefs are removed and substituted in earlier passes." );
		//output << "typedef ";
		//output << genType( typeDecl->get_base(), typeDecl->get_name(), pretty );
	}

	void CodeGenerator::visit( TypeDecl * typeDecl ) {
		// really, we should mutate this into something that isn't a TypeDecl but that requires large-scale changes,
		// still to be done
		extension( typeDecl );
		output << "extern unsigned long " << typeDecl->get_name();
		if ( typeDecl->get_base() ) {
			output << " = sizeof( " << genType( typeDecl->get_base(), "", pretty ) << " )";
		} // if
	}

	void CodeGenerator::printDesignators( std::list< Expression * > & designators ) {
		typedef std::list< Expression * > DesignatorList;
		if ( designators.size() == 0 ) return;
		for ( DesignatorList::iterator iter = designators.begin(); iter != designators.end(); ++iter ) {
			if ( dynamic_cast< NameExpr * >( *iter ) ) {
				// if expression is a name, then initializing aggregate member
				output << ".";
				(*iter)->accept( *this );
			} else {
				// if not a simple name, it has to be a constant expression, i.e. an array designator
				output << "[";
				(*iter)->accept( *this );
				output << "]";
			} // if
		} // for
		output << " = ";
	}

	void CodeGenerator::visit( SingleInit * init ) {
		printDesignators( init->get_designators() );
		init->get_value()->accept( *this );
	}

	void CodeGenerator::visit( ListInit * init ) {
		printDesignators( init->get_designators() );
		output << "{ ";
		if ( init->begin() == init->end() ) {
			// illegal to leave initializer list empty for scalar initializers, but always legal to have 0
			output << "0";
		} else {
			genCommaList( init->begin(), init->end() );
		} // if
		output << " }";
	}

	void CodeGenerator::visit( ConstructorInit * init ){
		assertf( false, "ConstructorInit nodes should not make it to CodeGen." );
	}

	void CodeGenerator::visit( Constant * constant ) {
		output << constant->get_value() ;
	}

	//*** Expressions
	void CodeGenerator::visit( ApplicationExpr * applicationExpr ) {
		extension( applicationExpr );
		if ( VariableExpr * varExpr = dynamic_cast< VariableExpr* >( applicationExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( varExpr->get_var()->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = applicationExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_PREFIXASSIGN:
				  case OT_POSTFIXASSIGN:
				  case OT_INFIXASSIGN:
				  case OT_CTOR:
				  case OT_DTOR:
					{
						assert( arg != applicationExpr->get_args().end() );
						if ( AddressExpr * addrExpr = dynamic_cast< AddressExpr * >( *arg ) ) {
							// remove & from first assignment/ctor argument
							*arg = addrExpr->get_arg();
						} else {
							// no address-of operator, so must be a pointer - add dereference
							// NOTE: if the assertion starts to trigger, check that the application expr isn't being shared.
							// Since its arguments are modified here, this assertion most commonly triggers when the application
							// is visited multiple times.
							UntypedExpr * newExpr = new UntypedExpr( new NameExpr( "*?" ) );
							newExpr->get_args().push_back( *arg );
							Type * type = InitTweak::getPointerBase( (*arg)->get_result() );
							assertf( type, "First argument to a derefence must be a pointer. Ensure that expressions are not being shared." );
							newExpr->set_result( type->clone() );
							*arg = newExpr;
						} // if
						break;
					}

				  default:
					// do nothing
					;
				} // switch

				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( applicationExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;

				  case OT_CALL:
					// there are no intrinsic definitions of the function call operator
					assert( false );
					break;

				  case OT_CTOR:
				  case OT_DTOR:
					if ( applicationExpr->get_args().size() == 1 ) {
						// the expression fed into a single parameter constructor or destructor may contain side
						// effects, so must still output this expression
						output << "(";
						(*arg++)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else if ( applicationExpr->get_args().size() == 2 ) {
						// intrinsic two parameter constructors are essentially bitwise assignment
						output << "(";
						(*arg++)->accept( *this );
						output << opInfo.symbol;
						(*arg)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else {
						// no constructors with 0 or more than 2 parameters
						assert( false );
					} // if
					break;

				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;


				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( applicationExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_CONSTANT:
				  case OT_LABELADDRESS:
					// there are no intrinsic definitions of 0/1 or label addresses as functions
					assert( false );
				} // switch
			} else {
				varExpr->accept( *this );
				output << "(";
				genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
				output << ")";
			} // if
		} else {
			applicationExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
			output << ")";
		} // if
	}

	void CodeGenerator::visit( UntypedExpr * untypedExpr ) {
		extension( untypedExpr );
		if ( NameExpr * nameExpr = dynamic_cast< NameExpr* >( untypedExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = untypedExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( untypedExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;

				  case OT_CALL:
					assert( false );

				  case OT_CTOR:
				  case OT_DTOR:
					if ( untypedExpr->get_args().size() == 1 ) {
						// the expression fed into a single parameter constructor or destructor may contain side
						// effects, so must still output this expression
						output << "(";
						(*arg++)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else if ( untypedExpr->get_args().size() == 2 ) {
						// intrinsic two parameter constructors are essentially bitwise assignment
						output << "(";
						(*arg++)->accept( *this );
						output << opInfo.symbol;
						(*arg)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else {
						// no constructors with 0 or more than 2 parameters
						assert( false );
					} // if
					break;

				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
				  case OT_LABELADDRESS:
					assert( untypedExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( untypedExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;

				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( untypedExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_CONSTANT:
					// there are no intrinsic definitions of 0 or 1 as functions
					assert( false );
				} // switch
			} else {
				if ( nameExpr->get_name() == "..." ) { // case V1 ... V2 or case V1~V2
					assert( untypedExpr->get_args().size() == 2 );
					(*untypedExpr->get_args().begin())->accept( *this );
					output << " ... ";
					(*--untypedExpr->get_args().end())->accept( *this );
				} else {								// builtin routines
					nameExpr->accept( *this );
					output << "(";
					genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
					output << ")";
				} // if
			} // if
		} else {
			untypedExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
			output << ")";
		} // if
	}

	void CodeGenerator::visit( RangeExpr * rangeExpr ) {
		rangeExpr->get_low()->accept( *this );
		output << " ... ";
		rangeExpr->get_high()->accept( *this );
	}

	void CodeGenerator::visit( NameExpr * nameExpr ) {
		extension( nameExpr );
		OperatorInfo opInfo;
		if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
			assert( opInfo.type == OT_CONSTANT );
			output << opInfo.symbol;
		} else {
			output << nameExpr->get_name();
		} // if
	}

	void CodeGenerator::visit( AddressExpr * addressExpr ) {
		extension( addressExpr );
		output << "(&";
		// this hack makes sure that we don't convert "constant_zero" to "0" if we're taking its address
		if ( VariableExpr * variableExpr = dynamic_cast< VariableExpr* >( addressExpr->get_arg() ) ) {
			output << mangleName( variableExpr->get_var() );
		} else {
			addressExpr->get_arg()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( CastExpr * castExpr ) {
		extension( castExpr );
		output << "(";
		if ( castExpr->get_result()->isVoid() ) {
			output << "(void)" ;
		} else if ( ! castExpr->get_result()->get_lvalue() ) {
			// at least one result type of cast, but not an lvalue
			output << "(";
			output << genType( castExpr->get_result(), "", pretty );
			output << ")";
		} else {
			// otherwise, the cast is to an lvalue type, so the cast should be dropped, since the result of a cast is
			// never an lvalue in C
		} // if
		castExpr->get_arg()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( UntypedMemberExpr * memberExpr ) {
		assert( false );
	}

	void CodeGenerator::visit( MemberExpr * memberExpr ) {
		extension( memberExpr );
		memberExpr->get_aggregate()->accept( *this );
		output << "." << mangleName( memberExpr->get_member() );
	}

	void CodeGenerator::visit( VariableExpr * variableExpr ) {
		extension( variableExpr );
		OperatorInfo opInfo;
		if ( variableExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( variableExpr->get_var()->get_name(), opInfo ) && opInfo.type == OT_CONSTANT ) {
			output << opInfo.symbol;
		} else {
			output << mangleName( variableExpr->get_var() );
		} // if
	}

	void CodeGenerator::visit( ConstantExpr * constantExpr ) {
		assert( constantExpr->get_constant() );
		extension( constantExpr );
		constantExpr->get_constant()->accept( *this );
	}

	void CodeGenerator::visit( SizeofExpr * sizeofExpr ) {
		extension( sizeofExpr );
		output << "sizeof(";
		if ( sizeofExpr->get_isType() ) {
			output << genType( sizeofExpr->get_type(), "", pretty );
		} else {
			sizeofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( AlignofExpr * alignofExpr ) {
		// use GCC extension to avoid bumping std to C11
		extension( alignofExpr );
		output << "__alignof__(";
		if ( alignofExpr->get_isType() ) {
			output << genType( alignofExpr->get_type(), "", pretty );
		} else {
			alignofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( UntypedOffsetofExpr * offsetofExpr ) {
		assert( false && "UntypedOffsetofExpr should not reach code generation." );
	}

	void CodeGenerator::visit( OffsetofExpr * offsetofExpr ) {
		// use GCC builtin
		output << "__builtin_offsetof(";
		output << genType( offsetofExpr->get_type(), "", pretty );
		output << ", " << mangleName( offsetofExpr->get_member() );
		output << ")";
	}

	void CodeGenerator::visit( OffsetPackExpr * offsetPackExpr ) {
		assert( false && "OffsetPackExpr should not reach code generation." );
	}

	void CodeGenerator::visit( LogicalExpr * logicalExpr ) {
		extension( logicalExpr );
		output << "(";
		logicalExpr->get_arg1()->accept( *this );
		if ( logicalExpr->get_isAnd() ) {
			output << " && ";
		} else {
			output << " || ";
		} // if
		logicalExpr->get_arg2()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( ConditionalExpr * conditionalExpr ) {
		extension( conditionalExpr );
		output << "(";
		conditionalExpr->get_arg1()->accept( *this );
		output << " ? ";
		conditionalExpr->get_arg2()->accept( *this );
		output << " : ";
		conditionalExpr->get_arg3()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( CommaExpr * commaExpr ) {
		extension( commaExpr );
		output << "(";
		commaExpr->get_arg1()->accept( *this );
		output << " , ";
		commaExpr->get_arg2()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( UntypedTupleExpr * tupleExpr ) { assertf( false, "UntypedTupleExpr should not make it to Code Gen" ); }

	void CodeGenerator::visit( TupleExpr * tupleExpr ) { assertf( false, "TupleExpr should not make it to Code Gen" ); }

	void CodeGenerator::visit( TypeExpr * typeExpr ) {}

	void CodeGenerator::visit( AsmExpr * asmExpr ) {
		if ( asmExpr->get_inout() ) {
			output << "[ ";
			asmExpr->get_inout()->accept( *this );
			output << " ] ";
		} // if
		asmExpr->get_constraint()->accept( *this );
		output << " ( ";
		asmExpr->get_operand()->accept( *this );
		output << " )";
	}

	void CodeGenerator::visit( CompoundLiteralExpr *compLitExpr ) {
		assert( compLitExpr->get_result() && dynamic_cast< ListInit * > ( compLitExpr->get_initializer() ) );
		output << "(" << genType( compLitExpr->get_result(), "", pretty ) << ")";
		compLitExpr->get_initializer()->accept( *this );
	}

	void CodeGenerator::visit( StmtExpr * stmtExpr ) {
		std::list< Statement * > & stmts = stmtExpr->get_statements()->get_kids();
		output << "({" << std::endl;
		cur_indent += CodeGenerator::tabsize;
		unsigned int numStmts = stmts.size();
		unsigned int i = 0;
		for ( Statement * stmt : stmts ) {
			output << indent << printLabels( stmt->get_labels() );
			if ( i+1 == numStmts ) {
				// last statement in a statement expression needs to be handled specially -
				// cannot cast to void, otherwise the expression statement has no value
				if ( ExprStmt * exprStmt = dynamic_cast< ExprStmt * >( stmt ) ) {
					exprStmt->get_expr()->accept( *this );
					output << ";" << endl;
					++i;
					break;
				}
			}
			stmt->accept( *this );
			output << endl;
			if ( wantSpacing( stmt ) ) {
				output << endl;
			} // if
			++i;
		}
		cur_indent -= CodeGenerator::tabsize;
		output << indent << "})";
	}

	//*** Statements
	void CodeGenerator::visit( CompoundStmt * compoundStmt ) {
		std::list<Statement*> ks = compoundStmt->get_kids();
		output << "{" << endl;

		cur_indent += CodeGenerator::tabsize;

		for ( std::list<Statement *>::iterator i = ks.begin(); i != ks.end();  i++ ) {
			output << indent << printLabels( (*i)->get_labels() );
			(*i)->accept( *this );

			output << endl;
			if ( wantSpacing( *i ) ) {
				output << endl;
			} // if
		} // for
		cur_indent -= CodeGenerator::tabsize;

		output << indent << "}";
	}

	void CodeGenerator::visit( ExprStmt * exprStmt ) {
		assert( exprStmt );
		// cast the top-level expression to void to reduce gcc warnings.
		Expression * expr = new CastExpr( exprStmt->get_expr() );
		expr->accept( *this );
		output << ";";
	}

	void CodeGenerator::visit( AsmStmt * asmStmt ) {
		output << "asm ";
		if ( asmStmt->get_voltile() ) output << "volatile ";
		if ( ! asmStmt->get_gotolabels().empty()  ) output << "goto ";
		output << "( ";
		if ( asmStmt->get_instruction() ) asmStmt->get_instruction()->accept( *this );
		output << " : ";
		genCommaList( asmStmt->get_output().begin(), asmStmt->get_output().end() );
		output << " : ";
		genCommaList( asmStmt->get_input().begin(), asmStmt->get_input().end() );
		output << " : ";
		genCommaList( asmStmt->get_clobber().begin(), asmStmt->get_clobber().end() );
		if ( ! asmStmt->get_gotolabels().empty() ) {
			output << " : ";
			for ( std::list<Label>::iterator begin = asmStmt->get_gotolabels().begin();; ) {
				output << *begin++;
				if ( begin == asmStmt->get_gotolabels().end() ) break;
				output << ", ";
			} // for
		} // if
		output << " );" ;
	}

	void CodeGenerator::visit( AsmDecl * asmDecl ) {
		output << "asm ";
		AsmStmt * asmStmt = asmDecl->get_stmt();
		output << "( ";
		if ( asmStmt->get_instruction() ) asmStmt->get_instruction()->accept( *this );
		output << " )" ;
	}

	void CodeGenerator::visit( IfStmt * ifStmt ) {
		output << "if ( ";
		ifStmt->get_condition()->accept( *this );
		output << " ) ";

		ifStmt->get_thenPart()->accept( *this );

		if ( ifStmt->get_elsePart() != 0) {
			output << " else ";
			ifStmt->get_elsePart()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( SwitchStmt * switchStmt ) {
		output << "switch ( " ;
		switchStmt->get_condition()->accept( *this );
		output << " ) ";

		output << "{" << std::endl;
		cur_indent += CodeGenerator::tabsize;
		acceptAll( switchStmt->get_statements(), *this );
		cur_indent -= CodeGenerator::tabsize;
		output << indent << "}";
	}

	void CodeGenerator::visit( CaseStmt * caseStmt ) {
		output << indent;
		if ( caseStmt->isDefault()) {
			output << "default";
		} else {
			output << "case ";
			caseStmt->get_condition()->accept( *this );
		} // if
		output << ":\n";

		std::list<Statement *> sts = caseStmt->get_statements();

		cur_indent += CodeGenerator::tabsize;
		for ( std::list<Statement *>::iterator i = sts.begin(); i != sts.end();  i++) {
			output << indent << printLabels( (*i)->get_labels() )  ;
			(*i)->accept( *this );
			output << endl;
		} // for
		cur_indent -= CodeGenerator::tabsize;
	}

	void CodeGenerator::visit( BranchStmt * branchStmt ) {
		switch ( branchStmt->get_type()) {
		  case BranchStmt::Goto:
			if ( ! branchStmt->get_target().empty() )
				output << "goto " << branchStmt->get_target();
			else {
				if ( branchStmt->get_computedTarget() != 0 ) {
					output << "goto *";
					branchStmt->get_computedTarget()->accept( *this );
				} // if
			} // if
			break;
		  case BranchStmt::Break:
			output << "break";
			break;
		  case BranchStmt::Continue:
			output << "continue";
			break;
		} // switch
		output << ";";
	}

	void CodeGenerator::visit( ReturnStmt * returnStmt ) {
		output << "return ";
		maybeAccept( returnStmt->get_expr(), *this );
		output << ";";
	}

	void CodeGenerator::visit( WhileStmt * whileStmt ) {
		if ( whileStmt->get_isDoWhile() ) {
			output << "do" ;
		} else {
			output << "while (" ;
			whileStmt->get_condition()->accept( *this );
			output << ")";
		} // if
		output << " ";

		output << CodeGenerator::printLabels( whileStmt->get_body()->get_labels() );
		whileStmt->get_body()->accept( *this );

		output << indent;

		if ( whileStmt->get_isDoWhile() ) {
			output << " while (" ;
			whileStmt->get_condition()->accept( *this );
			output << ");";
		} // if
	}

	void CodeGenerator::visit( ForStmt * forStmt ) {
		// initialization is always hoisted, so don't bother doing anything with that
		output << "for (;";

		if ( forStmt->get_condition() != 0 ) {
			forStmt->get_condition()->accept( *this );
		} // if
		output << ";";

		if ( forStmt->get_increment() != 0 ) {
			// cast the top-level expression to void to reduce gcc warnings.
			Expression * expr = new CastExpr( forStmt->get_increment() );
			expr->accept( *this );
		} // if
		output << ") ";

		if ( forStmt->get_body() != 0 ) {
			output << CodeGenerator::printLabels( forStmt->get_body()->get_labels() );
			forStmt->get_body()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( NullStmt * nullStmt ) {
		//output << indent << CodeGenerator::printLabels( nullStmt->get_labels() );
		output << "/* null statement */ ;";
	}

	void CodeGenerator::visit( DeclStmt * declStmt ) {
		declStmt->get_decl()->accept( *this );

		if ( doSemicolon( declStmt->get_decl() ) ) {
			output << ";";
		} // if
	}

	void CodeGenerator::handleStorageClass( DeclarationWithType * decl ) {
		if ( decl->get_storageClasses().any() ) {
			decl->get_storageClasses().print( output );
		} // if
	} // CodeGenerator::handleStorageClass

	std::string genName( DeclarationWithType * decl ) {
		CodeGen::OperatorInfo opInfo;
		if ( operatorLookup( decl->get_name(), opInfo ) ) {
			return opInfo.outputName;
		} else {
			return decl->get_name();
		} // if
	}
} // namespace CodeGen

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
