//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CastCost.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 06:57:43 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Feb  2 15:34:36 2016
// Update Count     : 7
//

#include "typeops.h"
#include "Cost.h"
#include "ConversionCost.h"
#include "SynTree/Type.h"
#include "SynTree/Visitor.h"
#include "SymTab/Indexer.h"


namespace ResolvExpr {
	class CastCost : public ConversionCost {
	  public:
		CastCost( Type *dest, const SymTab::Indexer &indexer, const TypeEnvironment &env );

		virtual void visit( BasicType *basicType );
		virtual void visit( PointerType *pointerType );
	};

	Cost castCost( Type *src, Type *dest, const SymTab::Indexer &indexer, const TypeEnvironment &env ) {
		if ( TypeInstType *destAsTypeInst = dynamic_cast< TypeInstType* >( dest ) ) {
			EqvClass eqvClass;
			NamedTypeDecl *namedType;
			if ( env.lookup( destAsTypeInst->get_name(), eqvClass ) ) {
				if ( eqvClass.type ) {
					return castCost( src, eqvClass.type, indexer, env );
				} else {
					return Cost::infinity;
				}
			} else if ( ( namedType = indexer.lookupType( destAsTypeInst->get_name() ) ) ) {
				TypeDecl *type = dynamic_cast< TypeDecl* >( namedType );
				// all typedefs should be gone by this point
				assert( type );
				if ( type->get_base() ) {
					return castCost( src, type->get_base(), indexer, env ) + Cost( 0, 0, 1 );
				} // if
			} // if
		} // if
		if ( typesCompatibleIgnoreQualifiers( src, dest, indexer, env ) ) {
			return Cost( 0, 0, 0 );
		} else if ( dynamic_cast< VoidType* >( dest ) ) {
			return Cost( 0, 0, 1 );
		} else if ( ReferenceType * refType = dynamic_cast< ReferenceType * > ( dest ) ) {
			return convertToReferenceCost( src, refType, indexer, env );
		} else {
			CastCost converter( dest, indexer, env );
			src->accept( converter );
			if ( converter.get_cost() == Cost::infinity ) {
				return Cost::infinity;
			} else {
				// xxx - why are we adding cost 0 here?
				return converter.get_cost() + Cost( 0, 0, 0 );
			} // if
		} // if
	}

	CastCost::CastCost( Type *dest, const SymTab::Indexer &indexer, const TypeEnvironment &env )
		: ConversionCost( dest, indexer, env ) {
	}

	void CastCost::visit( BasicType *basicType ) {
		PointerType *destAsPointer = dynamic_cast< PointerType* >( dest );
		if ( destAsPointer && basicType->isInteger() ) {
			// necessary for, e.g. unsigned long => void*
			cost = Cost( 1, 0, 0 );
		} else {
			cost = conversionCost( basicType, dest, indexer, env );
		} // if
	}

	void CastCost::visit( PointerType *pointerType ) {
		if ( PointerType *destAsPtr = dynamic_cast< PointerType* >( dest ) ) {
			if ( pointerType->get_qualifiers() <= destAsPtr->get_qualifiers() && typesCompatibleIgnoreQualifiers( pointerType->get_base(), destAsPtr->get_base(), indexer, env ) ) {
				cost = Cost( 0, 0, 1 );
			} else {
				TypeEnvironment newEnv( env );
				newEnv.add( pointerType->get_forall() );
				newEnv.add( pointerType->get_base()->get_forall() );
				int castResult = ptrsCastable( pointerType->get_base(), destAsPtr->get_base(), newEnv, indexer );
				if ( castResult > 0 ) {
					cost = Cost( 0, 0, 1 );
				} else if ( castResult < 0 ) {
					cost = Cost::infinity;
				} // if
			} // if
		} else if ( BasicType *destAsBasic = dynamic_cast< BasicType* >( dest ) ) {
			if ( destAsBasic->isInteger() ) {
				// necessary for, e.g. void* => unsigned long
				cost = Cost( 1, 0, 0 );
			} // if
		}
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
