#!python
from __future__ import print_function

from os import listdir
from os.path import isfile, join
from subprocess import Popen, PIPE, STDOUT

import argparse
import sys

################################################################################
#               help functions
################################################################################
def listTests():
	list = [f.rstrip('.c') for f in listdir('.')
		if not f.startswith('.') and (
			not isfile(f) or f.endswith('.c')
		)]

	return list

def sh(cmd, dry_run):
	if dry_run :
		print("cmd: %s" % cmd)
		return 0
	else :
		proc = Popen(cmd, stderr=STDOUT, shell=True)
		proc.communicate()
		return proc.returncode

################################################################################
#               running test functions
################################################################################
def run_test_instance(test, dry_run):
	sh("rm -f .out/%s.log" % test, dry_run)
	sh("rm -f %s > /dev/null 2>&1" % test, dry_run)

	# build, skipping to next test on error
	sh("make -j 1 %s >> .out/%s.log 2>&1" % (test, test), dry_run)

	# fetch optional input
	stdinput = "< .in/%s.txt" % test if isfile(".in/%s.txt" % test) else ""

	# run test
	sh("./%s %s >> .out/%s.log 2>&1" % (test, stdinput, test), dry_run)

	# touch expected files so empty output are supported by default
	sh("touch .expect/%s.txt" % test, dry_run)

	# diff the output of the files
	retcode = sh("diff .expect/%s.txt .out/%s.log" % (test, test), dry_run)

	# clean the executable
	sh("rm -f %s > /dev/null 2>&1" % test, dry_run)

	return retcode

def run_tests(tests, dry_run) :
	sh('make clean > /dev/null 2>&1', dry_run)

	failed = False;
	for t in tests:
		print("%10s  " % t, end="")
		sys.stdout.flush()
		test_failed = run_test_instance(t, dry_run)
		failed = test_failed or failed
		print("FAILED" if test_failed else "PASSED")

	sh('make clean > /dev/null 2>&1', dry_run)

	return 0 if failed else 1

################################################################################
#               generate expectation functions
################################################################################
def generate_test_expect(test):
	sh("rm -f .out/%s.log" % test, False)
	sh("rm -f %s > /dev/null 2>&1" % test, False)

	# build, skipping to next test on error
	sh("make -j 8 %s > .expect/%s.txt 2>&1" % (test, test), False)

	# fetch optional input
	stdinput = "< .in/%s.txt" % test if isfile(".in/%s.txt" % test) else ""

	# run test
	sh("./%s %s >> .expect/%s.txt 2>&1" % (test, stdinput, test), False)

	# clean the executable
	sh("rm -f %s > /dev/null 2>&1" % test, False)

def generate_expect(tests) :
	sh('make clean > /dev/null 2>&1', False)

	print( "Regenerate tests for" )

	print( ", ".join( tests ) )

	for t in tests:
		generate_test_expect( t )

	print( "Done" )

	sh('make clean > /dev/null 2>&1', False)

################################################################################
#               main loop
################################################################################
parser = argparse.ArgumentParser(description='Script which runs cforall tests')
parser.add_argument('--dry-run', help='Don\'t run the tests, only output the commands', action='store_true')
parser.add_argument('--all', help='Run all test available', action='store_true')
parser.add_argument('--generate-expected', help='Regenerate the .expect by running the specified tets, can be used with --all option', action='store_true')
parser.add_argument('tests', metavar='test', type=str, nargs='*', help='a list of tests to run')

options = parser.parse_args()

tests = listTests()

if options.all :
	if len(options.tests) > 0 :
		print('ERROR: cannot specify tests with \'--all\' option', file=sys.stderr)
		sys.exit(1)

	if options.generate_expected :
		generate_expect( tests )
	else :
		sys.exit( run_tests(tests, options.dry_run) )

else :
	if len(options.tests) == 0 :
		print('ERROR: must have option \'--all\' or non-empty test list', file=sys.stderr)
		parser.print_help()
		sys.exit(1)

	if options.generate_expected :
		generate_expect( options.tests )
	else :
		sys.exit( run_tests(options.tests, options.dry_run) )

sys.exit(0)
