#include <algorithm>
#include "InitTweak.h"
#include "SynTree/Visitor.h"
#include "SynTree/Statement.h"
#include "SynTree/Initializer.h"
#include "SynTree/Expression.h"
#include "SynTree/Attribute.h"
#include "GenPoly/GenPoly.h"
#include "ResolvExpr/typeops.h"

namespace InitTweak {
	namespace {
		class HasDesignations : public Visitor {
		public:
			bool hasDesignations = false;
			template<typename Init>
			void handleInit( Init * init ) {
				if ( ! init->get_designators().empty() ) hasDesignations = true;
				else Visitor::visit( init );
			}
			virtual void visit( SingleInit * singleInit ) { handleInit( singleInit); }
			virtual void visit( ListInit * listInit ) { handleInit( listInit); }
		};

		class InitDepthChecker : public Visitor {
		public:
			bool depthOkay = true;
			Type * type;
			int curDepth = 0, maxDepth = 0;
			InitDepthChecker( Type * type ) : type( type ) {
				Type * t = type;
				while ( ArrayType * at = dynamic_cast< ArrayType * >( t ) ) {
					maxDepth++;
					t = at->get_base();
				}
				maxDepth++;
			}
			virtual void visit( ListInit * listInit ) {
				curDepth++;
				if ( curDepth > maxDepth ) depthOkay = false;
				Visitor::visit( listInit );
				curDepth--;
			}
		};

		class InitFlattener : public Visitor {
			public:
			virtual void visit( SingleInit * singleInit );
			virtual void visit( ListInit * listInit );
			std::list< Expression * > argList;
		};

		void InitFlattener::visit( SingleInit * singleInit ) {
			argList.push_back( singleInit->get_value()->clone() );
		}

		void InitFlattener::visit( ListInit * listInit ) {
			// flatten nested list inits
			std::list<Initializer*>::iterator it = listInit->begin();
			for ( ; it != listInit->end(); ++it ) {
				(*it)->accept( *this );
			}
		}
	}

	std::list< Expression * > makeInitList( Initializer * init ) {
		InitFlattener flattener;
		maybeAccept( init, flattener );
		return flattener.argList;
	}

	bool isDesignated( Initializer * init ) {
		HasDesignations finder;
		maybeAccept( init, finder );
		return finder.hasDesignations;
	}

	bool checkInitDepth( ObjectDecl * objDecl ) {
		InitDepthChecker checker( objDecl->get_type() );
		maybeAccept( objDecl->get_init(), checker );
		return checker.depthOkay;
	}

	class InitExpander::ExpanderImpl {
	public:
		virtual std::list< Expression * > next( std::list< Expression * > & indices ) = 0;
		virtual Statement * buildListInit( UntypedExpr * callExpr, std::list< Expression * > & indices ) = 0;
	};

	class InitImpl : public InitExpander::ExpanderImpl {
	public:
		InitImpl( Initializer * init ) : init( init ) {}

		virtual std::list< Expression * > next( std::list< Expression * > & indices ) {
			// this is wrong, but just a placeholder for now
			// if ( ! flattened ) flatten( indices );
			// return ! inits.empty() ? makeInitList( inits.front() ) : std::list< Expression * >();
			return makeInitList( init );
		}

		virtual Statement * buildListInit( UntypedExpr * callExpr, std::list< Expression * > & indices );
	private:
		Initializer * init;
	};

	class ExprImpl : public InitExpander::ExpanderImpl {
	public:
		ExprImpl( Expression * expr ) : arg( expr ) {}

		~ExprImpl() { delete arg; }

		virtual std::list< Expression * > next( std::list< Expression * > & indices ) {
			std::list< Expression * > ret;
			Expression * expr = maybeClone( arg );
			if ( expr ) {
				for ( std::list< Expression * >::reverse_iterator it = indices.rbegin(); it != indices.rend(); ++it ) {
					// go through indices and layer on subscript exprs ?[?]
					++it;
					UntypedExpr * subscriptExpr = new UntypedExpr( new NameExpr( "?[?]") );
					subscriptExpr->get_args().push_back( expr );
					subscriptExpr->get_args().push_back( (*it)->clone() );
					expr = subscriptExpr;
				}
				ret.push_back( expr );
			}
			return ret;
		}

		virtual Statement * buildListInit( UntypedExpr * callExpr, std::list< Expression * > & indices );
	private:
		Expression * arg;
	};

	InitExpander::InitExpander( Initializer * init ) : expander( new InitImpl( init ) ) {}

	InitExpander::InitExpander( Expression * expr ) : expander( new ExprImpl( expr ) ) {}

	std::list< Expression * > InitExpander::operator*() {
		return cur;
	}

	InitExpander & InitExpander::operator++() {
		cur = expander->next( indices );
		return *this;
	}

	// use array indices list to build switch statement
	void InitExpander::addArrayIndex( Expression * index, Expression * dimension ) {
		indices.push_back( index );
		indices.push_back( dimension );
	}

	void InitExpander::clearArrayIndices() {
		deleteAll( indices );
		indices.clear();
	}

	namespace {
		/// given index i, dimension d, initializer init, and callExpr f, generates
		///   if (i < d) f(..., init)
		///   ++i;
		/// so that only elements within the range of the array are constructed
		template< typename OutIterator >
		void buildCallExpr( UntypedExpr * callExpr, Expression * index, Expression * dimension, Initializer * init, OutIterator out ) {
			UntypedExpr * cond = new UntypedExpr( new NameExpr( "?<?") );
			cond->get_args().push_back( index->clone() );
			cond->get_args().push_back( dimension->clone() );

			std::list< Expression * > args = makeInitList( init );
			callExpr->get_args().splice( callExpr->get_args().end(), args );

			*out++ = new IfStmt( noLabels, cond, new ExprStmt( noLabels, callExpr ), NULL );

			UntypedExpr * increment = new UntypedExpr( new NameExpr( "++?" ) );
			increment->get_args().push_back( new AddressExpr( index->clone() ) );
			*out++ = new ExprStmt( noLabels, increment );
		}

		template< typename OutIterator >
		void build( UntypedExpr * callExpr, InitExpander::IndexList::iterator idx, InitExpander::IndexList::iterator idxEnd, Initializer * init, OutIterator out ) {
			if ( idx == idxEnd ) return;
			Expression * index = *idx++;
			assert( idx != idxEnd );
			Expression * dimension = *idx++;

			// xxx - may want to eventually issue a warning here if we can detect
			// that the number of elements exceeds to dimension of the array
			if ( idx == idxEnd ) {
				if ( ListInit * listInit = dynamic_cast< ListInit * >( init ) ) {
					for ( Initializer * init : *listInit ) {
						buildCallExpr( callExpr->clone(), index, dimension, init, out );
					}
				} else {
					buildCallExpr( callExpr->clone(), index, dimension, init, out );
				}
			} else {
				std::list< Statement * > branches;

				unsigned long cond = 0;
				ListInit * listInit = dynamic_cast< ListInit * >( init );
				if ( ! listInit ) {
					// xxx - this shouldn't be an error, but need a way to
					// terminate without creating output, so should catch this error
					throw SemanticError( "unbalanced list initializers" );
				}

				static UniqueName targetLabel( "L__autogen__" );
				Label switchLabel( targetLabel.newName(), 0, std::list< Attribute * >{ new Attribute("unused") } );
				for ( Initializer * init : *listInit ) {
					Expression * condition;
					// check for designations
					// if ( init-> ) {
						condition = new ConstantExpr( Constant::from_ulong( cond ) );
						++cond;
					// } else {
					// 	condition = // ... take designation
					// 	cond = // ... take designation+1
					// }
					std::list< Statement * > stmts;
					build( callExpr, idx, idxEnd, init, back_inserter( stmts ) );
					stmts.push_back( new BranchStmt( noLabels, switchLabel, BranchStmt::Break ) );
					CaseStmt * caseStmt = new CaseStmt( noLabels, condition, stmts );
					branches.push_back( caseStmt );
				}
				*out++ = new SwitchStmt( noLabels, index->clone(), branches );
				*out++ = new NullStmt( std::list<Label>{ switchLabel } );
			}
		}
	}

	// if array came with an initializer list: initialize each element
	// may have more initializers than elements in the array - need to check at each index that
	// we haven't exceeded size.
	// may have fewer initializers than elements in the array - need to default construct
	// remaining elements.
	// To accomplish this, generate switch statement, consuming all of expander's elements
	Statement * InitImpl::buildListInit( UntypedExpr * dst, std::list< Expression * > & indices ) {
		if ( ! init ) return NULL;
		CompoundStmt * block = new CompoundStmt( noLabels );
		build( dst, indices.begin(), indices.end(), init, back_inserter( block->get_kids() ) );
		if ( block->get_kids().empty() ) {
			delete block;
			return NULL;
		} else {
			init = NULL; // init was consumed in creating the list init
			return block;
		}
	}

	Statement * ExprImpl::buildListInit( UntypedExpr * dst, std::list< Expression * > & indices ) {
		return NULL;
	}

	Statement * InitExpander::buildListInit( UntypedExpr * dst ) {
		return expander->buildListInit( dst, indices );
	}

	bool tryConstruct( ObjectDecl * objDecl ) {
		return ! LinkageSpec::isBuiltin( objDecl->get_linkage() ) &&
			(objDecl->get_init() == NULL ||
				( objDecl->get_init() != NULL && objDecl->get_init()->get_maybeConstructed() ))
			&& objDecl->get_storageClass() != DeclarationNode::Extern;
	}

	class CallFinder : public Visitor {
	public:
		typedef Visitor Parent;
		CallFinder( const std::list< std::string > & names ) : names( names ) {}

		virtual void visit( ApplicationExpr * appExpr ) {
			handleCallExpr( appExpr );
		}

		virtual void visit( UntypedExpr * untypedExpr ) {
			handleCallExpr( untypedExpr );
		}

		std::list< Expression * > * matches;
	private:
		const std::list< std::string > names;

		template< typename CallExpr >
		void handleCallExpr( CallExpr * expr ) {
			Parent::visit( expr );
			std::string fname = getFunctionName( expr );
			if ( std::find( names.begin(), names.end(), fname ) != names.end() ) {
				matches->push_back( expr );
			}
		}
	};

	void collectCtorDtorCalls( Statement * stmt, std::list< Expression * > & matches ) {
		static CallFinder finder( std::list< std::string >{ "?{}", "^?{}" } );
		finder.matches = &matches;
		maybeAccept( stmt, finder );
	}

	Expression * getCtorDtorCall( Statement * stmt ) {
		std::list< Expression * > matches;
		collectCtorDtorCalls( stmt, matches );
		assert( matches.size() <= 1 );
		return matches.size() == 1 ? matches.front() : NULL;
	}

	namespace {
		DeclarationWithType * getCalledFunction( Expression * expr ) {
			assert( expr );
			if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( expr ) ) {
				return varExpr->get_var();
			} else if ( MemberExpr * memberExpr = dynamic_cast< MemberExpr * >( expr ) ) {
				return memberExpr->get_member();
			} else if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( expr ) ) {
				return getCalledFunction( castExpr->get_arg() );
			}
			return nullptr;
		}
	}

	ApplicationExpr * isIntrinsicCallExpr( Expression * expr ) {
		ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( expr );
		if ( ! appExpr ) return NULL;
		DeclarationWithType * function = getCalledFunction( appExpr->get_function() );
		assert( function );
		// check for Intrinsic only - don't want to remove all overridable ctor/dtors because autogenerated ctor/dtor
		// will call all member dtors, and some members may have a user defined dtor.
		return function->get_linkage() == LinkageSpec::Intrinsic ? appExpr : NULL;
	}

	namespace {
		template <typename Predicate>
		bool allofCtorDtor( Statement * stmt, const Predicate & pred ) {
			std::list< Expression * > callExprs;
			collectCtorDtorCalls( stmt, callExprs );
			// if ( callExprs.empty() ) return false; // xxx - do I still need this check?
			return std::all_of( callExprs.begin(), callExprs.end(), pred);
		}
	}

	bool isIntrinsicSingleArgCallStmt( Statement * stmt ) {
		return allofCtorDtor( stmt, []( Expression * callExpr ){
			if ( ApplicationExpr * appExpr = isIntrinsicCallExpr( callExpr ) ) {
				FunctionType *funcType = GenPoly::getFunctionType( appExpr->get_function()->get_result() );
				assert( funcType );
				return funcType->get_parameters().size() == 1;
			}
			return false;
		});
	}

	bool isIntrinsicCallStmt( Statement * stmt ) {
		return allofCtorDtor( stmt, []( Expression * callExpr ) {
			return isIntrinsicCallExpr( callExpr );
		});
	}

	namespace {
		template<typename CallExpr>
		Expression *& callArg( CallExpr * callExpr, unsigned int pos ) {
			if ( pos >= callExpr->get_args().size() ) assertf( false, "asking for argument that doesn't exist. Return NULL/throw exception?" );
			for ( Expression *& arg : callExpr->get_args() ) {
				if ( pos == 0 ) return arg;
				pos--;
			}
			assert( false );
		}
	}

	Expression *& getCallArg( Expression * callExpr, unsigned int pos ) {
		if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( callExpr ) ) {
			return callArg( appExpr, pos );
		} else if ( UntypedExpr * untypedExpr = dynamic_cast< UntypedExpr * >( callExpr ) ) {
			return callArg( untypedExpr, pos );
		} else {
			assertf( false, "Unexpected expression type passed to getCallArg" );
		}
	}

	namespace {
		std::string funcName( Expression * func ) {
			if ( NameExpr * nameExpr = dynamic_cast< NameExpr * >( func ) ) {
				return nameExpr->get_name();
			} else if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( func ) ) {
				return varExpr->get_var()->get_name();
			}	else if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( func ) ) {
				return funcName( castExpr->get_arg() );
			} else if ( MemberExpr * memberExpr = dynamic_cast< MemberExpr * >( func ) ) {
				return memberExpr->get_member()->get_name();
			} else if ( UntypedMemberExpr * memberExpr = dynamic_cast< UntypedMemberExpr * > ( func ) ) {
				return funcName( memberExpr->get_member() );
			} else {
				assertf( false, "Unexpected expression type being called as a function in call expression" );
			}
		}
	}

	std::string getFunctionName( Expression * expr ) {
		if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( expr ) ) {
			return funcName( appExpr->get_function() );
		} else if ( UntypedExpr * untypedExpr = dynamic_cast< UntypedExpr * > ( expr ) ) {
			return funcName( untypedExpr->get_function() );
		} else {
			std::cerr << expr << std::endl;
			assertf( false, "Unexpected expression type passed to getFunctionName" );
		}
	}

	Type * getPointerBase( Type * type ) {
		if ( PointerType * ptrType = dynamic_cast< PointerType * >( type ) ) {
			return ptrType->get_base();
		} else if ( ArrayType * arrayType = dynamic_cast< ArrayType * >( type ) ) {
			return arrayType->get_base();
		} else {
			return NULL;
		}
	}

	Type * isPointerType( Type * type ) {
		if ( getPointerBase( type ) ) return type;
		else return NULL;
	}

	class ConstExprChecker : public Visitor {
	public:
		ConstExprChecker() : isConstExpr( true ) {}

		virtual void visit( ApplicationExpr *applicationExpr ) { isConstExpr = false; }
		virtual void visit( UntypedExpr *untypedExpr ) { isConstExpr = false; }
		virtual void visit( NameExpr *nameExpr ) {
			// xxx - temporary hack, because 0 and 1 really should be constexprs, even though they technically aren't in Cforall today
			if ( nameExpr->get_name() != "0" && nameExpr->get_name() != "1" ) isConstExpr = false;
		}
		// virtual void visit( CastExpr *castExpr ) { isConstExpr = false; }
		virtual void visit( AddressExpr *addressExpr ) {
			// address of a variable or member expression is constexpr
			Expression * arg = addressExpr->get_arg();
			if ( ! dynamic_cast< NameExpr * >( arg) && ! dynamic_cast< VariableExpr * >( arg ) && ! dynamic_cast< MemberExpr * >( arg ) && ! dynamic_cast< UntypedMemberExpr * >( arg ) ) isConstExpr = false;
		}
		virtual void visit( LabelAddressExpr *labAddressExpr ) { isConstExpr = false; }
		virtual void visit( UntypedMemberExpr *memberExpr ) { isConstExpr = false; }
		virtual void visit( MemberExpr *memberExpr ) { isConstExpr = false; }
		virtual void visit( VariableExpr *variableExpr ) { isConstExpr = false; }
		// these might be okay?
		// virtual void visit( SizeofExpr *sizeofExpr );
		// virtual void visit( AlignofExpr *alignofExpr );
		// virtual void visit( UntypedOffsetofExpr *offsetofExpr );
		// virtual void visit( OffsetofExpr *offsetofExpr );
		// virtual void visit( OffsetPackExpr *offsetPackExpr );
		// virtual void visit( AttrExpr *attrExpr );
		// virtual void visit( CommaExpr *commaExpr );
		// virtual void visit( LogicalExpr *logicalExpr );
		// virtual void visit( ConditionalExpr *conditionalExpr );
		virtual void visit( TypeExpr *typeExpr ) { isConstExpr = false; }
		virtual void visit( AsmExpr *asmExpr ) { isConstExpr = false; }
		virtual void visit( UntypedValofExpr *valofExpr ) { isConstExpr = false; }
		virtual void visit( CompoundLiteralExpr *compLitExpr ) { isConstExpr = false; }
		virtual void visit( TupleExpr *tupleExpr ) { isConstExpr = false; }
		virtual void visit( TupleAssignExpr *tupleExpr ) { isConstExpr = false; }

		bool isConstExpr;
	};

	bool isConstExpr( Expression * expr ) {
		if ( expr ) {
			ConstExprChecker checker;
			expr->accept( checker );
			return checker.isConstExpr;
		}
		return true;
	}

	bool isConstExpr( Initializer * init ) {
		if ( init ) {
			ConstExprChecker checker;
			init->accept( checker );
			return checker.isConstExpr;
		} // if
		// for all intents and purposes, no initializer means const expr
		return true;
	}

	bool isConstructor( const std::string & str ) { return str == "?{}"; }
	bool isDestructor( const std::string & str ) { return str == "^?{}"; }
	bool isAssignment( const std::string & str ) { return str == "?=?"; }
	bool isCtorDtor( const std::string & str ) { return isConstructor( str ) || isDestructor( str ); }
	bool isCtorDtorAssign( const std::string & str ) { return isCtorDtor( str ) || isAssignment( str ); }

	FunctionDecl * isCopyFunction( Declaration * decl, const std::string & fname ) {
		FunctionDecl * function = dynamic_cast< FunctionDecl * >( decl );
		if ( ! function ) return 0;
		if ( function->get_name() != fname ) return 0;
		FunctionType * ftype = function->get_functionType();
		if ( ftype->get_parameters().size() != 2 ) return 0;

		Type * t1 = ftype->get_parameters().front()->get_type();
		Type * t2 = ftype->get_parameters().back()->get_type();
		PointerType * ptrType = dynamic_cast< PointerType * > ( t1 );
		assert( ptrType );

		if ( ResolvExpr::typesCompatibleIgnoreQualifiers( ptrType->get_base(), t2, SymTab::Indexer() ) ) {
			return function;
		} else {
			return nullptr;
		}
	}

	FunctionDecl * isCopyConstructor( Declaration * decl ) {
		return isCopyFunction( decl, "?{}" );
	}
}
