//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Lvalue.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Mar 17 09:11:18 2017
// Update Count     : 5
//

#include <cassert>

#include "Lvalue.h"

#include "GenPoly.h"

#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Visitor.h"
#include "SynTree/Mutator.h"
#include "SymTab/Indexer.h"
#include "ResolvExpr/Resolver.h"
#include "ResolvExpr/typeops.h"

#include "Common/UniqueName.h"
#include "Common/utility.h"
#include "InitTweak/InitTweak.h"

#include "Common/PassVisitor.h"

// need to be careful about polymorphic references... e.g. in *? (___operator_deref__A0_1_0_0__Fd0_Pd0_intrinsic___1)
// the variable is automatically dereferenced and this causes errors dereferencing void*.

namespace GenPoly {
	namespace {
		struct ReferenceConversions final {
			Expression * postmutate( CastExpr * castExpr );
		};

		/// Intrinsic functions that take reference parameters don't REALLY take reference parameters -- their reference arguments must always be implicitly dereferenced.
		struct FixIntrinsicArgs final {
			Expression * postmutate( ApplicationExpr *appExpr );
		};


		/// Replace reference types with pointer types
		struct ReferenceTypeElimination final {
			Type * postmutate( ReferenceType * refType );
		};

		/// GCC-like Generalized Lvalues (which have since been removed from GCC)
		/// https://gcc.gnu.org/onlinedocs/gcc-3.4.6/gcc/Lvalues.html#Lvalues
		/// Replaces &(a,b) with (a, &b), &(a ? b : c) with (a ? &b : &c)
		struct GeneralizedLvalue final {
			Expression * postmutate( AddressExpr * addressExpr );
		};
	} // namespace

	void convertLvalue( std::list< Declaration* >& translationUnit ) {
		std::cerr << "convertLvalue" << std::endl;
		PassVisitor<ReferenceConversions> refCvt;
		PassVisitor<ReferenceTypeElimination> elim;
		PassVisitor<GeneralizedLvalue> genLval;
		PassVisitor<FixIntrinsicArgs> fixer;
		mutateAll( translationUnit, refCvt );
		mutateAll( translationUnit, fixer );
		mutateAll( translationUnit, elim );
		mutateAll( translationUnit, genLval );
	}

	namespace {
		Type* isLvalueRet( FunctionType *function ) {
			if ( function->get_returnVals().empty() ) return 0;
			Type *ty = function->get_returnVals().front()->get_type();
			return dynamic_cast< ReferenceType * >( ty ) ;
		}

		bool isIntrinsicApp( ApplicationExpr *appExpr ) {
			if ( VariableExpr *varExpr = dynamic_cast< VariableExpr* >( appExpr->get_function() ) ) {
				return varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic;
			} else {
				return false;
			} // if
		}

		bool isDeref( Expression * expr ) {
			if ( UntypedExpr * untyped = dynamic_cast< UntypedExpr * >( expr ) ) {
				return InitTweak::getFunctionName( untyped ) == "*?";
			} else if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * > ( expr ) ) {
				if ( DeclarationWithType * func = InitTweak::getFunction( appExpr ) ) {
					return func->get_linkage() == LinkageSpec::Intrinsic && InitTweak::getFunctionName( appExpr ) == "*?";
				}
			}
			return false;
		}

		bool isIntrinsicReference( Expression * expr ) {
			if ( isDeref( expr ) ) return true;
			else if ( UntypedExpr * untyped = dynamic_cast< UntypedExpr * >( expr ) ) {
				return InitTweak::getFunctionName( untyped ) == "?[?]";
			} else if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * > ( expr ) ) {
				if ( DeclarationWithType * func = InitTweak::getFunction( appExpr ) ) {
					return func->get_linkage() == LinkageSpec::Intrinsic && InitTweak::getFunctionName( appExpr ) == "?[?]";
				}
			}
			return false;
		}

		void fixArg( Expression *& arg, Type * formal ) {
			// if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( arg ) ) {
				if ( dynamic_cast<ReferenceType*>( formal ) ) { // xxx - but might be deref, in which case result isn't REALLY a reference, at least not in the sense that we need to add another deref...
					// doesn't work, for some reason left arg is skipped in assign
					ReferenceType * refType = safe_dynamic_cast< ReferenceType * >( arg->get_result() ) ;
					std::cerr << "appexpr arg is non-deref/index intrinsic call" << std::endl;
					std::cerr << arg << std::endl;
					PointerType * ptrType = new PointerType( Type::Qualifiers(), refType->get_base()->clone() );
					delete refType;
					arg->set_result( ptrType );
					arg = UntypedExpr::createDeref( arg );
				}

			// }
		}

		Expression * FixIntrinsicArgs::postmutate( ApplicationExpr * appExpr ) {
			if ( DeclarationWithType * function = InitTweak::getFunction( appExpr ) ) {
				if ( function->get_linkage() == LinkageSpec::Intrinsic ) { // intrinsic functions that turn pointers into references
					FunctionType * ftype = GenPoly::getFunctionType( function->get_type() );
					assertf( ftype, "Function declaration does not have function type." );
					for ( auto p : group_iterate( appExpr->get_args(), ftype->get_parameters() ) ) {
						Expression *& arg = std::get<0>( p );
						DeclarationWithType * formal = std::get<1>( p );
						std::cerr << "pair<0>: " << arg << std::endl;
						std::cerr << "pair<1>: " << formal->get_type() << std::endl;
						if ( isIntrinsicReference( arg ) ) {
							std::cerr << "skipping intrinsic reference" << std::endl;
							continue;
						} else {
							fixArg( arg, formal->get_type() );
						}
					}
				}
			}
			return appExpr;
		}

		Expression * ReferenceConversions::postmutate( CastExpr * castExpr ) {
			// conversion to reference type
			if ( ReferenceType * refType = dynamic_cast< ReferenceType * >( castExpr->get_result() ) ) {
				(void)refType;
				if ( ReferenceType * otherRef = dynamic_cast< ReferenceType * >( castExpr->get_arg()->get_result() ) ) {
					// nothing to do if casting from reference to reference.
					(void)otherRef;
					std::cerr << "convert reference to reference -- nop" << std::endl;
					if ( isIntrinsicReference( castExpr->get_arg() ) ) {
						Expression * callExpr = castExpr->get_arg();
						Expression ** arg = nullptr;
						Expression *& arg0 = InitTweak::getCallArg( callExpr, 0 );
						if ( dynamic_cast<PointerType *>( arg0->get_result() ) ) {
							arg = &arg0;
						} else {
							arg = &InitTweak::getCallArg( callExpr, 1 );
						}

						castExpr->set_arg( *arg );
						*arg = castExpr;
						std::cerr << "but arg is deref -- &" << std::endl;
						std::cerr << callExpr << std::endl;
						// castExpr->set_arg( new AddressExpr( castExpr->get_arg() ) );

						// move environment out to new top-level
						callExpr->set_env( castExpr->get_env() );
						castExpr->set_env( nullptr );
						return callExpr;
					}
					std::cerr << castExpr << std::endl;
					return castExpr;
				} else if ( castExpr->get_arg()->get_result()->get_lvalue() ) {
					// conversion from lvalue to reference
					// xxx - keep cast, but turn into pointer cast??
					// xxx - memory
					std::cerr << "convert lvalue to reference -- &" << std::endl;
					std::cerr << castExpr->get_arg() << std::endl;
					castExpr->set_arg( new AddressExpr( castExpr->get_arg() ) );
					// return new AddressExpr( castExpr->get_arg() );
					return castExpr;
				} else {
					// rvalue to reference conversion -- introduce temporary
				}
				assertf( false, "Only conversions to reference from lvalue are currently supported: %s", toString( castExpr ).c_str() );
			} else if ( ReferenceType * refType = dynamic_cast< ReferenceType * >( castExpr->get_arg()->get_result() ) ) {
				// should be easy, just need to move deref code up here?
				std::cerr << "convert reference to rvalue -- *" << std::endl;
				if ( isIntrinsicReference( castExpr->get_arg() ) ) {
					std::cerr << "but arg is intrinsic reference -- nop" << std::endl;
					return castExpr;
				}
				std::cerr << castExpr << std::endl;

				PointerType * ptrType = new PointerType( refType->get_qualifiers(), refType->get_base()->clone() );
				delete castExpr->get_result();
				castExpr->set_result( ptrType );
				Expression * deref = UntypedExpr::createDeref( castExpr );
				deref->set_env( castExpr->get_env() );
				castExpr->set_env( nullptr );
				return deref;
				// assertf( false, "Conversions from reference types are not currently supported." );
			}
			return castExpr;
		}

		Type * ReferenceTypeElimination::postmutate( ReferenceType * refType ) {
			Type * base = refType->get_base();
			refType->set_base( nullptr );
			delete refType;
			return new PointerType( Type::Qualifiers(), base );
		}

		Expression * GeneralizedLvalue::postmutate( AddressExpr * addrExpr ) {
			if ( CommaExpr * commaExpr = dynamic_cast< CommaExpr * >( addrExpr->get_arg() ) ) {
				Expression * arg1 = commaExpr->get_arg1()->clone();
				Expression * arg2 = commaExpr->get_arg2()->clone();
				delete addrExpr;
				return new CommaExpr( arg1, new AddressExpr( arg2 ) );
			} else if ( ConditionalExpr * condExpr = dynamic_cast< ConditionalExpr * >( addrExpr->get_arg() ) ) {
				Expression * arg1 = condExpr->get_arg1()->clone();
				Expression * arg2 = condExpr->get_arg2()->clone();
				Expression * arg3 = condExpr->get_arg3()->clone();
				delete addrExpr;
				return new ConditionalExpr( arg1, new AddressExpr( arg2 ), new AddressExpr( arg3 ) );
			}
			return addrExpr;
		}
	} // namespace
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
