//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// StatementNode.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Sat May 16 14:59:41 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun Aug 14 08:09:31 2016
// Update Count     : 271
//

#include <list>
#include <algorithm>
#include <cassert>

#include "ParseNode.h"
#include "SynTree/Statement.h"
#include "SynTree/Expression.h"
#include "parseutility.h"
#include "Common/utility.h"

using namespace std;

const char *StatementNode::StType[] = {
	"Exp",   "If",       "Switch", "Case",    "Default",  "Choose",   "Fallthru",
	"While", "Do",       "For",
	"Goto",  "Continue", "Break",  "Return",  "Throw",
	"Try",   "Catch",    "Finally", "Asm",
	"Decl"
};

StatementNode::StatementNode() : ParseNode(), control( 0 ), block( 0 ), labels( 0 ), target( 0 ), decl( 0 ), catchAny( false ) {}

StatementNode::StatementNode( const string *name ) : ParseNode( name ), control( 0 ), block( 0 ), labels( 0 ), target( 0 ), decl( 0 ), catchAny( false ) { assert( false ); }

StatementNode::StatementNode( DeclarationNode *decl ) : type( Decl ), control( 0 ), block( 0 ), labels( 0 ), target( 0 ), catchAny( false ) {
	assert( false );
	if ( decl ) {
		if ( DeclarationNode *agg = decl->extractAggregate() ) {
			this->decl = agg;
			StatementNode *nextStmt = new StatementNode;
			nextStmt->type = Decl;
			nextStmt->decl = decl;
			next = nextStmt;
			if ( decl->get_next() ) {
				next->set_next( new StatementNode( dynamic_cast<DeclarationNode *>( decl->get_next() ) ) );
				decl->set_next( 0 );
			} // if
		} else {
			if ( decl->get_next() ) {
				next = new StatementNode( dynamic_cast<DeclarationNode *>( decl->get_next() ) );
				decl->set_next( 0 );
			} // if
			this->decl = decl;
		} // if
	} // if
}

StatementNode2::StatementNode2( DeclarationNode *decl ) {
	if ( decl ) {
		DeclarationNode *agg = decl->extractAggregate();
		if ( agg ) {
			StatementNode *nextStmt = new StatementNode;
			nextStmt->type = Decl;
			nextStmt->decl = decl;
			next = nextStmt;
			if ( decl->get_next() ) {
				next->set_next( new StatementNode2( dynamic_cast<DeclarationNode *>(decl->get_next()) ) );
				decl->set_next( 0 );
			} // if
		} else {
			if ( decl->get_next() ) {
				next = new StatementNode2( dynamic_cast<DeclarationNode *>( decl->get_next() ) );
				decl->set_next( 0 );
			} // if
			agg = decl;
		} // if
		stmt = new DeclStmt( noLabels, maybeBuild<Declaration>(agg) );
	} else {
		assert( false );
	} // if
}

StatementNode::StatementNode( Type t, ExpressionNode *ctrl_label, StatementNode *block ) : type( t ), control( ctrl_label ), block( block ), labels( 0 ), target( 0 ), decl( 0 ), catchAny( false ) {
	this->control = ( t == Default ) ? 0 : control;
}

StatementNode::StatementNode( Type t, string *target ) : type( t ), control( 0 ), block( 0 ), labels( 0 ), target( target ), decl( 0 ), catchAny( false ) {}

StatementNode::~StatementNode() {
	delete control;
	delete block;
	delete target;
	delete decl;
}

StatementNode * StatementNode::clone() const {
	assert( false );
	StatementNode *newnode = new StatementNode( type, maybeClone( control ), maybeClone( block ) );
	if ( target ) {
		newnode->target = new string( *target );
	} else {
		newnode->target = 0;
	} // if
	newnode->decl = maybeClone( decl );
	return newnode;
}

StatementNode *StatementNode::add_label( const std::string *l ) {
	if ( l != 0 ) {
		labels.push_front( *l );
		delete l;
	} // if
	return this;
}

StatementNode *StatementNode::append_block( StatementNode *stmt ) {
	if ( stmt != 0 ) {
		if ( block == 0 )
			block = stmt;
		else
			block->set_last( stmt );
	} // if
	return this;
}

StatementNode *StatementNode::append_last_case( StatementNode *stmt ) {
	assert( false );
	if ( stmt != 0 ) {
		StatementNode *next = ( StatementNode *)get_next();
		if ( next && ( next->get_type() == StatementNode::Case || next->get_type() == StatementNode::Default ) )
			next->append_last_case( stmt );
		else
			if ( block == 0 )
				block = stmt;
			else
				block->set_last( stmt );
	} // if
	return this;
}

StatementNode *StatementNode2::append_last_case( StatementNode *stmt ) {
	StatementNode *prev = this;
	// find end of list and maintain previous pointer
	for ( StatementNode * curr = prev; curr != nullptr; curr = (StatementNode *)curr->get_next() ) {
		StatementNode2 *node = dynamic_cast<StatementNode2 *>(curr);
		assert( node );
		assert( dynamic_cast<CaseStmt *>(node->stmt) );
		prev = curr;
	} // for
	// conver from StatementNode list to Statement list
	StatementNode2 *node = dynamic_cast<StatementNode2 *>(prev);
	std::list<Statement *> stmts;
	buildList( stmt, stmts );
	// splice any new Statements to end of currents Statements
	CaseStmt * caseStmt = dynamic_cast<CaseStmt *>(node->stmt);
	caseStmt->get_statements().splice( caseStmt->get_statements().end(), stmts );
	return this;
}

void StatementNode::print( std::ostream &os, int indent ) const {
	if ( ! labels.empty() ) {
		std::list<std::string>::const_iterator i;

		os << string( indent, ' ' );
		for ( i = labels.begin(); i != labels.end(); i++ )
			os << *i << ":";
		os << endl;
	} // if

	switch ( type ) {
	  case Decl:
		decl->print( os, indent );
		break;
	  case Exp:
		if ( control ) {
			os << string( indent, ' ' );
			control->print( os, indent );
			os << endl;
		} else
			os << string( indent, ' ' ) << "Null Statement" << endl;
		break;
	  default:
		os << string( indent, ' ' ) << StatementNode::StType[type] << endl;
		if ( type == Catch ) {
			if ( decl ) {
				os << string( indent + ParseNode::indent_by, ' ' ) << "Declaration: " << endl;
				decl->print( os, indent + 2 * ParseNode::indent_by );
			} else if ( catchAny ) {
				os << string( indent + ParseNode::indent_by, ' ' ) << "Catches the rest " << endl;
			} else {
				; // should never reach here
			} // if
		} // if
		if ( control ) {
			os << string( indent + ParseNode::indent_by, ' ' ) << "Control: " << endl;
			control->printList( os, indent + 2 * ParseNode::indent_by );
		} // if
		if ( block ) {
			os << string( indent + ParseNode::indent_by, ' ' ) << "Cases: " << endl;
			block->printList( os, indent + 2 * ParseNode::indent_by );
		} // if
		break;
	} // switch
}

Statement *StatementNode::build() const {
	std::list<Statement *> branches;
	std::list<Expression *> exps;
	std::list<Label> labs;

	if ( ! labels.empty() ) {
		std::back_insert_iterator< std::list<Label> > lab_it( labs );
		copy( labels.begin(), labels.end(), lab_it );
	} // if

	buildList<Statement, StatementNode>( get_block(), branches );

	switch ( type ) {
	  case Decl:
		return new DeclStmt( labs, maybeBuild< Declaration >( decl ) );
		assert( false );
	  case Exp:
		// {
		// 	Expression *e = maybeBuild< Expression >( get_control() );

		// 	if ( e )
		// 		return new ExprStmt( labs, e );
		// 	else
		// 		return new NullStmt( labs );
		// }
		assert( false );
	  case If:
		// {
		// 	Statement *thenb = 0, *elseb = 0;
		// 	assert( branches.size() >= 1 );

		// 	thenb = branches.front();
		// 	branches.pop_front();
		// 	if ( ! branches.empty() ) {
		// 		elseb = branches.front();
		// 		branches.pop_front();
		// 	} // if
		// 	return new IfStmt( labs, notZeroExpr( maybeBuild<Expression>(get_control()) ), thenb, elseb );
		// }
		assert( false );
	  case Switch:
		// return new SwitchStmt( labs, maybeBuild<Expression>(get_control()), branches );
		assert( false );
	  case Case:
		//return new CaseStmt( labs, maybeBuild<Expression>(get_control() ), branches );
		assert( false );
	  case Default:
		//return new CaseStmt( labs, 0, branches, true );
		assert( false );
	  case While:
		// assert( branches.size() == 1 );
		// return new WhileStmt( labs, notZeroExpr( maybeBuild<Expression>(get_control()) ), branches.front() );
		assert( false );
	  case Do:
		// assert( branches.size() == 1 );
		// return new WhileStmt( labs, notZeroExpr( maybeBuild<Expression>(get_control()) ), branches.front(), true );
		assert( false );
	  case For:
	  	// {
	  	// 	assert( branches.size() == 1 );

	  	// 	ForCtlExprNode *ctl = dynamic_cast<ForCtlExprNode *>( get_control() );
	  	// 	assert( ctl != 0 );

	  	// 	std::list<Statement *> init;
	  	// 	if ( ctl->get_init() != 0 ) {
	  	// 		buildList( ctl->get_init(), init );
	  	// 	} // if

	  	// 	Expression *cond = 0;
	  	// 	if ( ctl->get_condition() != 0 )
	  	// 		cond = notZeroExpr( maybeBuild<Expression>(ctl->get_condition()) );

	  	// 	Expression *incr = 0;
	  	// 	if ( ctl->get_change() != 0 )
	  	// 		incr = maybeBuild<Expression>(ctl->get_change());

	  	// 	return new ForStmt( labs, init, cond, incr, branches.front() );
	  	// }
		assert( false );
	  case Goto:
		assert( false );
	  case Break:
		assert( false );
	  case Continue:
		assert( false );
	  case Return:
	  case Throw :
		// buildList( get_control(), exps );
		// if ( exps.size() ==0 )
		// 	return new ReturnStmt( labs, 0, type == Throw );
		// if ( exps.size() > 0 )
		// 	return new ReturnStmt( labs, exps.back(), type == Throw );
		assert( false );
	  case Try:
		// {
		// 	assert( branches.size() >= 0 );
		// 	CompoundStmt *tryBlock = dynamic_cast<CompoundStmt *>( branches.front());
		// 	branches.pop_front();
		// 	FinallyStmt *finallyBlock = 0;
		// 	if ( ( finallyBlock = dynamic_cast<FinallyStmt *>( branches.back())) ) {
		// 		branches.pop_back();
		// 	} // if
		// 	return new TryStmt( labs, tryBlock, branches, finallyBlock );
		// }
		assert( false );
	  case Catch:
		// {
		// 	assert( branches.size() == 1 );

		// 	return new CatchStmt( labs, maybeBuild< Declaration >( decl ), branches.front(), catchAny );
		// }
		assert( false );
	  case Finally:
		// {
		// 	assert( branches.size() == 1 );
		// 	CompoundStmt *block = dynamic_cast<CompoundStmt *>( branches.front() );
		// 	assert( block != 0 );

		// 	return new FinallyStmt( labs, block );
		// }
		assert( false );
	  case Asm:
		assert( false );
	  default:
		assert( false );
		return 0;
	} // switch
}

Statement *build_expr( ExpressionNode *ctl ) {
	Expression *e = maybeBuild< Expression >( ctl );

	if ( e )
		return new ExprStmt( noLabels, e );
	else
		return new NullStmt( noLabels );
}

Statement *build_if( ExpressionNode *ctl, StatementNode *then_stmt, StatementNode *else_stmt ) {
	Statement *thenb, *elseb = 0;
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( then_stmt, branches );
	assert( branches.size() == 1 );
	thenb = branches.front();

	if ( else_stmt ) {
		std::list<Statement *> branches;
		buildList<Statement, StatementNode>( else_stmt, branches );
		assert( branches.size() == 1 );
		elseb = branches.front();
	} // if
	return new IfStmt( noLabels, notZeroExpr( maybeBuild<Expression>(ctl) ), thenb, elseb );
}

Statement *build_switch( ExpressionNode *ctl, StatementNode *stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() >= 0 );						// size == 0 for switch (...) {}, i.e., no declaration or statements
	return new SwitchStmt( noLabels, maybeBuild<Expression>(ctl), branches );
}
Statement *build_case( ExpressionNode *ctl ) {
	std::list<Statement *> branches;
	return new CaseStmt( noLabels, maybeBuild<Expression>(ctl), branches );
}
Statement *build_default() {
	std::list<Statement *> branches;
	return new CaseStmt( noLabels, nullptr, branches, true );
}

Statement *build_while( ExpressionNode *ctl, StatementNode *stmt, bool kind ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );
	return new WhileStmt( noLabels, notZeroExpr( maybeBuild<Expression>(ctl) ), branches.front(), kind );
}

Statement *build_for( ForCtl *forctl, StatementNode *stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );

	std::list<Statement *> init;
	if ( forctl->init != 0 ) {
		buildList( forctl->init, init );
	} // if

	Expression *cond = 0;
	if ( forctl->condition != 0 )
		cond = notZeroExpr( maybeBuild<Expression>(forctl->condition) );

	Expression *incr = 0;
	if ( forctl->change != 0 )
		incr = maybeBuild<Expression>(forctl->change);

	delete forctl;
	return new ForStmt( noLabels, init, cond, incr, branches.front() );
}

Statement *build_branch( std::string identifier, BranchStmt::Type kind ) {
	return new BranchStmt( noLabels, identifier, kind );
}
Statement *build_computedgoto( ExpressionNode *ctl ) {
	return new BranchStmt( noLabels, maybeBuild<Expression>(ctl), BranchStmt::Goto );
}

Statement *build_return( ExpressionNode *ctl ) {
	std::list<Expression *> exps;
	buildList( ctl, exps );
	return new ReturnStmt( noLabels, exps.size() > 0 ? exps.back() : nullptr );
}
Statement *build_throw( ExpressionNode *ctl ) {
	std::list<Expression *> exps;
	buildList( ctl, exps );
	return new ReturnStmt( noLabels, exps.size() > 0 ? exps.back() : nullptr, true );
}

Statement *build_try( StatementNode *try_stmt, StatementNode *catch_stmt, StatementNode *finally_stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( catch_stmt, branches );
	CompoundStmt *tryBlock = dynamic_cast<CompoundStmt *>(maybeBuild<Statement>(try_stmt));
	assert( tryBlock );
	FinallyStmt *finallyBlock = dynamic_cast<FinallyStmt *>(maybeBuild<Statement>(finally_stmt) );
	return new TryStmt( noLabels, tryBlock, branches, finallyBlock );
}
Statement *build_catch( DeclarationNode *decl, StatementNode *stmt, bool catchAny ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );
	return new CatchStmt( noLabels, maybeBuild<Declaration>(decl), branches.front(), catchAny );
}
Statement *build_finally( StatementNode *stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );
	return new FinallyStmt( noLabels, dynamic_cast<CompoundStmt *>( branches.front() ) );
}


CompoundStmtNode::CompoundStmtNode() : first( 0 ), last( 0 ) {}

CompoundStmtNode::CompoundStmtNode( StatementNode *stmt ) : first( stmt ) {
	if ( first ) {
		last = ( StatementNode *)( stmt->get_last());
	} else {
		last = 0;
	} // if
}

CompoundStmtNode::~CompoundStmtNode() {
	delete first;
}

void CompoundStmtNode::add_statement( StatementNode *stmt ) {
	if ( stmt != 0 ) {
		last->set_last( stmt );
		last = ( StatementNode *)( stmt->get_next());
	} // if
}

void CompoundStmtNode::print( ostream &os, int indent ) const {
	if ( first ) {
		first->printList( os, indent+2 );
	} // if
}

Statement *CompoundStmtNode::build() const {
	std::list<Label> labs;
	const std::list<std::string> &labels = get_labels();

	if ( ! labels.empty() ) {
		std::back_insert_iterator< std::list<Label> > lab_it( labs );
		copy( labels.begin(), labels.end(), lab_it );
	} // if

	CompoundStmt *cs = new CompoundStmt( labs );
	buildList( first, cs->get_kids() );
	return cs;
}


AsmStmtNode::AsmStmtNode( Type t, bool voltile, ConstantExpr *instruction, ExpressionNode *output, ExpressionNode *input, ExpressionNode *clobber, LabelNode *gotolabels ) :
	StatementNode( t ), voltile( voltile ), instruction( instruction ), output( output ), input( input ), clobber( clobber ) {
	if ( gotolabels ) {
		this->gotolabels = gotolabels->get_labels();
		delete gotolabels;
	} // if
}

AsmStmtNode::~AsmStmtNode() {
	delete output; delete input; delete clobber;
}

void AsmStmtNode::print( std::ostream &os, int indent ) const {
	StatementNode::print( os, indent );					// print statement labels
	os << string( indent + ParseNode::indent_by, ' ' ) << "volatile:" << voltile << endl;
	if ( instruction ) {
		os << string( indent + ParseNode::indent_by, ' ' ) << "Instruction:" << endl;
//		instruction->printList( os, indent + 2 * ParseNode::indent_by );
	} // if
	if ( output ) {
		os << string( indent + ParseNode::indent_by, ' ' ) << "Output:" << endl;
		output->printList( os, indent + 2 * ParseNode::indent_by );
	} // if
	if ( input ) {
		os << string( indent + ParseNode::indent_by, ' ' ) << "Input:" << endl;
		input->printList( os, indent + 2 * ParseNode::indent_by );
	} // if
	if ( clobber ) {
		os << string( indent + ParseNode::indent_by, ' ' ) << "Clobber:" << endl;
//		clobber->printList( os, indent + 2 * ParseNode::indent_by );
	} // if
	if ( ! gotolabels.empty() ) {
		os << string( indent + ParseNode::indent_by, ' ' ) << "Goto Labels:" << endl;
		os << string( indent + 2 * ParseNode::indent_by, ' ' );
		for ( std::list<Label>::const_iterator i = gotolabels.begin();; ) {
			os << *i;
			i++;
		  if ( i == gotolabels.end() ) break;
			os << ", ";
		}
		os << endl;
	} // if
}

Statement *AsmStmtNode::build() const {
	std::list<Label> labs;

	if ( ! get_labels().empty() ) {
		std::back_insert_iterator< std::list<Label> > lab_it( labs );
		copy( get_labels().begin(), get_labels().end(), lab_it );
	} // if

	std::list< Expression * > out, in;
	std::list< ConstantExpr * > clob;
	buildList( output, out );
	buildList( input, in );
	buildList( clobber, clob );
	std::list< Label > gotolabs = gotolabels;
	return new AsmStmt( labs, voltile, instruction, out, in, clob, gotolabs );
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
