//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Statement.h -- 
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Jul 14 12:14:54 2015
// Update Count     : 24
//

#ifndef STATEMENT_H
#define STATEMENT_H

#include "SynTree.h"
#include "Visitor.h"
#include "Mutator.h"
#include "Common/SemanticError.h"

class Statement {
  public:
	Statement( std::list<Label> labels );
	virtual ~Statement();

	std::list<Label> & get_labels() { return labels; }
	const std::list<Label> & get_labels() const { return labels; }

	virtual Statement *clone() const = 0;
	virtual void accept( Visitor &v ) = 0;
	virtual Statement *acceptMutator( Mutator &m ) = 0;
	virtual void print( std::ostream &os, int indent = 0 ) const;
  protected:
	std::list<Label> labels;
};

class CompoundStmt : public Statement {
  public:
	CompoundStmt( std::list<Label> labels );
	CompoundStmt( const CompoundStmt &other );
	virtual ~CompoundStmt();

	std::list<Statement*>& get_kids() { return kids; }

	virtual CompoundStmt *clone() const { return new CompoundStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual CompoundStmt *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	std::list<Statement*> kids;
};

class ExprStmt : public Statement {
  public:
	ExprStmt( std::list<Label> labels, Expression *expr );
	virtual ~ExprStmt();

	Expression *get_expr() { return expr; }
	void set_expr( Expression *newValue ) { expr = newValue; }

	virtual ExprStmt *clone() const { return new ExprStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression *expr;
};

class IfStmt : public Statement {
  public:
	IfStmt( std::list<Label> labels, Expression *condition, Statement *thenPart, Statement *elsePart );
	virtual ~IfStmt();

	Expression *get_condition() { return condition; }
	void set_condition( Expression *newValue ) { condition = newValue; }
	Statement *get_thenPart() { return thenPart; }
	void set_thenPart( Statement *newValue ) { thenPart = newValue; }
	Statement *get_elsePart() { return elsePart; }
	void set_elsePart( Statement *newValue ) { elsePart = newValue; }
	
	virtual IfStmt *clone() const { return new IfStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression *condition;
	Statement *thenPart;
	Statement *elsePart;
};

class SwitchStmt : public Statement {
  public:
	SwitchStmt( std::list<Label> labels, Expression *condition, std::list<Statement *> &branches );
	virtual ~SwitchStmt();

	Expression *get_condition() { return condition; }
	void set_condition( Expression *newValue ) { condition = newValue; }

	std::list<Statement *>& get_branches() { return branches; }
	void add_case( CaseStmt * );

	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }

	virtual SwitchStmt *clone() const { return new SwitchStmt( *this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression * condition;
	std::list<Statement *> branches; // should be list of CaseStmt
};

class ChooseStmt : public Statement {
  public:
	ChooseStmt( std::list<Label> labels, Expression *condition, std::list<Statement *> &branches );
	virtual ~ChooseStmt();

	Expression *get_condition() { return condition; }
	void set_condition( Expression *newValue ) { condition = newValue; }

	std::list<Statement *>& get_branches() { return branches; }
	void add_case( CaseStmt * );

	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }

	virtual ChooseStmt *clone() const { return new ChooseStmt( *this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression *condition;
	std::list<Statement *> branches; // should be list of CaseStmt
};

class FallthruStmt : public Statement {
  public:
	FallthruStmt( std::list<Label> labels ) : Statement( labels ) { }

	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }

	virtual FallthruStmt *clone() const { return new FallthruStmt( *this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
};

class CaseStmt : public Statement {
  public:
	CaseStmt( std::list<Label> labels, Expression *conditions, 
	      std::list<Statement *> &stmts, bool isdef = false ) throw(SemanticError);
	virtual ~CaseStmt();

	static CaseStmt * makeDefault( std::list<Label> labels = std::list<Label>(),
		std::list<Statement *> stmts = std::list<Statement *>() );

	bool isDefault() const { return _isDefault; }
	void set_default(bool b) { _isDefault = b; }

	Expression * &get_condition() { return condition; }
	void set_condition( Expression *newValue ) { condition = newValue; }

	std::list<Statement *> &get_statements() { return stmts; }
	void set_statements( std::list<Statement *> &newValue ) { stmts = newValue; }
	
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }

	virtual CaseStmt *clone() const { return new CaseStmt( *this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression * condition;
	std::list<Statement *> stmts;
	bool _isDefault;
};

class WhileStmt : public Statement {
  public:
	WhileStmt( std::list<Label> labels, Expression *condition,
	       Statement *body, bool isDoWhile = false );
	virtual ~WhileStmt();

	Expression *get_condition() { return condition; }
	void set_condition( Expression *newValue ) { condition = newValue; }
	Statement *get_body() { return body; }
	void set_body( Statement *newValue ) { body = newValue; }
	bool get_isDoWhile() { return isDoWhile; }
	void set_isDoWhile( bool newValue ) { isDoWhile = newValue; }
	
	virtual WhileStmt *clone() const { return new WhileStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression *condition;
	Statement *body;
	bool isDoWhile;
};

class ForStmt : public Statement {
  public:
	ForStmt( std::list<Label> labels, std::list<Statement *> initialization,
	     Expression *condition = 0, Expression *increment = 0, Statement *body = 0 );
	virtual ~ForStmt();

	std::list<Statement *> &get_initialization() { return initialization; }
	void set_initialization( std::list<Statement *> newValue ) { initialization = newValue; }
	Expression *get_condition() { return condition; }
	void set_condition( Expression *newValue ) { condition = newValue; }
	Expression *get_increment() { return increment; }
	void set_increment( Expression *newValue ) { increment = newValue; }
	Statement *get_body() { return body; }
	void set_body( Statement *newValue ) { body = newValue; }
	
	virtual ForStmt *clone() const { return new ForStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	std::list<Statement *> initialization;
	Expression *condition;
	Expression *increment;
	Statement *body;
};

class BranchStmt : public Statement {
  public:
	enum Type { Goto = 0, Break, Continue };

	BranchStmt( std::list<Label> labels, Label target, Type ) throw (SemanticError);
	BranchStmt( std::list<Label> labels, Expression *computedTarget, Type ) throw (SemanticError);
	virtual ~BranchStmt() {}

	Label get_originalTarget() { return originalTarget; }
	Label get_target() { return target; }
	void set_target( Label newValue ) { target = newValue; }
	
	Expression *get_computedTarget() { return computedTarget; }
	void set_target( Expression * newValue ) { computedTarget = newValue; }

	Type get_type() { return type; }
	const char *get_typename() { return brType[ type ]; }

	virtual BranchStmt *clone() const { return new BranchStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	static const char *brType[];
	Label originalTarget;  // can give better error messages if we remember the label name that the user entered
	Label target;
	Expression *computedTarget;
	Type type;
};

class ReturnStmt : public Statement {
  public:
	ReturnStmt( std::list<Label> labels, Expression *expr, bool throwP = false );
	virtual ~ReturnStmt();

	Expression *get_expr() { return expr; }
	void set_expr( Expression *newValue ) { expr = newValue; }
	
	virtual ReturnStmt *clone() const { return new ReturnStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Expression *expr;
	bool isThrow;
};


class NullStmt : public CompoundStmt {
  public:
	NullStmt();
	NullStmt( std::list<Label> labels );
	virtual ~NullStmt();

	virtual NullStmt *clone() const { return new NullStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual NullStmt *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
	
  private:
};

class TryStmt : public Statement { 
  public:
	TryStmt( std::list<Label> labels, CompoundStmt *tryBlock, std::list<Statement *> &handlers, FinallyStmt *finallyBlock = 0 );
	TryStmt( const TryStmt &other );
	virtual ~TryStmt();

	CompoundStmt *get_block() const { return block; }
	void set_block( CompoundStmt *newValue ) { block = newValue; }
	std::list<Statement *>& get_catchers() { return handlers; }

	FinallyStmt *get_finally() const { return finallyBlock; }
	void set_finally( FinallyStmt *newValue ) { finallyBlock = newValue; }

	virtual TryStmt *clone() const { return new TryStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
	
  private:
	CompoundStmt *block;
	std::list<Statement *> handlers;
	FinallyStmt *finallyBlock;
}; 

class CatchStmt : public Statement {
  public:
	CatchStmt( std::list<Label> labels, Declaration *decl, Statement *body, bool isCatchRest = false );
	virtual ~CatchStmt();

	Declaration *get_decl() { return decl; }
	void set_decl( Declaration *newValue ) { decl = newValue; }

	Statement *get_body() { return body; }
	void set_body( Statement *newValue ) { body = newValue; }
	
	virtual CatchStmt *clone() const { return new CatchStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
	
  private:
	Declaration *decl;
	Statement *body;
	bool catchRest;
};

class FinallyStmt : public Statement { 
  public:
	FinallyStmt( std::list<Label> labels, CompoundStmt *block );
	virtual ~FinallyStmt();

	CompoundStmt *get_block() const { return block; }
	void set_block( CompoundStmt *newValue ) { block = newValue; }
	
	virtual FinallyStmt *clone() const { return new FinallyStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	CompoundStmt *block;
}; 


// represents a declaration that occurs as part of a compound statement
class DeclStmt : public Statement {
  public:
	DeclStmt( std::list<Label> labels, Declaration *decl );
	DeclStmt( const DeclStmt &other );
	virtual ~DeclStmt();

	Declaration *get_decl() { return decl; }
	void set_decl( Declaration *newValue ) { decl = newValue; }

	virtual DeclStmt *clone() const { return new DeclStmt( *this ); }
	virtual void accept( Visitor &v ) { v.visit( this ); }
	virtual Statement *acceptMutator( Mutator &m ) { return m.mutate( this ); }
	virtual void print( std::ostream &os, int indent = 0 ) const;
  private:
	Declaration *decl;
};

#endif // STATEMENT_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
