//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TypeEnvironment.cc --
//
// Author           : Aaron B. Moss
// Created On       : Sun May 17 12:19:47 2015
// Last Modified By : Aaron B. Moss
// Last Modified On : Fri Jun 29 15:51:00 2018
// Update Count     : 5
//

#include <cassert>                     // for assert
#include <algorithm>                   // for copy, set_intersection
#include <iterator>                    // for ostream_iterator, insert_iterator
#include <utility>                     // for pair, move

#include "Common/InternedString.h"     // for interned_string
#include "Common/PassVisitor.h"        // for PassVisitor<GcTracer>
#include "Common/option.h"             // for option<T>
#include "Common/utility.h"            // for maybeClone
#include "SymTab/Indexer.h"            // for Indexer
#include "SynTree/GcTracer.h"          // for PassVisitor<GcTracer>
#include "SynTree/Type.h"              // for Type, FunctionType, Type::Fora...
#include "SynTree/TypeSubstitution.h"  // for TypeSubstitution
#include "TypeEnvironment.h"
#include "typeops.h"                   // for occurs
#include "Unify.h"                     // for unifyInexact

namespace ResolvExpr {
	void printAssertionSet( const AssertionSet &assertions, std::ostream &os, int indent ) {
		for ( AssertionSet::const_iterator i = assertions.begin(); i != assertions.end(); ++i ) {
			i->first->print( os, indent );
			if ( i->second.isUsed ) {
				os << "(used)";
			} else {
				os << "(not used)";
			} // if
		} // for
	}

	void printOpenVarSet( const OpenVarSet &openVars, std::ostream &os, int indent ) {
		os << std::string( indent, ' ' );
		for ( OpenVarSet::const_iterator i = openVars.begin(); i != openVars.end(); ++i ) {
			os << i->first << "(" << i->second << ") ";
		} // for
	}

	std::pair<interned_string, interned_string> TypeEnvironment::mergeClasses( 
			interned_string root1, interned_string root2 ) {
		// merge classes
		Classes* newClasses = classes->merge( root1, root2 );

		// determine new root
		assertf(classes->get_mode() == Classes::REMFROM, "classes updated to REMFROM by merge");
		auto ret = std::make_pair( classes->get_root(), classes->get_child() );
		
		// finalize classes
		classes = newClasses;
		return ret;
	}

	ClassRef TypeEnvironment::lookup( interned_string var ) const {
		interned_string root = classes->find_or_default( var, nullptr );
		if ( root ) return { this, root };
		else return { nullptr, var };
	}

	bool isFtype( Type *type ) {
		if ( dynamic_cast< FunctionType* >( type ) ) {
			return true;
		} else if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( type ) ) {
			return typeInst->get_isFtype();
		} // if
		return false;
	}

	bool tyVarCompatible( const TypeDecl::Data & data, Type *type ) {
		switch ( data.kind ) {
		  case TypeDecl::Dtype:
			// to bind to an object type variable, the type must not be a function type.
			// if the type variable is specified to be a complete type then the incoming
			// type must also be complete
			// xxx - should this also check that type is not a tuple type and that it's not a ttype?
			return ! isFtype( type ) && (! data.isComplete || type->isComplete() );
		  case TypeDecl::Ftype:
			return isFtype( type );
		  case TypeDecl::Ttype:
			// ttype unifies with any tuple type
			return dynamic_cast< TupleType * >( type ) || Tuples::isTtype( type );
		} // switch
		return false;
	}

	bool TypeEnvironment::bindVar( TypeInstType* typeInst, Type* bindTo, 
			const TypeDecl::Data& data, AssertionSet& need, AssertionSet& have, 
			const OpenVarSet& openVars, WidenMode widenMode, const SymTab::Indexer& indexer ) {
		// remove references from other, so that type variables can only bind to value types
		bindTo = bindTo->stripReferences();
		
		auto tyVar = openVars.find( typeInst->get_name() );
		assert( tyVar != openVars.end() );
		if ( ! tyVarCompatible( tyVar->second, other ) ) return false;

		if ( occurs( bindTo, typeInst->get_name(), *this ) ) return false;

		if ( ClassRef curClass = lookup( typeInst->get_name() ) ) {
			BoundType curData = curClass.get_bound();
			if ( curData.type ) {
				Type* common = nullptr;
				// attempt to unify equivalence class type (which needs its qualifiers restored) 
				// with the target type
				Type* newType = curData.type->clone();
				newType->get_qualifiers() = typeInst->get_qualifiers();
				if ( unifyInexact( newType, bindTo, *this, need, have, openVars, 
						widenMode & WidenMode{ curData.allowWidening, true }, indexer, common ) ) {
					if ( common ) {
						// update bound type to common type
						common->get_qualifiers() = Type::Qualifiers{};
						curData.type = common;
						bindings = bindings->set( curClass.update_root(), curData );
					}
					return true;
				} else return false;
			} else {
				// update bound type to other type
				curData.type = bindTo->clone();
				curData.type->get_qualifiers() = Type::Qualifiers{};
				curData.allowWidening = widenMode.widenFirst && widenMode.widenSecond;
				bindings = bindings->set( curClass.get_root(), curData );
			}
		} else {
			// make new class consisting entirely of this variable
			BoundType curData{ bindTo->clone(), widenMode.first && widenMode.second, data };
			curData.type->get_qualifiers() = Type::Qualifiers{};
			classes = classes->add( curClass.get_root() );
			bindings = bindings->set( curClass.get_root(), curData );
		}
		return true;
	}
	
	bool TypeEnvironment::bindVarToVar( TypeInstType* var1, TypeInstType* var2, 
			const TypeDecl::Data& data, AssertionSet& need, AssertionSet& have, 
			const OpenVarSet& openVars, WidenMode widenMode, const SymTab::Indexer& indexer ) {
		ClassRef class1 = env.lookup( var1->get_name() );
		ClassRef class2 = env.lookup( var2->get_name() );
		
		// exit early if variables already bound together
		if ( class1 && class2 && class1 == class2 ) {
			BoundType data1 = class1.get_bound();
			// narrow the binding if needed
			if ( data1.allowWidening && widenMode.first != widenMode.second ) {
				data1.allowWidening = false;
				bindings = bindings->set( class1.get_root(), data1 );
			}
			return true;
		}

		BoundType data1 = class1 ? class1.get_bound() : BoundType{};
		BoundType data2 = class2 ? class2.get_bound() : BoundType{};

		bool widen1 = data1.allowWidening && widenMode.widenFirst;
		bool widen2 = data2.allowWidening && widenMode.widenSecond;

		if ( data1.type && data2.type ) {
			// attempt to unify bound classes
			Type* common = nullptr;
			if ( unifyInexact( data1.type->clone(), data2.type->clone(), *this, need, have, 
					openVars, WidenMode{ widen1, widen2 }, indexer, common ) ) {
				// merge type variables
				interned_string root = mergeClasses( class1.update_root(), class2.update_root() );
				// update bindings
				data1.allowWidening = widen1 && widen2;
				if ( common ) {
					common->get_qualifiers() = Type::Qualifiers{};
					data1.type = common;
				}
				bindings = bindings->set( root, data1 );
			} else return false;
		} else if ( class1 && class2 ) {
			// both classes exist, only one bound -- merge type variables
			auto merged = mergeClasses( class1.get_root(), class2.get_root() );
			// remove subordinate binding
			bindings = bindings->erase( merged.second );
			// update root binding (narrow widening as needed, or set new binding for changed root)
			if ( data1.type ) {
				if ( data1.allowWidening != widen1 ) {
					data1.allowWidening = widen1;
					bindings = bindings->set( merged.first, data1 );
				} else if ( merged.first == class2.get_root() ) {
					bindings = bindings->set( merged.first, data1 );
				}
			} else /* if ( data2.type ) */ {
				if ( data2.allowWidening != widen2 ) {
					data2.allowWidening = widen2;
					bindings = bindings->set( root, data2 );
				} else if ( merged.first == class1.get_root() ) {
					bindings = bindings->set( merged.first, data2 );
				}
			}
		} else if ( class1 ) {
			// add unbound var2 to class1
			classes = classes->add( class2.get_root() );
			auto merged = mergeClasses( class1.get_root(), class2.get_root() );
			// update bindings (narrow as needed, or switch binding to root)
			if ( merged.first == class2.get_root() ) {
				data1.allowWidening = widen1;
				bindings = bindings->erase( merged.second )->set( merged.first, data1 );
			} else if ( data1.allowWidening != widen1 ) {
				bindings = bindings->set( merged.first, data1 );
			}
		} else if ( class2 ) {
			// add unbound var1 to class1
			classes = classes->add( class1.get_root() );
			auto merged = mergeClasses( class1.get_root(), class2.get_root() );
			// update bindings (narrow as needed, or switch binding to root)
			if ( merged.first == class1.get_root() ) {
				data2.allowWidening = widen2;
				bindings = bindings->erase( merged.second )->set( merged.first, data2 );
			} else if ( data2.allowWidening != widen2 ) {
				bindings = bindings->set( merged.first, data2 );
			}
		} else {
			// make new class with pair of unbound vars
			classes = classes->add( class1.get_root() )->add( class2.get_root() );
			auto merged = mergeClasses( class1.get_root(), class2.get_root() );
			bindings = bindings->set( merged.first, BoundType{ nullptr, widen1 && widen2, data } );
		}
		return true;
	}

	void TypeEnvironment::add( const Type::ForallList &tyDecls ) {
		for ( Type::ForallList::const_iterator i = tyDecls.begin(); i != tyDecls.end(); ++i ) {
			interned_string name = (*i)->get_name();
			classes = classes->add( name );
			bindings = bindings->set( name, *i );
		} // for
	}

	void TypeEnvironment::add( const TypeSubstitution & sub ) {
		interned_string not_found{nullptr};

		for ( auto p : sub ) {
			interned_string var = p.first;
			
			// filter overlapping classes out of existing environment
			// (this is a very shady assumption, but has worked for a long time...)
			interned_string root = classes->find_or_default( v, not_found );
			if ( root != not_found ) {
				classes = classes->remove_class( root );
				bindings = bindings->erase( root );
			}

			// Minimal assumptions. Not technically correct, but might be good enough, and
			// is the best we can do at the moment since information is lost in the
			// transition to TypeSubstitution
			TypeDecl::Data data{ TypeDecl::Dtype, false };

			// add variable to class and bindings
			classes = classes->add( var );
			bindings = bindings->set( var, BoundType{ p.second->clone, false, data } );
		}
	}

	void TypeEnvironment::makeSubstitution( TypeSubstitution &sub ) const {
		bindings->apply_to_all([classes, &sub]( interned_string root, const BoundType& bound ){
			classes->apply_to_class(root, [&]( interned_string var ) {
				if ( bound.type ) {
					sub.add( var, bound.type );
				} else if ( var != root ) {
					sub.add( var, new TypeInstType{ 
						Type::Qualifiers{}, root, bound.data.kind == TypeDecl::Ftype } );
				}
			});
		});

		sub.normalize();
	}

	void TypeEnvironment::print( std::ostream &os, Indenter indent ) const {
		bindings->apply_to_all([classes,&]( interned_string root, const BoundType& bound ) {
			os << "( ";
			classes->apply_to_class( root, [&os]( interned_string var ) { os << var << " "; } );
			os << ")";
			if ( bound.type ) {
				os << " -> ";
				type->print( os, indent+1 );
			}
			if ( ! bound.allowWidening ) {
				os << " (no widening)";
			}
			os << std::endl;
		});
	}

	void TypeEnvironment::simpleCombine( const TypeEnvironment &o ) {
		o.bindings->apply_to_all( [&]( interned_string root, const BoundType& bound ) {
			// add members of new class
			interned_string new_root{nullptr};
			o.classes->apply_to_class( root, [this,&new_root]( interned_string var ) {
				classes = classes->add( var );
				new_root = new_root ? mergeClasses( new_root, var ).first : var;
			});
			// set binding for new class
			bindings = bindings->set( new_root, bound );
		});
	}

	void TypeEnvironment::extractOpenVars( OpenVarSet &openVars ) const {
		bindings->apply_to_all( [classes,&openVars]( interned_string root, const BoundType& bound ) {
			classes->apply_to_class( root, [&openVars,&bound]( interned_string var ) {
				openVars[ var ] = bound.data;
			} );
		} );
	}

	void TypeEnvironment::addActual( const TypeEnvironment& actualEnv, OpenVarSet& openVars ) {
		actualEnv.bindings->apply_to_all( [&]( interned_string root, const BoundType& bound ) {
			// add members of new class, setting openVars concurrently
			interned_string new_root{nullptr};
			actualEnv.classes->apply_to_class( root, [&]( interned_string var ) {
				classes = classes->add( var );
				new_root = new_root ? mergeClasses( new_root, var ).first : var;
				openVars[ var ] = bound.data;
			} );
			// add new type binding without widening
			bindings = bindings->set( new_root, 
				BoundType{ maybeClone(bound.type), false, bound.data } );
		} );
	}

	void TypeEnvironment::forbidWidening() {
		bindings = bindings->apply_to_all([]( const interned_string& k, BoundType& c ) {
			if ( c.allowWidening ) {
				option<BoundType> ret = c;
				c.allowWidening = false;
				return ret;
			} else return option<BoundType>{};
		});
	}

	std::ostream & operator<<( std::ostream & out, const TypeEnvironment & env ) {
		env.print( out );
		return out;
	}

	PassVisitor<GcTracer> & operator<<( PassVisitor<GcTracer> & gc, const TypeEnvironment & env ) {
		for ( ClassRef c : env ) {
			maybeAccept( c.get_bound().type, gc );
		}
		return gc;
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
